package csbase.client.util;

import java.lang.reflect.InvocationTargetException;
import java.text.DateFormat;
import java.util.Date;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.client.desktop.DesktopFrame;
import csbase.client.remote.srvproxies.MailProxy;
import csbase.logic.CommonClientProject;
import csbase.logic.User;

/**
 * Gerenciador de envio de mensagens.
 */
public final class ExceptionMessenger {

  /** singleton */
  private static ExceptionMessenger instance;

  /**
   * Construtor privado.
   */
  private ExceptionMessenger() {
  }

  /**
   * @return instncia.
   */
  public static ExceptionMessenger getInstance() {
    if (instance == null) {
      instance = new ExceptionMessenger();
    }
    return instance;
  }

  /**
   * Envia um email ao admin.
   * 
   * @param throwable - um erro.
   * @return TRUE se a mensagem foi enviada com sucesso.
   */
  public boolean send(Throwable throwable) {
    return send(throwable, null, null);
  }

  /**
   * Envia um email ao admin.
   * 
   * @param throwable - um erro.
   * @param comment - comentario relacionado ao erro.
   * @return TRUE se a mensagem foi enviada com sucesso.
   */
  public boolean send(Throwable throwable, String comment) {
    return send(throwable, comment, null);
  }

  /**
   * Envia um email ao 'suporte'.
   * 
   * @param throwable - um erro.
   * @param comment - comentario relacionado ao erro.
   * @param additionalInfo - lista de informaes adicionais para serem
   *        includas no email.
   * @return TRUE se a mensagem foi enviada com sucesso.
   */
  public boolean send(Throwable throwable, String comment,
    String[] additionalInfo) {
    String messageBody = getMessageBody(throwable, comment, additionalInfo);
    return MailProxy.mailSupport(messageBody);
  }

  /**
   * @param throwable - um erro.
   * @return um texto com a descrio do erro.
   */
  private static String getExceptionText(Throwable throwable) {
    String separator = "";
    String tab = "";
    String text = "";
    Throwable curr;
    Throwable prox = throwable;
    do {
      curr = prox;
      text += separator;
      text += tab + "Classe: " + curr.getClass().getName() + "\n";
      text += tab + "Mensagem: " + curr.getLocalizedMessage() + "\n";
      text += tab + "Pilha:\n";
      StackTraceElement[] stackTrace = curr.getStackTrace();
      for (int i = 0; i < stackTrace.length; i++) {
        text += tab + stackTrace[i] + "\n";
      }
      separator = "\nCausada pela exceo:\n";
      tab += "  ";
      if (curr instanceof InvocationTargetException) {
        InvocationTargetException ite = (InvocationTargetException) curr;
        prox = ite.getTargetException();
      }
      else {
        prox = curr.getCause();
      }
    } while (prox != null && curr != prox);
    return text;
  }

  /**
   * Cria o texto da mensagem.
   * 
   * @param throwable - um erro.
   * @param comment - comentario relacionado ao erro.
   * @param additionalInfo - lista de informaes adicionais para serem
   *        includas no email.
   * @return o texto para a mensagem.
   */
  private String getMessageBody(Throwable throwable, String comment,
    String[] additionalInfo) {
    StringBuilder text = new StringBuilder();
    User user = User.getLoggedUser();
    text.append("Nome do usurio: ").append(user.getName());
    text.append(" (").append(user.getLogin()).append(")");
    text.append("\nE-mails do usurio: ");
    String[] emails = user.getEmails();
    if (emails == null || emails.length == 0) {
      text.append("-");
    }
    else {
      for (String email : emails) {
        text.append("\n\t" + email);
      }
    }
    text.append("\nData: ");
    text.append(FormatUtils.format(new Date(), DateFormat.SHORT));

    // projeto atual:
    if (DesktopFrame.getInstance() != null) {
      CommonClientProject project = DesktopFrame.getInstance().getProject();
      if (project == null) {
        text.append("\n").append(LNG.get("UTIL_ADDITIONAL_INFO_NO_PROJECT"));
      }
      else {
        text.append("\n").append(LNG.get("UTIL_ADDITIONAL_INFO_PROJECT"));
        text.append(" ").append(project.getName());
      }
    }

    // verso
    final String version = Client.getInstance().getVersion();
    if (version != null) {
      text.append("\n").append(LNG.get("UTIL_ADDITIONAL_INFO_VERSION"));
      text.append(" ").append(version);
    }

    if (additionalInfo != null && additionalInfo.length > 0) {
      text.append("\nInformaes do problema:");
      for (int i = 0; i < additionalInfo.length; i++) {
        text.append("\n -> ").append(additionalInfo[i]);
      }
    }

    if (comment != null) {
      text.append("\nComentrio: ").append(comment);
    }

    text.append("\nExceo: ").append(getExceptionText(throwable));

    return text.toString();
  }
}
