package csbase.client.algorithms;

import java.awt.Window;
import java.text.MessageFormat;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.algorithms.parameters.ParameterView.Mode;
import csbase.client.algorithms.parameters.SimpleAlgorithmConfiguratorView;
import csbase.client.applications.flowapplication.configurator.FlowAlgorithmConfiguratorView;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.exception.BugException;
import csbase.exception.PermissionException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.remote.ClientRemoteLocator;

/**
 * Fbrica de vises de configuradores de algoritmo.
 */
public final class AlgorithmConfiguratorFactory {

  /**
   * A nica instncia desta classe.
   */
  private static AlgorithmConfiguratorFactory instance;

  /**
   * Cria a fbrica.
   */
  private AlgorithmConfiguratorFactory() {
  }

  /**
   * Obtm a instncia desta classe.
   * 
   * @return .
   */
  public static AlgorithmConfiguratorFactory getInstance() {
    if (instance == null) {
      instance = new AlgorithmConfiguratorFactory();
    }
    return instance;
  }

  /**
   * Cria a viso do configurador no modo de configurao {@link Mode#REPORT}.
   * 
   * @param owner A janela que ser dona da viso (No aceita {@code null}).
   * @param configurator O configurador do algoritmo (No aceita {@code null}).
   * 
   * @return A viso.
   */
  public AlgorithmConfiguratorView createReportView(final Window owner,
    AlgorithmConfigurator configurator) {
    if (owner == null) {
      throw new IllegalArgumentException("O parmetro owner est nulo.");
    }
    if (configurator == null) {
      throw new IllegalArgumentException("O parmetro configurator est nulo.");
    }
    return createView(owner, configurator, Mode.REPORT, ValidationMode.FULL);
  }

  /**
   * Cria a viso do configurador no modo de configurao
   * {@link Mode#CONFIGURATION}.
   * 
   * @param owner A janela que ser dona da viso (No aceita {@code null}).
   * @param version A verso do algoritmo.
   * @param defaultValidationMode O modo de validao padro.
   * 
   * @return A viso.
   */
  public AlgorithmConfiguratorView createConfigurationView(final Window owner,
    final AlgorithmVersionInfo version, ValidationMode defaultValidationMode) {
    if (owner == null) {
      throw new IllegalArgumentException("O parmetro owner est nulo.");
    }
    if (version == null) {
      throw new IllegalArgumentException("O parmetro version est nulo.");
    }
    RemoteTask<AlgorithmConfigurator> task =
      new RemoteTask<AlgorithmConfigurator>() {

        @Override
        protected void performTask() throws Exception {
          String algorithmName = version.getInfo().getName();
          AlgorithmVersionId algorithmVersionId = version.getId();
          AlgorithmConfigurator algorithmConfigurator =
            ClientRemoteLocator.algorithmService.createAlgorithmConfigurator(
              algorithmName, algorithmVersionId);
          setResult(algorithmConfigurator);
        }
      };
    if (task.execute(owner, getString("reading_configurator_title"),
      getString("reading_configurator"))) {
      AlgorithmConfigurator algorithmConfigurator = task.getResult();
      return createView(owner, algorithmConfigurator, Mode.CONFIGURATION,
        defaultValidationMode);
    }
    return null;
  }

  /**
   * Cria a viso do configurador no modo escolhido.
   * 
   * @param owner A janela que ser dona da viso (No aceita {@code null}).
   * @param algorithmConfigurator O configurador do algoritmo (No aceita
   *        {@code null}).
   * @param mode Modo de visualizao do configurador. No aceita {@code null},
   *        os possveis valores so: {@link Mode#CONFIGURATION} ou
   *        {@link Mode#REPORT}
   * @param defaultValidationMode O modo de validao padro.
   * 
   * @return A viso.
   */
  private AlgorithmConfiguratorView createView(final Window owner,
    final AlgorithmConfigurator algorithmConfigurator, Mode mode,
    ValidationMode defaultValidationMode) {
    if (mode == null) {
      throw new IllegalArgumentException("O parmetro mode est nulo.");
    }
    if (algorithmConfigurator == null) {
      throw new IllegalArgumentException(
        "O parmetro algorithmConfigurator est nulo.");
    }

    AlgorithmConfiguratorView configuratorView = null;
    try {
      ConfiguratorType configuratorType =
        algorithmConfigurator.getConfiguratorType();
      switch (configuratorType) {
        case SIMPLE:
          SimpleAlgorithmConfigurator simpleAlgorithmConfigurator =
            (SimpleAlgorithmConfigurator) algorithmConfigurator;
          configuratorView =
            new SimpleAlgorithmConfiguratorView(owner,
              simpleAlgorithmConfigurator, mode, defaultValidationMode);
          break;
        case FLOW:
          FlowAlgorithmConfigurator flowAlgorithmConfigurator =
            (FlowAlgorithmConfigurator) algorithmConfigurator;
          configuratorView =
            new FlowAlgorithmConfiguratorView(owner, flowAlgorithmConfigurator,
              mode, defaultValidationMode);
          break;
        default:
          throw new IllegalArgumentException(
            "Tipo de configurador no conhecido: " + configuratorType);
      }
    }
    catch (ClientException e) {
      if (!(e.getCause() instanceof PermissionException)) {
        throw new BugException(getString("error_creating_configurator_view",
          algorithmConfigurator.getAlgorithmName()), e);
      }
    }
    return configuratorView;
  }

  /**
   * <p>
   * Obtm uma mensagem no mecanismo de internacionalizao.
   * </p>
   * 
   * <p>
   *  um informado um sufixo ({@code suffixKey}) que  anexado ao nome da
   * classe.
   * </p>
   * 
   * @param suffixKey O sufixo da chave (No aceita {@code null}).
   * @param arguments argumentos.
   * 
   * @return A mensagem.
   */
  private String getString(String suffixKey, Object... arguments) {
    return MessageFormat.format(
      LNG.get(AlgorithmConfiguratorFactory.class.getName() + "." + suffixKey),
      arguments);
  }
}
