/*
 * $Id$
 */

package csbase.client.applications;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Window;

import javax.swing.ImageIcon;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;

import csbase.client.desktop.DesktopErrorEvent;
import csbase.client.desktop.DesktopFrame;

/**
 * Modela um frame criado por uma aplicao.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ApplicationComponentFrame extends JDialog {

  /**
   * A aplicao que chamou o frame
   */
  private Application application = null;

  /**
   * O contedo da janela.
   */
  private JPanel frameContents;

  /**
   * Barra de status
   */
  private JLabel statusBar;

  /**
   * Redefine o mtodo setVisible() do frame.  chamada tambm no caso
   * <code>dispose()</code>
   * 
   * @param visible indica se o frame deve ou no ser exibido.
   */
  @Override
  public void setVisible(boolean visible) {
    super.setVisible(visible);
    if (visible) {
      application.addWindow(this);
    }
    else {
      application.removeWindow(this);
    }
  }

  /**
   * Centraliza a janela da aplicao.
   */
  public void center() {
    center(application.getApplicationFrame());
  }

  /**
   * Centraliza a janela da aplicao em relao outra janela.
   * 
   * @param window .
   */
  public void center(Window window) {
    Dimension sSize = window.getSize();
    Dimension wSize = getSize();
    int newX = (sSize.width - wSize.width) / 2;
    int newY = (sSize.height - wSize.height) / 2;
    setLocation(window.getX() + newX, window.getY() + newY);
  }

  /**
   * Retorno da aplicao que criou o frame.
   * 
   * @return .
   */
  public Application getApplication() {
    return application;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Container getContentPane() {
    return frameContents;
  }

  /**
   * Ajuste da exibio da barra de status.
   */
  public void showStatusBar() {
    if ((statusBar.getText() == null) || statusBar.getText().trim().equals("")) {
      statusBar.setText(" ");
    }
  }

  /**
   * Retorno da status bar.
   * 
   * @return a barra.
   */
  public JLabel getStatusBar() {
    return this.statusBar;
  }

  /**
   * Sinaliza um erro na janela.
   * 
   * @param errorEvent erro a ser sinalizado.
   */
  public void signalError(DesktopErrorEvent errorEvent) {
    /* Coloca a mensagem na barra de status */
    if (errorEvent.errorMessage != null) {
      DesktopFrame dktFrame = DesktopFrame.getInstance();
      Image img = null;
      if (dktFrame != null) {
        img = dktFrame.getDesktopErrorImage();
      }
      if (statusBar != null) {
        if (img != null) {
          statusBar.setIcon(new ImageIcon(img));
        }
        statusBar.setText(errorEvent.errorMessage);
      }
    }
  }

  /**
   * Remove uma sinalizao de erro na janela.
   * 
   * @param errorEvent erro sinalizado.
   */
  public void resetError(DesktopErrorEvent errorEvent) {
    /* Retira a mensagem na barra de status */
    if (errorEvent.errorMessage != null) {
      statusBar.setIcon(null);
      statusBar.setText("");
    }
  }

  /**
   * Construtor
   * 
   * @param application aplicao.
   */
  public ApplicationComponentFrame(Application application) {
    this(null, application);
  }

  /**
   * Construtor
   * 
   * @param title ttulo
   * @param application aplicao.
   */
  public ApplicationComponentFrame(String title, Application application) {
    super(application.getApplicationFrame(), (title == null) ? application
      .getName() : title);
    this.application = application;
    frameContents = new JPanel(new BorderLayout());
    // statusBar
    statusBar = new JLabel();
    statusBar.setHorizontalAlignment(JLabel.LEFT);
    Container cp = super.getContentPane();
    cp.add(frameContents, BorderLayout.CENTER);
    cp.add(statusBar, BorderLayout.SOUTH);
  }
}
