package csbase.client.applications.algorithmsmanager;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.util.Collections;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;

import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.algorithmsmanager.actions.AlgorithmManagementAction;
import csbase.client.applications.algorithmsmanager.actions.AlgorithmsManagerAction;
import csbase.client.applications.algorithmsmanager.actions.CategoryManagementAction;
import csbase.client.applications.algorithmsmanager.actions.ListOutdatedAlgorithmAction;
import csbase.client.applications.algorithmsmanager.actions.ReloadAlgorithmsAction;
import csbase.client.applications.algorithmsmanager.actions.RunTestsAction;
import csbase.client.applications.algorithmsmanager.dialogs.AlgorithmsPropertiesDialog;
import csbase.client.applications.algorithmsmanager.versiontree.actions.ShowHistory;
import csbase.client.remote.AlgorithmManagementListener;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy.AlgorithmOperation;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;
import csbase.logic.algorithms.HistoryProvider;
import csbase.logic.algorithms.HistoryRecord;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe que representa a aplicao para Gerncia de Algoritmos. Essa aplicao
 * visa inicialmente prover funcionalidades administrativas relacionadas a
 * instalao de algoritmos no sistema. Por exemplo, essa aplicao deve prover
 * uma funcionalidade que permite associar categorias a algoritmos e vice-versa.
 *
 */
public class AlgorithmsManager extends ApplicationProject {
  /** Valor default para o tamanho da janela principal da aplicao */
  private static final Dimension DEFAULT_MAIN_FRAME_SIZE = new Dimension(1024,
    768);

  /** Painel principal */
  private JPanel mainPanel;

  /** Ao de gerncia de algoritmos */
  private CategoryManagementAction categoryManagementAction;

  /** Ao de gerncia de algoritmos */
  private AlgorithmManagementAction algorithmManagementAction;

  /** Ao de recarregar os algoritmos a partir do servidor */
  private ReloadAlgorithmsAction reloadAlgorithmsAction;

  /** Ao de listar fluxos desatualizados */
  private ListOutdatedAlgorithmAction listOutdatedAction;
  private RunTestsAction runTestsAction;

  /** Boto toggle que ativa o gerenciamento de categorias */
  private JToggleButton categoryToggleButton;

  /** Boto toggle que ativa o gerenciamento de algoritmos */
  private JToggleButton algorithmToggleButton;

  /**
   * Listeners interessados em mudanas na aplicao.
   */
  private List<AlgorithmsManagerAdapter> algoManagertList;

  /**
   * Constri a aplicao para gerncia de algoritmos.
   *
   * @param id Identificador da aplicao
   */
  public AlgorithmsManager(String id) {
    super(id);
    algoManagertList = new Vector<>();
    addAlgorithmProxyListener();
    buildFrame();
    initApplicationAction();
  }

  /**
   * Adiciona um listener para mudanas ocorridas na aplicao de gerncia de
   * algoritmos.
   *
   * @param listener listener a ser adicionado
   */
  public void addAlgorithmsManagerListener(AlgorithmsManagerAdapter listener) {
    if (!algoManagertList.contains(listener)) {
      algoManagertList.add(listener);
    }
  }

  /**
   * Remove um listener da aplicao.
   *
   * @param listener listener
   */
  public void removeAlgorithmsManagerListener(
    AlgorithmsManagerAdapter listener) {
    algoManagertList.remove(listener);
  }

  /**
   * Adiciona um listener para o proxy de algoritmos.
   */
  private void addAlgorithmProxyListener() {
    AlgorithmManagementProxy.addManagementListener(
      new AlgorithmManagementListener() {

        @Override
        public void categoryUpdated(CategorySet modifiedCategorySet) {
          notifyCategoryUpdated(modifiedCategorySet);
        }

        @Override
        public void categoryRemoved(Category category) {
          notifyCategoryRemoved(category);
        }

        @Override
        public void categoryCreated(Category category) {
          notifyCategoryCreated(category);
        }

        @Override
        public void algorithmCreated(AlgorithmInfo algoInfo) {
          notifyAlgorithmCreated(algoInfo);
        }

        @Override
        public void algorithmRemoved(AlgorithmInfo algoInfo) {
          notifyAlgorithmRemoved(algoInfo);
        }

        @Override
        public void algorithmUpdated(AlgorithmInfo algoInfo) {
          notifyAlgorithmUpdated(algoInfo);
        }

        @Override
        public void algorithmsReloaded() {
          refreshApplication();
        }
      });
  }

  /**
   * Notifica os listeners da aplicao que uma categoria foi criada.
   *
   * @param category categoria criada ou a categoria raiz que recebeu a nova
   *        categoria
   *
   */
  private void notifyCategoryCreated(Category category) {
    for (AlgorithmsManagerAdapter listener : algoManagertList) {
      listener.categoryCreated(category);
    }
  }

  /**
   * Notifica os listeners da aplicao que uma categoria foi removida.
   *
   * @param category categoria removida
   */
  private void notifyCategoryRemoved(Category category) {
    for (AlgorithmsManagerAdapter listener : algoManagertList) {
      listener.categoryRemoved(category);
    }
  }

  /**
   * Notifica os listeners da aplicao que um conjunto de categorias foi
   * alterada.
   *
   * @param modifiedCategorySet conjunto com as categorias modificadas
   */
  private void notifyCategoryUpdated(CategorySet modifiedCategorySet) {
    for (AlgorithmsManagerAdapter listener : algoManagertList) {
      listener.categoryUpdated(modifiedCategorySet);
    }
  }

  /**
   * Notifica os listeners da aplicao que um algoritmo foi criado.
   *
   * @param algoInfo algoritmo criado
   *
   */
  private void notifyAlgorithmCreated(AlgorithmInfo algoInfo) {
    for (AlgorithmsManagerAdapter listener : algoManagertList) {
      listener.algorithmCreated(algoInfo);
    }
  }

  /**
   * Notifica os listeners da aplicao que um algoritmo foi removido.
   *
   * @param algoInfo algoritmo removido
   *
   */
  private void notifyAlgorithmRemoved(AlgorithmInfo algoInfo) {
    for (AlgorithmsManagerAdapter listener : algoManagertList) {
      listener.algorithmRemoved(algoInfo);
    }
  }

  /**
   * Notifica os listeners da aplicao que um algoritmo foi alterado.
   *
   * @param algoInfo algoritmo alterado
   *
   */
  private void notifyAlgorithmUpdated(AlgorithmInfo algoInfo) {
    for (AlgorithmsManagerAdapter listener : algoManagertList) {
      listener.algorithmUpdated(algoInfo);
    }
  }

  /**
   * Inicializa a ao principal da aplicao.
   */
  private void initApplicationAction() {
    if (categoryToggleButton.isSelected()) {
      getCategoryManagementAction().initAction();
    }
    else {
      AlgorithmManagementAction algorithmAction =
        getAlgorithmManagementAction();
      algorithmAction.initAction();
      getListOutdatedAction().setSelectionPanel(algorithmAction
        .getSelectionPanel());
    }
  }

  /**
   * Constri a janela principal da aplicao.
   *
   */
  private void buildFrame() {
    ApplicationFrame mainFrame = getApplicationFrame();
    mainFrame.setJMenuBar(buildMenuBar());
    mainFrame.getContentPane().add(buildToolBar(), BorderLayout.NORTH);
    mainFrame.getContentPane().add(getMainPanel());
    mainFrame.showStatusBar();
    mainFrame.setSize(DEFAULT_MAIN_FRAME_SIZE);
    mainFrame.setLocationRelativeTo(null);
  }

  /**
   * Obtm o painel principal da aplicao. Se ainda no existir, cria pela
   * primeira vez.
   *
   * @return o painel principal
   */
  private JPanel getMainPanel() {
    if (mainPanel == null) {
      mainPanel = new JPanel();
      mainPanel.setLayout(new BorderLayout());
    }
    return mainPanel;
  }

  /**
   * Obtm o tamanho do painel principal da aplicao.
   *
   * @return o tamanho do painel principal da aplicao
   */
  public Dimension getMainPanelSize() {
    return getMainPanel().getSize();
  }

  /**
   * Constri a barra de menu.
   *
   * @return a barra de menu
   */
  private JMenuBar buildMenuBar() {
    JMenuBar menuBar = new JMenuBar();
    menuBar.add(buildAdminMenu());
    menuBar.add(buildRepositoryMenu());
    return menuBar;
  }

  /**
   * Constri o menu de administrao.
   *
   * @return o menu criado
   */
  private JMenu buildAdminMenu() {
    JMenu adminMenu = new JMenu(getString("AlgorithmsManager.menu.admin"));
    adminMenu.add(new JMenuItem(getAlgorithmManagementAction()));
    adminMenu.add(new JMenuItem(getCategoryManagementAction()));
    adminMenu.addSeparator();
    adminMenu.add(new JMenuItem(getReloadAlgorithmsAction()));
    adminMenu.add(new JMenuItem(getListOutdatedAction()));
    adminMenu.add(new JMenuItem(getRunTestsAction()));
    adminMenu.addSeparator();
    adminMenu.add(new ApplicationExitAction(this));
    return adminMenu;
  }

  /**
   * Constri o menu de administrao.
   *
   * @return o menu criado
   */
  private JMenu buildRepositoryMenu() {
    JMenu repositoryMenu = new JMenu(getString(
      "AlgorithmsManager.menu.repository"));
    repositoryMenu.add(new JMenuItem(new ShowHistory(getApplicationFrame(),
      new RepositoryHistory())));
    JMenuItem algoPropsMenuItem = repositoryMenu.add(getString(
      "AlgorithmsManager.menu.repository.properties"));
    algoPropsMenuItem.addActionListener(e -> {
      AlgorithmsPropertiesDialog algoPropDialog = AlgorithmsPropertiesDialog
        .getFrame(AlgorithmsManager.this);
      algoPropDialog.setVisible(true);
    });
    repositoryMenu.add(algoPropsMenuItem);
    return repositoryMenu;
  }

  /**
   * Constri a barra de ferramentas.
   *
   * @return o componente criado
   */
  private JComponent buildToolBar() {
    JToolBar toolBar = new JToolBar();
    ButtonGroup manageGroup = new ButtonGroup();
    categoryToggleButton = getToggleButton(manageGroup,
      getCategoryManagementAction(), false);
    algorithmToggleButton = getToggleButton(manageGroup,
      getAlgorithmManagementAction(), true);
    toolBar.add(algorithmToggleButton);
    toolBar.add(categoryToggleButton);
    toolBar.addSeparator();
    toolBar.add(getReloadAlgorithmsAction());
    return toolBar;
  }

  /**
   * Obtm a ao de gerenciamento de categorias.
   *
   * @return a ao
   */
  public CategoryManagementAction getCategoryManagementAction() {
    if (categoryManagementAction == null) {
      categoryManagementAction = new CategoryManagementAction(this,
        ApplicationImages.ICON_ALGORITHM_CATEGORY_16);
      categoryManagementAction.setEnabled(true);
    }
    return categoryManagementAction;
  }

  /**
   * Obtm a ao de gerenciamento de categorias.
   *
   * @return a ao
   */
  public AlgorithmManagementAction getAlgorithmManagementAction() {
    if (algorithmManagementAction == null) {
      algorithmManagementAction = new AlgorithmManagementAction(this,
        ApplicationImages.ICON_ALGORITHM_MNG_16);
      algorithmManagementAction.setEnabled(true);
    }
    return algorithmManagementAction;
  }

  /**
   * Obtm a ao de recarregar os algoritmos instalados no servidor.
   *
   * @return a ao
   */
  public ReloadAlgorithmsAction getReloadAlgorithmsAction() {
    if (reloadAlgorithmsAction == null) {
      reloadAlgorithmsAction = new ReloadAlgorithmsAction(this);
      reloadAlgorithmsAction.setEnabled(true);
    }
    return reloadAlgorithmsAction;
  }

  /**
   * Obtm a ao de recarregar os algoritmos instalados no servidor.
   *
   * @return a ao
   */
  public ListOutdatedAlgorithmAction getListOutdatedAction() {
    if (listOutdatedAction == null) {
      listOutdatedAction = new ListOutdatedAlgorithmAction(this, null);
      listOutdatedAction.setEnabled(true);
    }
    return listOutdatedAction;
  }

  /**
   * Obtm a ao de recarregar os algoritmos instalados no servidor.
   *
   * @return a ao
   */
  public RunTestsAction getRunTestsAction() {
    if (runTestsAction == null) {
      runTestsAction = new RunTestsAction(this);
      runTestsAction.setEnabled(true);
    }
    return runTestsAction;
  }

  /**
   * Obtm o boto com estado (toggle) associado a uma determinada ao. Somente
   * um boto de gerenciamento pode estar selecionado de cada vez, alternando
   * ento seus estados que indicam a seleo.
   *
   * @param manageGroup grupo dos botes de gerenciamento
   *
   * @param action ao associada ao boto
   * @param setSelected se true, o boto vai ser selecionado
   * @return o boto com o estado de selecionado ou no
   */
  private JToggleButton getToggleButton(ButtonGroup manageGroup,
    AlgorithmsManagerAction action, boolean setSelected) {
    JToggleButton button = new JToggleButton(action);
    button.setText(null);
    button.setSelected(setSelected);
    manageGroup.add(button);
    return button;
  }

  /**
   * Modifica o painel principal da janela.
   *
   * @param panel painel a ser exibido na janela principal da aplicao
   */
  public void changeMainPanel(JPanel panel) {
    mainPanel.removeAll();
    if (panel != null) {
      mainPanel.add(panel);
    }
    mainPanel.revalidate();
    getApplicationFrame().repaint();
  }

  /**
   * (non-Javadoc)
   *
   * @see csbase.client.applications.Application#killApplication()
   */
  @Override
  public void killApplication() {
    final JFrame mainFrame = getApplicationFrame();
    mainFrame.dispose();
  }

  /**
   * (non-Javadoc)
   *
   * @see csbase.client.applications.Application#userCanKillApplication()
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * Reconstri as aes de gerenciamento de algoritmos e de categorias.
   *
   */
  private void rebuildManagementActions() {
    getAlgorithmManagementAction().updateToReload();
    getCategoryManagementAction().updateToReload();
    getListOutdatedAction().updateToReload();
    getRunTestsAction().updateToReload();
  }

  /**
   * Obter todas as categorias disponveis no servidor, inclusive as
   * subcategorias.
   *
   * @param includeSubCategories se true, inclui as sub-categorias no resultado,
   *        caso contrrio, retorna somente as categorias raiz (do primeiro
   *        nvel)
   *
   * @return as categorias disponveis no servidor
   */
  public SortedSet<Category> getAllCategories(boolean includeSubCategories) {
    SortedSet<Category> categories = null;
    CategorySet categorySet = AlgorithmManagementProxy.getAllCategories(
      getApplicationFrame(), AlgorithmOperation.ADMIN_ALGORITHM);
    if (categorySet != null) {
      if (includeSubCategories) {
        categories = categorySet.getAllCategories();
      }
      else {
        categories = categorySet.getCategories();
      }
    }
    return categories;
  }

  /**
   * Obter todas os algoritmos disponveis no servidor.
   *
   * @return as algoritmos disponveis no servidor
   */
  public SortedSet<AlgorithmInfo> getAllAgorithms() {
    AlgorithmInfo[] algorithmInfos = AlgorithmManagementProxy
      .getAllAlgorithmInfos(getApplicationFrame(),
        AlgorithmOperation.ADMIN_ALGORITHM);
    SortedSet<AlgorithmInfo> algorithmsSet = new TreeSet<>();
    Collections.addAll(algorithmsSet, algorithmInfos);
    return algorithmsSet;
  }

  /**
   * Essa classe implementa o histrico do repositrio de algoritmos.
   *
   */
  private class RepositoryHistory implements HistoryProvider {

    /**
     * (non-Javadoc)
     *
     * @see csbase.logic.algorithms.HistoryProvider#getHistory()
     */
    @Override
    public List<HistoryRecord> getHistory() {
      final String repositoryName = LNG.get("algomanager.tree.root");
      final String[] spath = new String[] { repositoryName };
      return AlgorithmManagementProxy.retrieveHistory(spath);
    }

    /**
     * (non-Javadoc)
     *
     * @see csbase.logic.algorithms.HistoryProvider#getName()
     */
    @Override
    public String getName() {
      String repositoryName = LNG.get("algomanager.tree.root");
      return repositoryName;
    }

  }

  /**
   * Verifica se j existe em memria um algoritmo cadastrado com esse nome.
   *
   * @param algoName nome do algoritmo
   *
   * @return retorna true se j existe em memria um algoritmo cadastrado com
   *         esse nome, caso contrrio, retorna false
   */
  public boolean containsAlgorithm(String algoName) {
    SortedSet<AlgorithmInfo> allAgorithms = getAllAgorithms();
    for (AlgorithmInfo algo : allAgorithms) {
      if (algo.getName().trim().toLowerCase().equals(algoName.trim()
        .toLowerCase())) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se j existe em memria um algoritmo cadastrado com esse
   * identificador.
   *
   * @param algoId identificador do algoritmo
   *
   * @return retorna true se j existe em memria um algoritmo cadastrado com
   *         esse identificador, caso contrrio, retorna false
   */
  public boolean containsAlgorithmWithId(String algoId) {
    SortedSet<AlgorithmInfo> allAgorithms = getAllAgorithms();
    for (AlgorithmInfo algo : allAgorithms) {
      if (algo.getId().equals(algoId)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se j existe em memria uma categoria (no nvel raiz) cadastrada
   * com esse nome.
   *
   * @param categoryName nome da categoria
   *
   * @return retorna true se j existe em memria uma categoria cadastrada com
   *         esse nome, caso contrrio, retorna false
   */
  public boolean containsCategory(String categoryName) {
    SortedSet<Category> categories = getAllCategories(false);
    for (Category category : categories) {
      if (category.getName().trim().toLowerCase().equals(categoryName.trim()
        .toLowerCase())) {
        return true;
      }
    }
    return false;
  }

  /**
   * Recarrega os algoritmos instalados no servidor.
   */
  public void reloadAlgorithmsFromServer() {
    AlgorithmManagementProxy.reloadAlgorithms(getApplicationFrame());
    refreshApplication();
  }

  /**
   * Atualiza a interface da aplicao.
   */
  private void refreshApplication() {
    algoManagertList.clear();
    rebuildManagementActions();
    initApplicationAction();
  }

}
