package csbase.client.applications.algorithmsmanager.actions;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import csbase.client.applications.algorithmsmanager.dialogs.CommonSelectionPanel;
import csbase.client.applications.algorithmsmanager.models.CategoryNodeInterface;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.logic.algorithms.Category;

/**
 * Essa classe representa a ao de remover uma ou mais categorias de
 * algoritmos, na funcionalidade de gerenciamento de categorias.
 * 
 * O mtodo <code> handleEditOperation </code> deve ser redefinido para que seja
 * confirmado pelo usurio a remoo de todas as categorias selecionadas no
 * painel de seleo.
 * 
 */
public class CategoryRemoveAction extends CommonEditAction {
  /**
   * Constri a ao de remover categorias de algoritmos.
   * 
   * @param selectionPanel painel pai que criou a ao
   * @param icon imagem da ao
   */
  public CategoryRemoveAction(CommonSelectionPanel selectionPanel,
    ImageIcon icon) {
    super(selectionPanel, icon);
  }

  /**
   * Remove efetivamente as categorias no servidor.
   * 
   * @param categories lista com as categorias a serem removidas
   * @return o nmero de categorias que foram efetivamente removidas
   * 
   */
  private int removeCategories(final List<Category> categories) {
    int removedCounter = 0;
    for (Category category : categories) {
      boolean removed =
        AlgorithmManagementProxy.removeCategory(category.getId(),
          getApplication().getApplicationFrame());
      if (removed) {
        removedCounter++;
      }
    }
    return removedCounter;
  }

  @Override
  protected void handleEditOperation() {
    List<DataInterface> categoriesNodeList =
      getSelectionPanel().getSelectedDataList();
    int option =
      getApplication().showOptionDialog(
        getString("CategoryRemoveAction.msg.cancel.confirm"),
        new String[] {
            getString("CategoryRemoveAction.msg.cancel.confirm.yes"),
            getString("CategoryRemoveAction.msg.cancel.confirm.no") });
    if (option == 0) {
      int removedCounter = removeCategories(getCategories(categoriesNodeList));
      if (removedCounter != categoriesNodeList.size()) {
        String msg =
          MessageFormat.format(
            getString("CategoryRemoveAction.msg.error.remove.no_sucess"),
            new Object[] { removedCounter });
        getApplication().showMessage(msg);
      }
    }
  }

  /**
   * Obtm uma lista de categorias a partir de uma lista de ns de categorias.
   * 
   * @param categoriesNodeList lista de ns de categorias
   * @return a lista de categorias correspondente aos ns
   */
  private List<Category> getCategories(List<DataInterface> categoriesNodeList) {
    List<Category> categories = new ArrayList<>();
    for (DataInterface node : categoriesNodeList) {
      CategoryNodeInterface categoryNode = (CategoryNodeInterface) node;
      categories.add(categoryNode.getNode());
    }
    return categories;
  }

}
