/*
 * DiskInfo.java
 * 
 * @version $Revision$
 */
package csbase.client.applications.diskinfo;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.JToolBar;
import javax.swing.Timer;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.remote.srvproxies.DiskUsageProxy;
import csbase.exception.BugException;

/**
 * A classe <code>DiskInfo</code> modela a aplicao de monitorao de uso de
 * disco.
 * 
 * @author Cristina Ururahy
 */
public class DiskInfo extends Application {

  /** Painel que contm o grfico */
  private ApplicationFrame mainFrame = null;

  /** Lista de painis de monitorao */
  private final List<DiskInfoPanel> monitoringPanels;

  /** Painel de usurios */
  private DiskUserPanel uPanel;

  /** Painel de espao total */
  private DiskSpacePanel pPanel;

  /** Painel de projetos */
  private DiskProjectPanel prjPanel;

  /** Toggle de animao */
  private final JCheckBox animationToggle = new JCheckBox();

  /** Temporizador de atualizao */
  private Timer updateTimer = null;

  /**
   * Montagem do menu.
   * 
   * @return a barra de menu.
   */
  private JMenuBar buildMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(buildOptionsMenu());
    menuBar.add(new JPanel());
    menuBar.add(buildHelpMenu());
    return menuBar;
  }

  /**
   * Montagem do menu de edio
   * 
   * @return um menu de edio.
   */
  private JMenu buildOptionsMenu() {
    final JMenu optMenu = new JMenu(getString("menu.options"));
    final JMenuItem updItem = optMenu.add(getString("menu.options.update"));
    updItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        refreshPies();
      }
    });
    updItem.setIcon(ApplicationImages.ICON_REFRESH_16);

    // Item de menu para ver dados dos usurios de forma tabular
    final JMenuItem tableItem =
      optMenu.add(getString("menu.options.usersTable"));
    tableItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        showUserDataInTable();
      }
    });
    tableItem.setIcon(ApplicationImages.ICON_TABLE_16);

    optMenu.addSeparator();
    optMenu.add(new ApplicationExitAction(this));
    return optMenu;
  }

  /**
   * Montagem do menu de ajuda
   * 
   * @return um menu de about.
   */
  private JMenu buildHelpMenu() {
    final JMenu aboutMenu = new JMenu(getString("menu.help"));
    aboutMenu.add(new ApplicationAboutAction(this));
    return aboutMenu;
  }

  /**
   * Cria a janela.
   * 
   * @return referncia para a janela criada
   */
  private ApplicationFrame createMainFrame() {
    animationToggle.setText(getString("toggle.animation"));
    final ApplicationFrame frame = getApplicationFrame();
    final Container cp = frame.getContentPane();
    final JPanel panel = new JPanel();
    panel.setLayout(new BorderLayout());

    /* Se h mais de um tipo de monitorao configurado, usa diversos painis */
    if (monitoringPanels.size() > 1) {
      final JTabbedPane tabbedPane = new JTabbedPane();
      for (final DiskInfoPanel p : monitoringPanels) {
        tabbedPane.add(p);
      }
      panel.add(tabbedPane, BorderLayout.CENTER);
    }
    else {
      panel.add(monitoringPanels.get(0), BorderLayout.CENTER);
    }

    final JToolBar optionPanel = new JToolBar(JToolBar.HORIZONTAL);
    final JButton refButton = new JButton();
    refButton.setToolTipText(getString("tooltip.refresh"));
    refButton.setText(null);
    refButton.setIcon(ApplicationImages.ICON_REFRESH_16);
    refButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ae) {
        refreshPies();
      }
    });
    optionPanel.add(refButton);

    optionPanel.addSeparator();
    // Cria boto de visualizao de dados dos usurios em forma de tabela
    final JButton tableButton = new JButton();
    tableButton.setToolTipText(getString("tooltip.usersTable"));
    tableButton.setText(null);
    tableButton.setIcon(ApplicationImages.ICON_TABLE_16);
    tableButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ae) {
        showUserDataInTable();
      }
    });
    optionPanel.add(tableButton);

    optionPanel.add(new JPanel());
    optionPanel.add(animationToggle);
    panel.add(optionPanel, BorderLayout.NORTH);

    cp.add(panel);
    return frame;
  }

  /**
   * Exibe a janela da aplicao.
   */
  public void start() {
    mainFrame.pack();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    updateTimer.stop();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userCanKillApplication() {
    return (true);
  }

  /**
   * Atualizaco dos paineis.
   */
  private void rotatePies() {
    if (!animationToggle.isSelected()) {
      return;
    }
    for (final DiskInfoPanel p : monitoringPanels) {
      p.rotatePie();
    }
  }

  /**
   * Atualizaco dos paineis.
   */
  private void refreshPies() {
    for (final DiskInfoPanel p : monitoringPanels) {
      p.refreshPie(this);
    }
    final DateFormat df =
      DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.LONG,
        LNG.getLocale());
    getApplicationFrame().getStatusBar().setStatus(df.format(new Date()));
  }

  /**
   * Criaco da tarefa de atualizaco dos grficos.
   */
  private void createUpdateTimer() {
    final int delay = 1000; // milisegundos
    final ActionListener updateTask = new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent evt) {
        rotatePies();
      }
    };
    updateTimer = new Timer(delay, updateTask);
    updateTimer.start();
  }

  /**
   * Construtor da aplicao DiskInfo.
   * 
   * @param id identificador da aplicaco.
   * @throws BugException se nenhum tipo de monitorao foi configurado pelas
   *         propriedades da aplicao
   */
  public DiskInfo(final String id) throws BugException {
    super(id);
    monitoringPanels = new ArrayList<DiskInfoPanel>();

    /* Painel de monitorao de espao por usurio */
    if (getBooleanSpecificProperty("hasUserPanel", true)) {
      uPanel = new DiskUserPanel(this);
      monitoringPanels.add(uPanel);
    }
    else {
      uPanel = null;
    }

    /* Painel de monitorao de espao total */
    if (getBooleanSpecificProperty("hasSpacePanel", false)) {
      pPanel = new DiskSpacePanel(this);
      monitoringPanels.add(pPanel);
    }
    else {
      pPanel = null;
    }

    /* Painel de monitorao da rea de projetos */
    if (getBooleanSpecificProperty("hasProjectPanel", false)) {
      prjPanel = new DiskProjectPanel(this);
      monitoringPanels.add(prjPanel);
    }
    else {
      prjPanel = null;
    }
    if (monitoringPanels.size() == 0) {
      throw new BugException("Nenhum tipo de monitorao foi configurado");
    }

    mainFrame = createMainFrame();
    mainFrame.setJMenuBar(buildMenuBar());
    start();
    refreshPies();
    createUpdateTimer();
  }

  /**
   * Cria o frame que vai mostrar os dados em formato tabular e mostra-o
   */
  private void showUserDataInTable() {
    DiskUserTableFrame dutf =
      new DiskUserTableFrame(this, DiskUsageProxy.getUsersList());
    dutf.showFrame();
  }
}
