package csbase.client.applications.flowapplication.filters;

import java.awt.Cursor;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;

import csbase.client.applications.flowapplication.CompassDirection;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphElement;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.messages.ChangeCursorMessage;
import csbase.client.applications.flowapplication.messages.PickNodeMessage;

/**
 * Filtro de resize
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ResizeNodeFilter extends WorkspaceFilter {
  /**
   * Estado: start
   */
  private static final int START = 0;
  /**
   * Estado: esperando click
   */
  private static final int WAITING_PRESS_TO_RESIZE_NODE = 1;
  /**
   * Estado: esperando drag
   */
  private static final int WAITING_DRAG_TO_RESIZE_NODE = 2;

  /**
   * Tipo de resize
   */
  private CompassDirection compassDirection;

  /**
   * N corrente
   */
  private GraphNode currentNode;

  /**
   * Estado corrente.
   */
  private int currentState;

  /**
   * Construtor
   * 
   * @param workspace workspace
   */
  public ResizeNodeFilter(final Workspace workspace) {
    super(workspace);
    this.currentState = START;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackButton(final Point2D pt, final MouseEvent ev) {
    boolean eventWasConsumed = false;
    switch (this.currentState) {
      case WAITING_PRESS_TO_RESIZE_NODE:
        if (wasPressed(ev)) {
          this.currentState = WAITING_DRAG_TO_RESIZE_NODE;
          eventWasConsumed = true;
        }
        break;
      case WAITING_DRAG_TO_RESIZE_NODE:
        if (wasReleased(ev)) {
          unmarkResizeNode();
        }
        break;
    }
    if (!eventWasConsumed) {
      super.callbackButton(pt, ev);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackDrag(final Point2D pt, final MouseEvent ev) {
    if (this.currentState == WAITING_DRAG_TO_RESIZE_NODE) {
      this.currentNode.resize(pt, this.compassDirection);
      repaint();
    }
    else {
      super.callbackDrag(pt, ev);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackMove(final Point2D pt, final MouseEvent ev) {
    switch (this.currentState) {
      case START:
        pickNode(pt);
        if (this.currentNode != null) {
          this.compassDirection = this.currentNode.getCompassDirection(pt);
          if (this.compassDirection != null) {
            changeCursor(getCursor());
            this.currentState = WAITING_PRESS_TO_RESIZE_NODE;
          }
          else {
            this.compassDirection = null;
            this.currentNode = null;
          }
        }
        break;
      case WAITING_PRESS_TO_RESIZE_NODE:
        boolean unmark = true;
        pickNode(pt);
        if (this.currentNode != null) {
          final CompassDirection newCompassDirection =
            this.currentNode.getCompassDirection(pt);
          if (newCompassDirection != null) {
            if (!this.compassDirection.equals(newCompassDirection)) {
              this.compassDirection = newCompassDirection;
              changeCursor(getCursor());
            }
            unmark = false;
          }
        }
        if (unmark) {
          unmarkResizeNode();
        }
        break;
    }
    super.callbackMove(pt, ev);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasElementRemoved(final Graph graph, final GraphElement element) {
    reset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasReseted(final Graph graph) {
    reset();
  }

  /**
   * @param cursor
   */
  private void changeCursor(final Cursor cursor) {
    if (getVS() != null) {
      new ChangeCursorMessage(cursor).sendVS(getVS());
    }
  }

  /**
   * Pick
   * 
   * @param pt ponto
   */
  private void pickNode(final Point2D pt) {
    final PickNodeMessage message = new PickNodeMessage(pt);
    message.sendVO(this);
    this.currentNode = message.getNode();
  }

  /**
   * Reinicializao.
   */
  private void reset() {
    unmarkResizeNode();
    if (getVS() != null) {
      changeCursor(getDefaultCursor());
    }
  }

  /**
   * Desmarcar
   */
  private void unmarkResizeNode() {
    changeCursor(getDefaultCursor());
    this.compassDirection = null;
    this.currentNode = null;
    this.currentState = START;
  }

  /**
   * Busca de cursor.
   * 
   * @return cursor
   */
  private Cursor getCursor() {
    return Cursor.getPredefinedCursor(getCursorType());
  }

  /**
   * Busca de cursor ativo.
   * 
   * @return cursor
   */
  private int getCursorType() {
    if (CompassDirection.NORTH.equals(this.compassDirection)) {
      return Cursor.N_RESIZE_CURSOR;
    }
    if (CompassDirection.SOUTH.equals(this.compassDirection)) {
      return Cursor.S_RESIZE_CURSOR;
    }
    if (CompassDirection.EAST.equals(this.compassDirection)) {
      return Cursor.E_RESIZE_CURSOR;
    }
    if (CompassDirection.WEST.equals(this.compassDirection)) {
      return Cursor.W_RESIZE_CURSOR;
    }
    if (CompassDirection.NORTH_EAST.equals(this.compassDirection)) {
      return Cursor.NE_RESIZE_CURSOR;
    }
    if (CompassDirection.NORTH_WEST.equals(this.compassDirection)) {
      return Cursor.NW_RESIZE_CURSOR;
    }
    if (CompassDirection.SOUTH_EAST.equals(this.compassDirection)) {
      return Cursor.SE_RESIZE_CURSOR;
    }
    if (CompassDirection.SOUTH_WEST.equals(this.compassDirection)) {
      return Cursor.SW_RESIZE_CURSOR;
    }
    return Cursor.DEFAULT_CURSOR;
  }

  /**
   * Busca cursor corrente.
   * 
   * @return cursor
   */
  private Cursor getDefaultCursor() {
    return Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR);
  }

  /**
   * Indicao de mouse pressionado.
   * 
   * @param ev evento
   * @return indicativo
   */
  private boolean wasPressed(final MouseEvent ev) {
    return (ev.getButton() == MouseEvent.BUTTON1)
      && (ev.getID() == MouseEvent.MOUSE_PRESSED);
  }

  /**
   * Indicao de mouse solto.
   * 
   * @param ev evento
   * @return indicativo.
   */
  private boolean wasReleased(final MouseEvent ev) {
    return (ev.getButton() == MouseEvent.BUTTON1)
      && (ev.getID() == MouseEvent.MOUSE_RELEASED);
  }
}
