package csbase.client.applications.preferenceviewer.tree;

import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeSelectionModel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.preferenceviewer.PreferencePanel;
import csbase.client.applications.preferenceviewer.PreferenceViewer;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.util.PreferenceEditorListener;

/**
 * Painel que exibe a rvore de preferncias.
 * 
 * @author Tecgraf
 */
public class PreferenceTreePanel extends JPanel {

  /** Referncia para o painel que gerencia a exibio das preferncias. */
  private PreferencePanel preferencePanel;

  /** Referncia para a aplicao. */
  private PreferenceViewer app;

  /** rvore de preferncias. */
  private JTree tree;

  /** Referncia para a ltima categoria selecionada. */
  private PreferenceCategory lastSelected;

  /** Mapa com os listeneres cadastrados em cada valor. */
  private Map<PreferenceValue<?>, List<PreferenceEditorListener>> listeners;

  /**
   * Construtor padro.
   * 
   * @param preferencePanel - referncia para o painel que exibe as
   *        preferncias.
   * @param app - referncia para a aplicao.
   */
  public PreferenceTreePanel(PreferencePanel preferencePanel,
    PreferenceViewer app) {
    super(new GridBagLayout());
    listeners =
      new HashMap<PreferenceValue<?>, List<PreferenceEditorListener>>();

    this.preferencePanel = preferencePanel;
    this.app = app;
    this.tree = createTree();

    add(new JScrollPane(tree), new GBC(0, 0).both());
  }

  /** Marca o primeiro elemento da rvore como visvel. */
  public void defaultSelection() {
    tree.setSelectionRow(0);
  }

  /**
   * Reconstroi a rvore de preferncias a partir de uma nova categoria raz.
   * 
   * @param newRoot - nova raz.
   */
  public void rebuildTree(PreferenceCategory newRoot) {
    tree.setModel(createTreeModel(newRoot));
    tree.setRootVisible(!newRoot.isRoot());
    defaultSelection();
  }

  /**
   * Retorna a ltima categoria selecionada.
   * 
   * @return ltima categoria selecionada.
   */
  public PreferenceCategory getLastSelected() {
    return lastSelected;
  }

  /** Remove todos os ouvintes. */
  public void removeAllListeners() {
    PreferenceCategory allPrefs =
      PreferenceManager.getInstance().loadPreferences();

    for (PreferenceCategory pc : allPrefs.getCategories()) {
      removeListeners(pc);
    }
  }

  /**
   * Remove todos os ouvintes.
   * 
   * @param pc - categoria de preferncias.
   */
  private void removeListeners(PreferenceCategory pc) {
    for (PreferenceValue<?> pv : pc.getPreferences(PreferencePolicy.READ_ONLY,
      PreferencePolicy.READ_WRITE)) {
      if (listeners.containsKey(pv)) {
        for (PreferenceEditorListener l : listeners.get(pv)) {
          pv.getEditor().removeListener(l);
        }
      }
    }

    for (PreferenceCategory child : pc.getCategories()) {
      removeListeners(child);
    }
  }

  /**
   * Constri a rvore que exibe todas as categorias de preferncias.
   * 
   * @return rvore de preferncias.
   */
  private JTree createTree() {
    PreferenceManager pm = PreferenceManager.getInstance();
    PreferenceCategory rootPref = pm.loadPreferences();

    JTree tree = new JTree(createTreeModel(rootPref));
    tree.getSelectionModel().setSelectionMode(
      TreeSelectionModel.SINGLE_TREE_SELECTION);
    tree.putClientProperty("JTree.lineStyle", "None");
    tree.setRowHeight(20);
    tree.setRootVisible(!rootPref.isRoot());

    tree.addTreeSelectionListener(new TreeSelectionListener() {
      @Override
      public void valueChanged(TreeSelectionEvent e) {
        JTree tree = (JTree) e.getSource();
        PreferenceCategory pc =
          (PreferenceCategory) tree.getLastSelectedPathComponent();

        if (pc != null) {
          preferencePanel.showPreferencePanel(pc);
          tree.expandPath(tree.getSelectionPath());

          memorizeLastSelected(pc);
        }
      }
    });

    tree.setCellRenderer(new PreferenceTreeCellRenderer(preferencePanel, app));

    return tree;
  }

  /**
   * Memoriza a ltima categoria selecionada e adiciona os ouvintes nos editores
   * da categoria.
   * 
   * @param pc - categoria selecionada.
   */
  private void memorizeLastSelected(final PreferenceCategory pc) {
    lastSelected = pc;

    PreferenceEditorListener l = new PreferenceEditorListener() {
      @Override
      public void action() {
        preferencePanel.addModifiedCategory(pc);
        repaint();
      }
    };

    for (PreferenceValue<?> pv : pc.getPreferences(PreferencePolicy.READ_ONLY,
      PreferencePolicy.READ_WRITE)) {
      if (!listeners.containsKey(pv)) {
        pv.getEditor().addListener(l);

        List<PreferenceEditorListener> editorListeners =
          new ArrayList<PreferenceEditorListener>();
        editorListeners.add(l);
        listeners.put(pv, editorListeners);
      }
    }
  }

  /**
   * Cria o modelo para a rvore que exibe as preferncias.
   * 
   * @param pc - categoria raz.
   * @return modelo da rvore.
   */
  private TreeModel createTreeModel(PreferenceCategory pc) {
    List<String> order = app.getStringListSpecificProperty("tree.order");

    PreferenceTreeModel model = new PreferenceTreeModel(pc);
    model.setOrder(order);
    return model;
  }

}