/*
 * ProjectDirectoryChooser.java
 * 
 * $Author: oikawa $ $Revision: 150777 $ - $Date: 2006-08-16 18:31:12 -0300
 * (qua, 16 ago 2006) $
 */
package csbase.client.project;

import java.awt.BorderLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.kernel.ClientException;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.filters.ProjectFileDirectoryFilter;

/**
 * A classe <code>ProjectDirectoryChooser</code> mostra um dilogo modal para
 * seleo de um diretrio de projeto.
 * 
 * @author Maria Julia
 */
public class ProjectDirectoryChooser {
  /** O ttulo do dilogo */
  private static final String TITLE = LNG.get("DIR_CHOOSER_TITLE");

  /** O caminho do diretrio selecionado */
  private ProjectTreePath treePath;

  /** A rvore de um projeto */
  private ProjectTree projectTree;

  /** O boto default */
  private JButton defaultButton;

  /** Campo de texto correspondente ao nome do diretrio selecionado */
  private JTextField dirNameText;

  /** O ouvinte de selees na rvore de projetos */
  private ProjectTreeSelectionListener treeSelectionListener;

  /** A janela pai onde o dilogo ser exibido */
  private Window owner;

  /** O dilogo propriamente dito */
  private DesktopComponentDialog dialog;

  /**
   * Retorna o diretrio selecionado no dilogo.
   * 
   * @return O diretrio selecionado ou null, caso nenhum diretrio tenha sido
   *         selecionado.
   */
  public ProjectTreePath getSelectedDirectory() {
    return this.treePath;
  }

  /**
   * Cria um dilogo que aguarda a seleo de um diretrio de projeto.
   * 
   * <p>
   * S deve ser chamado depois que a rvore j est criada.
   * </p>
   */
  private void buildDialog() {
    /* Cria o painel com a rvore e os campos de texto */
    JPanel contentPanel = new JPanel(new BorderLayout(0, 5));
    contentPanel.add(new JScrollPane(this.projectTree.getTree()),
      BorderLayout.CENTER);
    contentPanel.add(makeTextFieldPanel(), BorderLayout.SOUTH);
    /* Cria o painel principal da janela */
    JPanel mainPanel = new JPanel(new BorderLayout(0, 5));
    mainPanel.add(contentPanel, BorderLayout.CENTER);
    mainPanel.add(makeButtonPanel(), BorderLayout.SOUTH);

    this.dialog = new DesktopComponentDialog(this.owner, TITLE);
    this.dialog.getContentPane().add(mainPanel);
    this.dialog.getRootPane().setDefaultButton(this.defaultButton);
  }

  /**
   * Cria o painel com os botes do dilogo.
   * 
   * <ul>
   * <li>
   * o boto default, que possui a confirmao da seleo.</li>
   * <li>
   * o boto de cancelamento da seleo.</li>
   * </ul>
   * 
   * 
   * @return O painel com os botes.
   */
  private JPanel makeButtonPanel() {
    JPanel butPanel = new JPanel();
    this.defaultButton =
      (JButton) butPanel.add(new JButton(LNG.get("DIR_CHOOSER_OPEN")));
    this.defaultButton.setEnabled(false);
    this.defaultButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ProjectDirectoryChooser.this.handleAction();
      }
    });

    JButton cancelButton =
      (JButton) butPanel.add(new JButton(LNG.get("DIR_CHOOSER_CANCEL")));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ProjectDirectoryChooser.this.treePath = null;
        ProjectDirectoryChooser.this.close();
      }
    });

    ClientUtilities.adjustEqualSizes(this.defaultButton, cancelButton);
    return butPanel;
  }

  /**
   * Cria o painel com o campo de texto com o nome do diretrio selecionado.
   * 
   * @return O painel com o nome do diretrio selecionado.
   */
  private JPanel makeTextFieldPanel() {
    this.dirNameText = new JTextField(30);
    this.dirNameText.setEditable(false);
    this.dirNameText.setText("");
    JComponent[][] components =
      { { new JLabel(LNG.get("DIR_CHOOSER_SELECTED_NAME")), this.dirNameText } };
    JPanel txtPanel = GUIUtils.createBasicGridPanel(components);
    return txtPanel;
  }

  /**
   * Executa a ao do boto default. Nesse caso, a ao  guardar o diretrio
   * selecionado pelo usurio em um campo desse objeto e fechar o dilogo.
   */
  private void handleAction() {
    ClientProjectFile selectedFile = this.projectTree.getSelectedFile();
    if (selectedFile == null) {
      StandardErrorDialogs.showErrorDialog(this.dialog,
        LNG.get("DIR_CHOOSER_ERROR") + " - " + TITLE,
        LNG.get("DIR_CHOOSER_NOT_SELECTED_ERROR"));
      return;
    }
    this.treePath = new ProjectTreePath(selectedFile);
    this.close();
  }

  /**
   * Fecha o dilogo, removendo os seus ouvintes que esto cadastrados na rvore
   * de projetos.
   */
  private void close() {
    this.projectTree.removeTreeSelectionListener(this.treeSelectionListener);
    this.projectTree.release();
    this.dialog.close();
  }

  /**
   * Constri uma viso de dilogo para seleo de um diretrio de um projeto.
   * 
   * @param project O projeto do qual o usurio vai selecionar um diretrio.
   * @param owner A janela pai deste dilogo.
   * 
   * @throws ClientException Caso ocorra algum erro durante a criao da rvore
   *         de projetos.
   */
  public ProjectDirectoryChooser(final CommonClientProject project, Window owner)
    throws ClientException {
    this.owner = owner;
    this.projectTree =
      ProjectTreeFactory.getInstance().makeProjectTree(owner, TITLE, project,
        ProjectFileDirectoryFilter.getInstance());
    this.projectTree.setSingleSelectionMode(true);
    this.treeSelectionListener = new ProjectTreeSelectionListener() {
      @Override
      public void update(ProjectTreeSelectionEvent event) {
        ClientProjectFile[] selectedFiles = event.getSelectedFiles();
        if (selectedFiles.length == 0) {
          ProjectDirectoryChooser.this.dirNameText.setText("");
          ProjectDirectoryChooser.this.defaultButton.setEnabled(false);
        }
        else {
          String[] filePath = selectedFiles[0].getPath();
          ProjectDirectoryChooser.this.dirNameText.setText(ProjectTreePath
            .makePath(project.getName(), "/", filePath));
          ProjectDirectoryChooser.this.defaultButton.setEnabled(true);
        }
      }
    };
    this.projectTree.addTreeSelectionListener(this.treeSelectionListener);
    this.buildDialog();
    this.show();
  }

  /**
   * Exibe o dilogo centralizado em sua janela pai.
   */
  private void show() {
    this.dialog.pack();
    this.dialog.center(this.owner);
    this.dialog.setVisible(true);
  }
}
