package csbase.client.algorithms.commands.view;

import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;

import javax.swing.JDialog;
import javax.swing.JFrame;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.algorithms.AlgorithmConfiguratorFactory;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.cache.events.AbstractCommandUpdatedEventListener;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.configurator.FlowAlgorithmConfiguratorView;
import csbase.client.applications.flowapplication.filters.AddNodePopupActionFilter;
import csbase.client.applications.flowapplication.filters.PopupFilter;
import csbase.client.applications.flowapplication.filters.WorkspaceFilter;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.graph.GraphNodeImageDecoration.DecorationType;
import csbase.client.applications.flowapplication.graph.actions.Action;
import csbase.client.applications.flowapplication.graph.actions.GraphElementAction;
import csbase.client.applications.flowapplication.messages.PickNodeMessage;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.kernel.ClientException;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationInfo.FinalizationInfoType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;

/**
 * Dilogo de visualizao de comandos de fluxo.
 * 
 * @author Tecgraf / PUC-Rio
 */
final class FlowAlgorithmCommandView extends AbstractAlgorithmCommandView {

  /**
   * Painel de fluxo
   */
  private FlowAlgorithmConfiguratorView configuratorView;

  /**
   * Tab preferencial
   */
  private TabType preferredTab;

  /**
   * @param index identificador da view. Usado para garantir a unicidade.
   * @param owner A janela pai deste dilogo. No pode ser nula e tem que ser
   *        uma instncia de {@link JFrame} ou {@link JDialog}.
   * @param command O comando (No aceita {@code null}).
   * @param configurator O configurador a ter seus parmetros apresentados.
   *        Deve-se ignorar o configurador do comando pois pode ser diferente
   *        deste. Isso ocorrerira no caso de estar representando um n de um
   *        comando de fluxo (No aceita {@code null}). arquivos de log do
   *        comando.
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  public FlowAlgorithmCommandView(final Object index,
    final DesktopComponentFrame owner, final CommandInfo command,
    final AlgorithmConfigurator configurator) throws ClientException {
    super(index, owner, command, configurator, CommandViewType.FLOW, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void initialize(AlgorithmConfigurator configurator)
    throws ClientException {

    AlgorithmConfiguratorFactory configuratorFactory =
      AlgorithmConfiguratorFactory.getInstance();
    configuratorView =
      (FlowAlgorithmConfiguratorView) configuratorFactory.createReportView(
        owner, configurator);
  }

  /**
   * Mostra essa janela dando preferncia para abrir com a aba escolhida.
   * 
   * @param tab Qual a aba gostaria que estivesse selecionada quando esta janela
   *        for criada. No  garantido que esta janela abra com a aba escolhida
   *        selecionada, pois aquela tab pode nem existir caso seja uma aba de
   *        log.
   */
  @Override
  public void show(final TabType tab) {
    this.preferredTab = tab;
    super.show(tab);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ConfigurationReportTab createConfigurationReportTab(
    final AlgorithmConfigurator configurator) throws ClientException {
    return new FlowConfiguratorReportTab(configurator);
  }

  /**
   * @author Tecgraf/PUC-Rio
   */
  private final class FlowConfiguratorReportTab extends ConfigurationReportTab {

    /**
     * Recebe notificaes sobre atualizaces feitas no comando.
     */
    private AbstractCommandUpdatedEventListener commandListener;

    /**
     * Construtor
     * 
     * @param configurator configurador
     * @throws ClientException se houver problemas
     */
    FlowConfiguratorReportTab(final AlgorithmConfigurator configurator)
      throws ClientException {

      addCommandListener();
      createFilters(configuratorView.getWorkspace());

      setLayout(new GridLayout());
      add(configuratorView.getMainComponent(), new GBC(0, 0).both());
      addCommandStatusDecoration(getCommand(), configuratorView.getGraph());
    }

    /**
     * Adiciona um listener para o comando a ser visualizado, caso seja
     * necessrio.
     */
    private void addCommandListener() {
      CommandInfo command = getCommand();
      if (command != null && command.getStatus() != CommandStatus.FINISHED) {
        this.commandListener = createCommandListener(command);
        CommandsCache.getInstance().addEventListener(commandListener);
      }
    }

    /**
     * Cria um listener para o comando a ser visualizado, para que a viso possa
     * ser atualizada.
     * 
     * @param command O comando a ser visualizado.
     * @return um listener para o comando.
     */
    private AbstractCommandUpdatedEventListener createCommandListener(
      CommandInfo command) {
      return new AbstractCommandUpdatedEventListener(command.getProjectId(),
        command.getId()) {
        /**
         * {@inheritDoc}
         */
        @Override
        public void eventFired(CommandUpdatedEvent.Type type, CommandInfo cmd) {
          try {
            addCommandStatusDecoration(cmd, configuratorView.getGraph());
            if (cmd.getStatus() == CommandStatus.FINISHED) {
              if (commandListener != null) {
                CommandsCache.getInstance()
                  .removeEventListener(commandListener);
              }
            }
          }
          catch (ClientException e) {
            StandardErrorDialogs.showErrorDialog(FlowAlgorithmCommandView.this,
              e);
          }
        }
      };
    }

    /**
     * Adiciona as decoraes que representam o estado do comando s caixas do
     * fluxo.
     * 
     * @param graph O grafo que contm as caixas do fluxo.
     * @param command O comando atualizado.
     * 
     * @throws ClientException em caso de falha.
     */
    private void addCommandStatusDecoration(final CommandInfo command,
      final Graph graph) throws ClientException {
      if (command != null) {
        if (command.getStatus() == CommandStatus.FINISHED) {
          FlowAlgorithmConfigurator configurator =
            (FlowAlgorithmConfigurator) getConfigurator();
          CommandFinalizationInfo info = command.getFinalizationInfo();
          if (info != null) {
            if (configurator.canBeRunAsSimpleCommand()) {
              GraphNode node = graph.getNodeCollection().iterator().next();
              decorateNode(node, info, false, false);
            }
            else if (info.getInfoType() == FinalizationInfoType.EXTENDED) {
              ExtendedCommandFinalizationInfo finalizationInfo =
                (ExtendedCommandFinalizationInfo) info;
              for (GraphNode node : graph.getNodeCollection()) {
                CommandFinalizationInfo nodeFinalizationInfo =
                  finalizationInfo.getFinalizationInfoForNode(node.getId());
                Integer guiltyNodeId = finalizationInfo.getGuiltyNodeId();
                /*
                 * Se existir o id do n "culpado" pelo erro, quer dizer que o
                 * comando foi interrompido pelo flowmonitor. Neste caso, os
                 * algoritmos que no retornaram cdigo de sada podem ser
                 * considerados como "interrompidos" ao invs de
                 * "faltando cdigo de retorno".
                 */
                boolean commandWasInterrupted = false;
                boolean isGuilty = false;
                if (guiltyNodeId != null) {
                  commandWasInterrupted = true;
                  if (guiltyNodeId == node.getId()) {
                    isGuilty = true;
                  }
                }
                decorateNode(node, nodeFinalizationInfo, commandWasInterrupted,
                  isGuilty);
              }
            }
          }
        }
        else {
          DecorationType decoration;
          switch (command.getStatus()) {
            case SYSTEM_FAILURE:
              decoration = DecorationType.SYSTEM_FAILURE;
              break;
            case SCHEDULED:
              decoration = DecorationType.SCHEDULED;
              break;
            case INIT:
            case UPLOADING:
            case EXECUTING:
            case DOWNLOADING:
              decoration = DecorationType.EXECUTION;
              break;
            default:
              decoration = DecorationType.BLANK;
          }

          for (GraphNode graphNode : graph.getNodeCollection()) {
            if (!graphNode.isBypassed()) {
              graphNode.addImageDecoration(decoration);
            }
          }
        }
      }
      else {
        throw new ClientException("No foi possvel encontrar comando!");
      }
    }

    /**
     * Adiciona a decorao que representa o estado de uma caixa do fluxo a
     * partir do estado de finalizao do algoritmo.
     * 
     * @param node A caixa a ser decorada.
     * @param finalizationInfo O estado de finalizao da caixa.
     * @param commandWasInterrupted Indica se a execuo do comando foi
     *        interrompida.
     * @param isGuilty Indica se o n foi o responsvel pela interrupo do
     *        fluxo.
     */
    private void decorateNode(GraphNode node,
      CommandFinalizationInfo finalizationInfo, boolean commandWasInterrupted,
      boolean isGuilty) {
      DecorationType decoration = DecorationType.BLANK;
      if (!node.isBypassed()) {
        if (commandWasInterrupted && isGuilty) {
          decoration = DecorationType.FATAL_ERROR;
        }
        else {
          switch (finalizationInfo.getFinalizationType()) {
            case EXECUTION_ERROR:
              if (commandWasInterrupted) {
                decoration = DecorationType.NON_FATAL_ERROR;
              }
              else {
                decoration = DecorationType.EXECUTION_ERROR;
              }
              break;
            case NO_EXIT_CODE:
              if (commandWasInterrupted) {
                decoration = DecorationType.INTERRUPTED;
              }
              else {
                decoration = DecorationType.EXECUTION_UNKNOWN;
              }
              break;
            case SUCCESS:
              if (finalizationInfo.hasWarnings()) {
                decoration = DecorationType.EXECUTION_WARNING;
              }
              else {
                decoration = DecorationType.EXECUTION_SUCCESS;
              }
              break;
            default:
              decoration = DecorationType.BLANK;
              break;
          }
        }
      }
      node.addImageDecoration(decoration);
    }

    /**
     * @see Tab#getTitle()
     */
    @Override
    public String getTitle() {
      return LNG.get("FlowAlgorithmCommandView.tab.params.title");
    }

    /**
     * Cria os filtros.
     * 
     * @param workspace workspace
     */
    private void createFilters(final Workspace workspace) {

      // Cria o menu popup.
      new AddNodePopupActionFilter(workspace) {
        @Override
        protected Action createAction(GraphNode graphNode, Point2D point) {
          return new ShowAlgorithmCommandViewPopupAction(graphNode);
        }
      }.attach();
      new PopupFilter(workspace).attach();
      // Cria a ao que ser chamada quando houver um duplo clique em algum n.
      new ShowAlgorithmCommandViewDoubleClickAction(workspace).attach();
    }

    /**
     * Ao de que exibe a viso do comando de um n de fluxo de algoritmos.
     * Nenhum log ser mostrado nesta viso pois os arquivos de log foram
     * persistidos como sendo do fluxo e no dos ns.
     */
    private class ShowAlgorithmCommandViewDoubleClickAction extends
      WorkspaceFilter {

      /**
       * @param workspace A rea de trabalho (No aceita {@code null}).
       */
      public ShowAlgorithmCommandViewDoubleClickAction(final Workspace workspace) {
        super(workspace);
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void callbackButton(Point2D pt, MouseEvent ev) {
        if (ev.getButton() == MouseEvent.BUTTON1
          && ev.getID() == MouseEvent.MOUSE_CLICKED && ev.getClickCount() == 2) {
          PickNodeMessage pickNodeMessage = new PickNodeMessage(pt);
          pickNodeMessage.sendVO(this);
          GraphNode node = pickNodeMessage.getNode();
          if (node != null) {
            try {
              AlgorithmCommandViewFactory
                .showView(FlowAlgorithmCommandView.this, getCommand(), node,
                  preferredTab);
            }
            catch (Exception ex) {
              StandardErrorDialogs.showErrorDialog(
                FlowAlgorithmCommandView.this, ex);
            }
          }
        }
        super.callbackButton(pt, ev);
      }
    }

    /**
     * Ao de menu popup que exibe a viso do comando de um n de fluxo de
     * algoritmos.
     */
    private final class ShowAlgorithmCommandViewPopupAction extends
      GraphElementAction {

      /**
       * Cria a ao.
       * 
       * @param node O n (No aceita {@code null}).
       */
      public ShowAlgorithmCommandViewPopupAction(final GraphNode node) {
        super(node,
          "FlowAlgorithmCommandView.tab.params.action.show.node.params",
          ApplicationImages.ICON_INFORMATION_16);
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void actionPerformed(ActionEvent e) {
        GraphNode node = (GraphNode) getElement();
        if (node != null) {
          try {
            AlgorithmCommandViewFactory.showView(FlowAlgorithmCommandView.this,
              getCommand(), node, preferredTab);
          }
          catch (Exception ex) {
            StandardErrorDialogs.showErrorDialog(FlowAlgorithmCommandView.this,
              ex);
          }
        }
      }
    }
  }
}
