package csbase.client.algorithms.parameters;

import java.awt.Color;
import java.awt.Font;
import java.util.HashMap;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JTextField;
import javax.swing.border.Border;
import javax.swing.text.JTextComponent;

import csbase.client.algorithms.parameters.ParameterView.Mode;

/**
 * Propriedades dos componentes de interface grfica dos parmetros.  utilizada
 * para configurar as propriedades visuais dos componentes de forma consistente
 * de acordo com o modo de visualizao {@link ParameterView.Mode}
 */
public class ComponentProperties {

  /**
   * Propriedades dos componentes mapeadas por modo de viso do parmetro (
   * {@link ParameterView.Mode}).
   */
  private static final Map<Mode, ComponentProperties> COMPONENT_PROPERTIES;

  /** Cor do plano de fundo do componente. */
  private Color bgColor;

  /** Cor do primeiro plano do componente. */
  private Color fgColor;

  /** Borda do componente. */
  private Border border;

  /** Fonte utilizada no componente. */
  private Font font;

  /** Indica se o componente  opaco. */
  private boolean isOpaque;

  /** Indica se o componente  editvel. */
  private boolean isEditable;

  static {
    COMPONENT_PROPERTIES = new HashMap<Mode, ComponentProperties>();
    for (Mode mode : Mode.values()) {
      COMPONENT_PROPERTIES.put(mode, createComponentProperties(mode));
    }
  }

  /**
   * Construtor.
   * 
   * @param bgColor a cor do plano de fundo.
   * @param fgColor a cor do primeiro plano.
   * @param border a borda.
   * @param font a fonte.
   * @param isOpaque indica se  opaco.
   * @param isEditable indica se  editvel.
   */
  private ComponentProperties(Color bgColor, Color fgColor, Border border,
    Font font, boolean isOpaque, boolean isEditable) {
    super();
    this.bgColor = bgColor;
    this.fgColor = fgColor;
    this.border = border;
    this.font = font;
    this.isOpaque = isOpaque;
    this.isEditable = isEditable;
  }

  /**
   * Obtm a instncia nica de {@link ComponentProperties} para o modo de
   * visualizao ({@link ParameterView.Mode}).
   * 
   * @param mode o modo
   * @return a instncia de ComponentProperties
   */
  public static final ComponentProperties getInstance(Mode mode) {
    return COMPONENT_PROPERTIES.get(mode);
  }

  /**
   * Configura o componente de campo de texto a partir das propriedades do modo
   * de visualizao indicado.
   * 
   * @param component o componente.
   * @param mode o modo de viso do parmetro dono do componente.
   * @param setBorder indica se deve configurar a borda do componente.
   */
  public static final void setProperties(JTextComponent component, Mode mode,
    boolean setBorder) {
    ComponentProperties properties = getInstance(mode);
    component.setEditable(properties.isEditable());
    setProperties(component, properties, setBorder);
  }

  /**
   * Configura o componente a partir das propriedades do modo de visualizao
   * indicado.
   * 
   * @param component o componente.
   * @param mode o modo de viso do parmetro dono do componente.
   * @param setBorder indica se deve configurar a borda do componente.
   */
  public static final void setProperties(JComponent component, Mode mode,
    boolean setBorder) {
    ComponentProperties properties = getInstance(mode);
    setProperties(component, properties, setBorder);
  }

  /**
   * Configura o componente a partir das propriedades indicadas.
   * 
   * @param component o componente.
   * @param properties as propriedades.
   * @param setBorder indica se deve configurar a borda do componente.
   */
  public static final void setProperties(JComponent component,
    ComponentProperties properties, boolean setBorder) {
    component.setBackground(properties.getBackground());
    component.setForeground(properties.getForeGround());
    component.setFont(properties.getFont());
    component.setOpaque(properties.isOpaque());
    if (setBorder) {
      component.setBorder(properties.getBorder());
    }
  }

  /**
   * Obtm a cor do plano de fundo.
   * 
   * @return a cor.
   */
  public Color getBackground() {
    return bgColor;
  }

  /**
   * Obtm a cor do primeiro plano.
   * 
   * @return a cor.
   */
  public Color getForeGround() {
    return fgColor;
  }

  /**
   * Obtm a borda.
   * 
   * @return a borda.
   */
  public Border getBorder() {
    return border;
  }

  /**
   * Obtm a fonte.
   * 
   * @return a fonte.
   */
  public Font getFont() {
    return font;
  }

  /**
   * Determina se o componente deve ser opaco.
   * 
   * @return verdadeiro se deve ser opaco ou falso, caso contrrio.
   */
  public boolean isOpaque() {
    return isOpaque;
  }

  /**
   * Determina se o componente deve ser editvel.
   * 
   * @return verdadeiro se deve ser editvel ou falso, caso contrrio.
   */
  public boolean isEditable() {
    return isEditable;
  }

  /**
   * Cria as propriedades visuais dos componentes para o modo de visualizao
   * indicado.
   * 
   * @param mode o modo de visualizao.
   * @return as propriedades.
   */
  private static ComponentProperties createComponentProperties(Mode mode) {
    JTextField template = new JTextField();
    template.setEditable(Mode.CONFIGURATION == mode);

    Color bgColor = template.getBackground();
    Color fgColor = template.getForeground();
    Font font = template.getFont();
    boolean isOpaque = template.isOpaque();
    boolean isEditable = template.isEditable();

    Color borderColor =
      Mode.CONFIGURATION == mode ? new Color(0x7a, 0x8a, 0x99) : new Color(
        0xb8, 0xcf, 0xe5);
    Border border = BorderFactory.createLineBorder(borderColor);

    return new ComponentProperties(bgColor, fgColor, border, font, isOpaque,
      isEditable);
  }
}
