package csbase.client.csdk.v2.core;

import java.awt.Window;
import java.nio.charset.Charset;
import java.util.Locale;
import java.util.Map;

import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.csdk.v2.application.CSDKApplication;
import csbase.client.util.StandardErrorDialogs;
import csdk.v2.api.application.ApplicationException;
import csdk.v2.api.application.IApplicationContext;
import csdk.v2.api.application.IMessage;
import csdk.v2.api.core.ICSDKEnvironment;
import csdk.v2.api.core.IContext;
import tecgraf.javautils.core.lng.LNG;

/**
 * Implementao da interface das aplicaes com o ambiente CSBase, incluindo o
 * acesso aos contextos disponveis.
 */
public class CSDKEnvironment implements ICSDKEnvironment {

  /**
   * Contextos da aplicao.
   */
  private final Map<Class<? extends IContext>, IContext> contexts;

  /**
   * Construtor.
   * 
   * @param contexts o mapa de contextos da aplicao.
   */
  public CSDKEnvironment(Map<Class<? extends IContext>, IContext> contexts) {
    this.contexts = contexts;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void finishApplication() {
    CSDKApplication app = getApplicationInstance();
    app.closeApplication();
  }

  /**
   * Obtm o contexto da aplicao a partir da classe que o implementa.
   * 
   * @param contextClass a classe do contexto.
   * @return o contexto.
   */
  @Override
  public <T extends IContext> T getContext(Class<T> contextClass) {
    if (contextClass == null) {
      return null;
    }
    IContext context = this.contexts.get(contextClass);
    if (context != null) {
      try {
        T t = contextClass.cast(context);
        if (!context.isActive()) {
          t.onContextActivate();
        }
        return t;
      }
      catch (Exception e) {
        handleException(e, null);
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void handleException(Exception error, Window parent) {
    IApplicationContext appContext = (IApplicationContext) contexts.get(
      IApplicationContext.class);
    String title = null;
    if (appContext != null) {
      title = appContext.getApplicationName();
    }
    StandardErrorDialogs.showExceptionDialog(parent, title, error);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String runApplication(String id) throws ApplicationException {
    final ApplicationManager manager = ApplicationManager.getInstance();
    try {
      if (!manager.hasApplicationRegistry(id)) {
        return null;
      }
      ApplicationType app = manager.runApplication(id);
      if (app == null) {
        return null;
      }
      return app.getInstanceId();
    }
    catch (csbase.client.applicationmanager.ApplicationException e) {
      throw new ApplicationException(e);
    }
  }

  /**
   * Obtm a instncia da aplicao relacionada a esse contexto.
   * 
   * @return a instncia.
   */
  private CSDKApplication getApplicationInstance() {
    final ApplicationManager manager = ApplicationManager.getInstance();
    IApplicationContext appContext = (IApplicationContext) contexts.get(
      IApplicationContext.class);
    if (appContext != null) {
      return (CSDKApplication) manager.getApplicationInstance(appContext
        .getInstanceId());
    }
    else {
      return null;
    }
  }

  /** {@inheritDoc} */
  @Override
  public void sendAsyncMessage(String appInstanceId, IMessage message)
    throws ApplicationException {
    ApplicationManager manager = ApplicationManager.getInstance();
    ApplicationType senderApp = getApplicationInstance();
    String senderId = null;
    if (senderApp != null) {
      senderId = senderApp.getInstanceId();
    }
    manager.sendAsyncMessage(appInstanceId, message.getType(), message
      .getValue(), senderId);
  }

  /** {@inheritDoc} */
  @Override
  public void broadcastAsyncMessage(IMessage message)
    throws ApplicationException {
    ApplicationManager manager = ApplicationManager.getInstance();
    ApplicationType senderApp = getApplicationInstance();
    String senderId = null;
    if (senderApp != null) {
      senderId = senderApp.getInstanceId();
    }
    manager.broadcastAsyncMessage(message.getType(), message.getValue(),
      senderId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object sendSyncMessage(String appInstanceId, IMessage message)
    throws ApplicationException {
    ApplicationManager manager = ApplicationManager.getInstance();
    ApplicationType senderApp = getApplicationInstance();
    String senderId = null;
    if (senderApp != null) {
      senderId = senderApp.getInstanceId();
    }
    return manager.sendSyncMessage(appInstanceId, message.getType(), message
      .getValue(), senderId);
  }

  /**
   * Limpa os contextos.
   */
  public void cleanup() {
    for (IContext context : contexts.values()) {
      if (context.isActive()) {
        context.onContextDeactivate();
      }
    }
    contexts.clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Locale getLocale() {
    return LNG.getLocale();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getServerCharset() {
    return Client.getInstance().getSystemDefaultCharset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getCharset() {
    return Client.getInstance().getClientHostCharset();
  }

}
