/*
 * $Id: ApplicationType.java 176509 2016-10-05 20:44:22Z isabella $
 */
package csbase.client.applicationmanager;

import java.io.InputStream;
import java.net.URL;
import java.util.List;
import java.util.Locale;
import java.util.Observable;

import javax.swing.ImageIcon;
import javax.swing.JOptionPane;

import csbase.logic.applicationservice.ApplicationRegistry;
import tecgraf.javautils.core.timestamp.TStamp32;

/**
 * Implementa uma aplicao genrica.
 *
 * @author Tecgraf
 */
public abstract class ApplicationType extends Observable {

  /**
   * Nome do diretrio de resources das aplicaes.
   */
  private static final String RES_DIR_NAME = "resources";

  /**
   * Identificador do tipo da aplicao
   */
  private ApplicationRegistry applicationRegistry = null;

  /**
   * Identificador da instncia
   */
  private final String instanceId;

  /**
   * Mtodo protegido para lanamento de aplicao.
   *
   * @see #startApplication()
   * @throws ApplicationException em caso de erro.
   */
  final protected void launchApplication() throws ApplicationException {
    startApplication();
  }

  /**
   * Obtm os tipos de arquivo que abrem diretamente esta aplicao.
   *
   * @return os tipos de arquivo.
   */
  final public List<String> getFileTypes() {
    return this.applicationRegistry.getFileTypes();
  }

  /**
   * Obtm o registry da aplicao.
   *
   * @return o registry.
   */
  final protected ApplicationRegistry getApplicationRegistry() {
    return this.applicationRegistry;
  }

  /**
   * Obtm o id do tipo da aplicao.
   *
   * @return o id.
   */
  public final String getId() {
    return this.applicationRegistry.getId();
  }

  /**
   * Obtm o id da instncia da aplicao.
   *
   * @return o id.
   */
  public final String getInstanceId() {
    return this.instanceId;
  }

  /**
   * Obtm o id da instncia da aplicao.
   *
   * @return o id
   */
  final public int getInstanceIndex() {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    final int id = appManager.getInstanceIndex(this);
    return id;
  }

  /**
   * Obtm o cone da aplicao.
   *
   * @return a imagem.
   */
  public final ImageIcon getImageIcon() {
    final byte[] imgDef = applicationRegistry.getImageDefinition();
    if (imgDef == null) {
      return null;
    }
    return new ImageIcon(imgDef);
  }

  /**
   * Obtm o nome da aplicao.
   *
   * @return o nome.
   */
  public final String getName() {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    return appManager.getApplicationName(applicationRegistry);
  }

  /**
   * Obtm a verso da aplicao com nmero (<code>double</code>) para
   * comparao.
   *
   * @return texto indicativo.
   */
  public final int getVersionNumber() {
    return this.applicationRegistry.getVersionNumber();
  }

  /**
   * Obtm a verso da aplicao.
   *
   * @return texto indicativo.
   */
  public final String getVersion() {
    return this.applicationRegistry.getVersion();
  }

  /**
   * Obtm o e-mail do autor da aplicao.
   *
   * @return o nome.
   */
  public final String getAuthorMail() {
    return this.applicationRegistry.getAuthorMail();
  }

  /**
   * Obtm o nome do autor da aplicao.
   *
   * @return o nome.
   */
  public final String getAuthorName() {
    return this.applicationRegistry.getAuthorName();
  }

  /**
   * Obtm a descrio da aplicao.
   *
   * @return a descrio.
   */
  public final String getDescription() {
    final ApplicationManager applicationManager =
      ApplicationManager.getInstance();

    return applicationManager.getApplicationDescription(applicationRegistry);
  }

  /**
   * Construtor protegido para atualizao do campo de identificador de
   * instncia.
   *
   * @param id o identificador da aplicao.
   */
  protected ApplicationType(final String id) {
    this.instanceId = generateInstanceId(id);
    final ApplicationManager applicationManager =
      ApplicationManager.getInstance();
    this.applicationRegistry = applicationManager.getApplicationRegistry(id);
  }

  /**
   * Gera o identificador da instncia da aplicao.
   *
   * @param appId o identificador da aplicao.
   * @return o identificador da instncia.
   */
  private String generateInstanceId(String appId) {
    TStamp32 idSuffix = new TStamp32();
    return appId + "_" + idSuffix;
  }

  /**
   * Consulta o valor inteiro de uma propriedade
   *
   * @param propName o nome
   * @param defaultValue o valor default
   * @return a propriedade
   */
  public final int getIntSpecificProperty(final String propName,
    final int defaultValue) {
    if (applicationRegistry.isPropertyNull(propName)) {
      return defaultValue;
    }
    return applicationRegistry.getIntSpecificProperty(propName);
  }

  /**
   * Consulta o valor long de uma propriedade
   *
   * @param propName o nome
   * @param defaultValue o valor default
   * @return a propriedade
   */
  public final long getLongSpecificProperty(final String propName,
    final long defaultValue) {
    if (applicationRegistry.isPropertyNull(propName)) {
      return defaultValue;
    }
    return applicationRegistry.getLongSpecificProperty(propName);
  }

  /**
   * Consulta o valor long de uma propriedade
   *
   * @param propName o nome
   * @param defaultValue o valor default
   * @return a propriedade
   */
  public final int getLongSpecificProperty(final String propName,
    final int defaultValue) {
    if (applicationRegistry.isPropertyNull(propName)) {
      return defaultValue;
    }
    return applicationRegistry.getIntSpecificProperty(propName);
  }

  /**
   * Consulta o valor textual de uma propriedade
   *
   * @param propName o nome
   * @return a propriedade
   */
  public final String getStringSpecificProperty(final String propName) {
    return applicationRegistry.getStringSpecificProperty(propName);
  }

  /**
   * Consulta o valor lista de strings de uma propriedade
   *
   * @param propName o nome
   * @return a lista
   */
  public final List<String> getStringListSpecificProperty(final String propName) {
    return applicationRegistry.getStringListSpecificProperty(propName);
  }

  /**
   * Consulta o valor booleano de uma propriedade
   *
   * @param propName o nome
   * @param defaultValue o valor default
   * @return a propriedade
   */
  public final boolean getBooleanSpecificProperty(final String propName,
    final boolean defaultValue) {
    if (applicationRegistry.isPropertyNull(propName)) {
      return defaultValue;
    }
    return applicationRegistry.getBooleanSpecificProperty(propName);
  }

  /**
   * Consulta o valor double de uma propriedade
   *
   * @param propName o nome
   * @param defaultValue o valor default
   * @return a propriedade
   */
  public final double getDoubleSpecificProperty(final String propName,
    final double defaultValue) {
    if (applicationRegistry.isPropertyNull(propName)) {
      return defaultValue;
    }
    return applicationRegistry.getDoubleSpecificProperty(propName);
  }

  /**
   * Mtodo para enviar atributos para uma aplicao; deve ser definido na
   * aplicao de acordo com a necessidade da mesma.
   *
   * @deprecated Sua implementao deve ser movida para os mtodos
   * {@link #onAsyncMessageReceived(String, Object, String)} e
   * {@link #onSyncMessageReceived(String, Object, String)}.
   *
   * Para enviar uma mensagem diretamente a uma aplicao, utilizar
   * {@link ApplicationManager#sendAsyncMessage(String, String, Object, String)}
   * ou
   * {@link ApplicationManager#sendSyncMessage(String, String, Object, String)}.
   *
   * @param name nome do atributo.
   * @param value seu valor.
   * @param senderId a identificao da aplicao que enviou a mensagem.
   */
  @Deprecated
  public void sendMessage(String name, Object value, String senderId) {
    // Mtodo a ser implementado pelas aplicaes.
  }

  /**
   * Trata uma mensagem assncrona recebida pela aplicao.
   *
   * [ATENO] Este mtodo *no* deve ser chamado explicitamente para enviar
   * mensagem para a aplicao. Somente o {@link ApplicationManager} deve us-lo.
   *
   * A aplicao poder receber uma mensagem a qualquer momento de sua
   * execuo.
   * O remetente da mensagem pode ser outra aplicao, representada pelo
   * parmetro senderId ou o prprio ambiente, nesse caso o
   * parmetro senderId vem nulo. Uma mesma aplicao pode
   * receber diferentes tipos de mensagem, definidos por
   * {@code type}. O valor da mensagem  definido em
   * {@code value}.
   *
   * @param type tipo da mensagem recebida.
   * @param value valor da mensagem recebida.
   * @param senderId o id da instncia da aplicao que enviou a mensagem.
   * (pode ser {@code null} - caso a mensagem tenha vindo do sistema, por
   * exemplo)
   */
  protected void onAsyncMessageReceived(String type, Object value,
    String senderId) {
  }

  /**
   * Trata uma mensagem sncrona recebida pela aplicao.
   *
   * [ATENO] Este mtodo *no* deve ser chamado explicitamente para enviar
   * mensagem para a aplicao. Somente o {@link ApplicationManager} deve us-lo.
   *
   * A aplicao poder receber uma mensagem a qualquer momento de sua
   * execuo.
   * O remetente da mensagem pode ser outra aplicao, representada pelo
   * parmetro senderId ou o prprio ambiente, nesse caso o
   * parmetro senderId vem nulo. Uma mesma aplicao pode
   * receber diferentes tipos de mensagem, definidos por
   * {@code type}. O valor da mensagem  definido em
   * {@code value}.
   *
   * @param type tipo da mensagem recebida.
   * @param value valor da mensagem recebida.
   * @param senderId o id da instncia da aplicao que enviou a mensagem.
   * (pode ser {@code null} - caso a mensagem tenha vindo do sistema, por
   * exemplo)
   * @return o objeto de resposta.
   */
  protected Object onSyncMessageReceived(String type, Object value,
    String senderId) {
    return null;
  }

  /**
   * Mtodo que sinaliza a aplicao que ela j foi iniciada.
   *
   * @throws ApplicationException em caso de falha.
   */
  public void postInitialization() throws ApplicationException {
    // Mtodo a ser implementado pelas aplicaes.
  }

  /**
   * Mtodo para encerramento da execuo, que pode ser redefinido para
   * comportamentos prprios da aplicaes.
   *
   * @throws ApplicationException em caso de erro no fechamento da applicao.
   */
  public abstract void killApplication() throws ApplicationException;

  /**
   * Mtodo abstrato para abertura do dilogo das aplicaes
   *
   * @throws ApplicationException Em caso de erro ao iniciar a aplicao.
   */
  public abstract void startApplication() throws ApplicationException;

  /**
   * Mtodo abstrato para fechamento do dilogo das aplicaes
   */
  public abstract void finishApplication();

  /**
   * Mtodo abstrato para fechamento da aplicao.
   *
   * @return um flag indicativo de sucesso no fechamento.
   */
  public final boolean closeApplication() {
    final boolean canKill = userCanKillApplication();
    if (!canKill) {
      return false;
    }
    try {
      killApplication();
    }
    catch (Exception e) {
      showApplicationError(e);
    }
    finishApplication();

    return true;
  }

  /**
   * Exibio de mensagem de erro interna da aplicao.
   *
   * @param t a exceo.
   */
  private void showApplicationError(Throwable t) {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    final Locale locale = appManager.getLocale();
    final String title = applicationRegistry.getApplicationName(locale);
    JOptionPane.showMessageDialog(null, t.getMessage(), title,
      JOptionPane.ERROR_MESSAGE);
  }

  /**
   * Obtm a classe principal da aplicao.
   *
   * @return a classe principal.
   */
  protected Class<?> getMainApplicationClass() {
    return getClass();
  }

  /**
   * Busca uma imagem para a aplicao. A verso no esttica deste mtodo deve
   * ser preferida, quando uma instncia da aplicao est disponvel:
   * {@link #getImageIcon(String)}.
   *
   * @param baseClass classe base para busca do recurso.
   * @param imagePath caminho dentro do diretrio de imagens da aplicao.
   * @return a imagem ou null
   */
  final static public ImageIcon getImageIcon(final Class<?> baseClass,
    final String imagePath) {
    final String imgPath = "images/" + imagePath;
    try {
      final URL url = getApplicationResourceAsURL(baseClass, imgPath);
      if (url == null) {
        return null;
      }
      return new ImageIcon(url);
    }
    catch (final Exception e) {
      // Comentado para que no saia mensagens de erro
      // quando a aplicao no usar o mecanismo default
      // de imagens. O cdigo abaixo deve ser usado par depurao.
      // System.out.println("Path errado: " + path);
      // System.out.println("Path errado (resource): " + resPath);
      // e.printStackTrace();
      return null;
    }
  }

  /**
   * Busca um stream de um resource da aplicao. A verso no esttica deste
   * mtodo deve ser preferida, quando uma instncia da aplicao est
   * disponvel: {@link #getApplicationResourceAsURL(String)}.
   *
   * @param baseClass classe base para busca do recurso.
   * @param path path relativo dentro do diretrio 'resources'
   * @return o stream.
   */
  final static public URL getApplicationResourceAsURL(final Class<?> baseClass,
    final String path) {
    final String resourcePath = RES_DIR_NAME + "/" + path;
    final URL url = baseClass.getResource(resourcePath);
    return url;
  }

  /**
   * Busca um stream de um resource da aplicao. A verso no esttica deste
   * mtodo deve ser preferida, quando uma instncia da aplicao est
   * disponvel: {@link #getApplicationResourceAsStream(String)}.
   *
   * @param baseClass classe base para busca do recurso.
   * @param path path relativo dentro do diretrio 'resources'
   * @return o stream.
   */
  final static public InputStream getApplicationResourceAsStream(
    final Class<?> baseClass, final String path) {
    final String resourcePath = RES_DIR_NAME + "/" + path;
    final InputStream stream = baseClass.getResourceAsStream(resourcePath);
    return stream;
  }

  /**
   * Busca uma imagem para a aplicao.
   *
   * @param imagePath caminho dentro do diretrio de imagens da aplicao.
   * @return a imagem ou null
   */
  final public ImageIcon getImageIcon(final String imagePath) {
    return getImageIcon(getMainApplicationClass(), imagePath);
  }

  /**
   * Busca um stream de um resource da aplicao.
   *
   * @param path path relativo dentro do diretrio 'resources'
   * @return o stream.
   */
  final public URL getApplicationResourceAsURL(final String path) {
    return getApplicationResourceAsURL(getMainApplicationClass(), path);
  }

  /**
   * Busca um stream de um resource da aplicao.
   *
   * @param path path relativo dentro do diretrio 'resources'
   * @return o stream.
   */
  final public InputStream getApplicationResourceAsStream(final String path) {
    return getApplicationResourceAsStream(getMainApplicationClass(), path);
  }

  /**
   * Mtodo de kill pelo usurio  especfico do desktop applications
   *
   * @return um flag indicativo de possibilidade de fechamento da aplicao.
   */
  protected abstract boolean userCanKillApplication();



}
