package csbase.client.applications.algorithmsmanager.actions;

import java.awt.Window;
import java.awt.event.ActionEvent;

import javax.swing.Action;
import javax.swing.ImageIcon;

import csbase.client.applications.algorithmsmanager.dialogs.AlgorithmVersionInfoPanel;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;

/**
 * Essa classe abstrata representa uma a ao comum a operaes realizadas sobre
 * uma verso de algoritmo selecionada.
 * 
 * Essa ao mantm o algoritmo selecionado para edio, assim como a verso do
 * algoritmo sobre a qual ser realizada uma ao.
 * 
 * O mtodo <code> handleVersionOperation </code> deve ser redefinido para que
 * seja aberto o dilogo referente a cada ao especfica.
 * 
 */
public abstract class CommonVersionAction extends AlgorithmsManagerAction {
  /** Painel com as informaes de verses do algoritmo */
  private AlgorithmVersionInfoPanel versionInfoPanel;

  /** Algoritmo selecionado para realizao de aes sobre suas verses */
  private AlgorithmInfo algoInfo;

  /** Verso do algoritmo selecionada para edio */
  private AlgorithmVersionInfo versionInfo;

  /**
   * Constri a ao de importar um pacote de verso de algoritmo.
   * 
   * @param versionInfoPanel painel pai que criou a ao
   * @param icon imagem da ao
   */
  public CommonVersionAction(AlgorithmVersionInfoPanel versionInfoPanel,
    ImageIcon icon) {
    super(versionInfoPanel.getApplication(), icon);
    this.versionInfoPanel = versionInfoPanel;
    setEnabled(false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleActionPerformed(ActionEvent ae) throws Exception {
    setSelectedAlgorithm();
    setSelectedVersion();
    handleVersionOperation();
  }

  /**
   * Trata a operao realizada sobre uma ou mais verses selecionadas.
   */
  protected abstract void handleVersionOperation();

  /**
   * Obtm o algoritmo selecionado para edio das verses.
   * 
   * @return o algoritmo selecionado
   */
  public AlgorithmInfo getSelectedAlgorithm() {
    return algoInfo;
  }

  /**
   * Atribui um algoritmo selecionado para realizao de aes sobre suas
   * verses.
   * 
   */
  private void setSelectedAlgorithm() {
    algoInfo = null;
    AlgorithmListItem algoItem = versionInfoPanel.getSelectedAlgorithmItem();
    if (algoItem != null) {
      algoInfo = algoItem.getItem();
    }
  }

  /**
   * Obtm a verso selecionada na rvore de verses.
   * 
   * @return a verso selecionada
   */
  public AlgorithmVersionInfo getSelectedVersion() {
    return versionInfo;
  }

  /**
   * Atribui uma verso de algoritmo selecionada para a ao ser realizada.
   * 
   */
  private void setSelectedVersion() {
    this.versionInfo = versionInfoPanel.getSelectedVersion();
  }

  /**
   * Obtm a janela da aplicao.
   * 
   * @return a janela da aplicao
   */
  public Window getWindow() {
    return getApplication().getApplicationFrame();
  }

  /**
   * Obtm o nome da ao.
   * 
   * @return nome da ao.
   */
  public String getName() {
    return (String) getValue(Action.NAME);
  }
}
