package csbase.client.applications.filetransfer;

import java.util.Arrays;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;

import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.remote.ProjectAdminObserver;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommonClientProject;
import csbase.logic.CommonProjectInfo;
import csbase.logic.ProjectUserEvent;
import csbase.logic.User;
import csbase.logic.UserProjectInfo;

/**
 * Classe que representa uma combo para seleo de projetos do usurio corrente.
 * Se cadastra como observadora de {@link ProjectAdminObserver} para se
 * atualizar caso algum projeto seja removido ou criado.
 * 
 * @author Tecgraf
 */
public final class ProjectComboBox extends JComboBox implements Observer {

  /** Identificador do usurio corrente */
  private Object userId;

  /**
   * Construtor.
   * 
   * @param application aplicao.
   */
  public ProjectComboBox(Application application) {
    userId = User.getLoggedUser().getId();
    /* Carregando os projetos do usurio */
    loadProjects(application);
    setSelectedIndex(-1);
    /* Se cadastrando como observador de projetos em ProjectAdminObserver */
    ProjectAdminObserver.getInstance().addObserver(this);
  }

  /** Remove a combo como observadora de ProjectAdminObserver. */
  public void deleteObserver() {
    ProjectAdminObserver.getInstance().deleteObserver(this);
  }

  /**
   * Chamado por notityObservers de ProjectAdminObserver quando um projeto 
   * criado ou removido no servidor. arg ProjectUserEvent
   * 
   * @param o observado.
   * @param arg argumento.
   */
  @Override
  public void update(Observable o, Object arg) {
    if (!(arg instanceof ProjectUserEvent)) {
      return;
    }
    ProjectUserEvent event = (ProjectUserEvent) arg;
    final UserProjectInfo info =
      new UserProjectInfo((CommonProjectInfo) event.item);
    switch (event.type) {
      case ProjectUserEvent.CREATE:
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            addItem(info);
          }
        });
        break;
      case ProjectUserEvent.DELETE:
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            if ((getSelectedItem()).equals(info)) {
              setSelectedIndex(-1);
            }
            removeItem(info);
          }
        });
        break;
    }
  }

  /**
   * Atualiza a projectCombo com os projetos existentes para o usurio. Utiliza
   * tarefa remota para obteno dos projetos.
   * 
   * @param application aplicao.
   */
  private void loadProjects(Application application) {
    try {
      List<UserProjectInfo> projects = null;
      // Cria uma tarefa remota para obteno dos projetos
      Task<List<UserProjectInfo>> task =
        new RemoteTask<List<UserProjectInfo>>() {
          @Override
          public void performTask() throws Exception {

            List<UserProjectInfo> projects =
              CommonClientProject.getAllProjectsInfo(userId);

            setResult(projects);
          }
        };
      if (task.execute(application.getApplicationFrame(),
        application.getName(), application.getString("searching.projects.msg"))) {
        projects = task.getResult();
      }
      else {
        throw task.getError();
      }
      final UserProjectInfo[] projsArray =
        projects.toArray(new UserProjectInfo[0]);
      Arrays.sort(projsArray);
      setModel(new DefaultComboBoxModel(projsArray));
    }
    catch (Exception e) {
      ApplicationFrame frame = application.getApplicationFrame();
      StandardErrorDialogs.showErrorDialog(frame, frame.getTitle(), e);
    }
  }
}