package csbase.client.project.action;

import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.security.AccessControlException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JFileChooser;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.externalresources.ExternalResources;
import csbase.client.externalresources.LocalFile;
import csbase.client.externalresources.StandaloneLocalFile;
import csbase.client.project.ImportTask;
import csbase.client.project.ProjectFileContainer;
import csbase.client.remote.srvproxies.NotificationProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseException;
import csbase.logic.ClientProjectFile;
import csbase.logic.User;

/**
 * Ao para importao de arquivos, a partir do sistema de arquivos do cliente
 * para o projeto no servidor.
 */
public class CommonFileImportAction extends CommonProjectAction {

  /** Mensagem para erro de sistema. No  possvel continuar. */
  private static final String CRITICAL_ERROR =
    "PRJ_PROJECT_FILE_IMPORT_CRITICAL_ERROR";

  /**
   * Mensagem de erro para caracteres invlidos nos nomes de
   * arquivos/diretrios.
   */
  private static final String NAME_CHARACTER_ERROR =
    "UTIL_NAME_CHARACTER_ERROR";

  /** Mensagem de aviso para arquivo j existente */
  private static final String FILE_EXISTS =
    "PRJ_PROJECT_FILE_IMPORT_FILE_EXISTS";

  /** Mensagem de aviso para arquivo no encontrado. */
  private static final String FILE_NOT_FOUND =
    "PRJ_PROJECT_FILE_IMPORT_FILE_NOT_FOUND";

  /** Mensagem em caso de erro de I/O */
  private static final String IO_ERROR = "PRJ_PROJECT_FILE_IMPORT_IO_ERROR";

  /** Mensagem de notificao do incio da importao */
  private static final String IMPORT_IN_PROGRESS_MSG =
    "PRJ_PROJECT_FILE_IMPORT_IN_PROGRESS";

  /** Mensagem para importao efetuada com sucesso */
  private static final String SUCCESS_ALL =
    "PRJ_PROJECT_FILE_IMPORT_SUCCESS_ALL";
  /**
   * Importao concluda sem erro (exceo), mas alguns arquivos no foram
   * importados.
   */
  private static final String SUCCESS_PARTIAL =
    "PRJ_PROJECT_FILE_IMPORT_SUCCESS_PARTIAL";
  /**
   * Importao concluda sem erro (exceo), mas nenhum arquivo foi importado.
   */
  private static final String SUCCESS_NONE =
    "PRJ_PROJECT_FILE_IMPORT_SUCCESS_NONE";

  /** Ttulo do dilogo de importao */
  private static final String TITLE = "PRJ_PROJECT_FILE_IMPORT_TITLE";

  /** Mensagem de aviso de importao restrita */
  private static final String RESTRICTED_IMPORT_WARNING =
    "PRJ_PROJECT_FILE_IMPORT_RESTRICTED_WARNING";

  /** Diretrio de destino para os arquivos importados */
  private ClientProjectFile targetDir;

  /**
   * Indica se o diretrio de destino  dinmico (obtido da seleo atual do
   * <code>container</code>) ou fixo (passado em um construtor).
   */
  private boolean isDynamicTarget;

  /**
   * Diretrio fixo para importao de arquivos - passado no construtor, no 
   * oriundo de seleo na rvore de projetos.
   */
  private ClientProjectFile fixedTargetDir;

  /**
   * Timestamp de incio da criao de arquivos no projeto, operao que consome
   * um razovel tempo computacional e precede a transferncia.
   */
  public long fileCreationStart;

  /**
   * Modo de seleo do chooser (importao)
   */
  private int chooserSelectionMode;

  /**
   * Cria a ao de importao. O diretrio de destino  dinmico, corresponde 
   * seleo no <code>container</code> no momento em que a ao  disparada.
   * 
   * @param container objeto contendo arquivos de projeto.
   */
  public CommonFileImportAction(ProjectFileContainer container) {
    super(container);
    this.isDynamicTarget = true;
    chooserSelectionMode = JFileChooser.FILES_AND_DIRECTORIES;
  }

  /**
   * Cria a ao de importao. O diretrio de destino  fixo, recebido neste
   * construtor.
   * 
   * @param container objeto contendo arquivos de projeto.
   * @param targetDir diretrio de destino para a importao.
   */
  public CommonFileImportAction(ProjectFileContainer container,
    ClientProjectFile targetDir) {
    super(container);
    this.isDynamicTarget = false;
    this.fixedTargetDir = targetDir;
    chooserSelectionMode = JFileChooser.FILES_AND_DIRECTORIES;
  }

  /**
   * Ajuste do modo de seleo do filechooser de navegao de arquivos locais
   * para permitir diretrios, arquivos ou ambos.
   * 
   * @param chooserSelectionMode o modo
   */
  final public void setFileChooserSelectionMode(final int chooserSelectionMode) {
    this.chooserSelectionMode = chooserSelectionMode;
  }

  /**
   * Consulta do modo de seleo do filechooser.
   * 
   * @see #setFileChooserSelectionMode(int)
   * 
   * @return o modo.
   */
  final public int getFileChooserSelectionMode() {
    return this.chooserSelectionMode;
  }

  /**
   * Mtodo que dispara a ao. Dispara uma <code>ImportTask</code> encarregada
   * de fazer a transferncia dos arquivos selecionados no sistema de arquivos
   * do cliente para o projeto no servidor.
   * 
   * @param e informaes sobre o evento ocorrido.
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    this.targetDir =
      (isDynamicTarget) ? (ClientProjectFile) getSelectedFile()
        : this.fixedTargetDir;
    try {
      LocalFile[] sourceFiles = null;
      try {
        // Acesso ao sistema de arquivos permitido: obtm arquivos com o
        // JFileChooser
        JFileChooser chooser = getFileChooser();
        chooser.setMultiSelectionEnabled(true);
        chooser.setFileSelectionMode(chooserSelectionMode);
        int returnVal = chooser.showOpenDialog(getWindow());
        if (returnVal == JFileChooser.CANCEL_OPTION) {
          return;
        }
        File[] selectedFiles = chooser.getSelectedFiles();
        if (selectedFiles == null || selectedFiles.length == 0) {
          return;
        }
        sourceFiles = new LocalFile[selectedFiles.length];
        for (int i = 0; i < selectedFiles.length; i++) {
          sourceFiles[i] = new StandaloneLocalFile(selectedFiles[i]);
        }
      }
      catch (AccessControlException ex1) {
        // Acesso ao sistema de arquivos negado (rodando em "sandbox"): obtm
        // arquivos com a API JNLP. No  possvel a importao de diretrios.
        if (!ExternalResources.getInstance().isEnabled()) {
          showError(LNG.get(CRITICAL_ERROR));
        }
        try {
          showWarning(LNG.get(RESTRICTED_IMPORT_WARNING));
          sourceFiles =
            ExternalResources.getInstance().openMultiFileDialog(".", null);
          if (sourceFiles == null || sourceFiles.length == 0) {
            return;
          }
        }
        catch (CSBaseException ex2) {
          showError(LNG.get(CRITICAL_ERROR), ex2);
          return;
        }
      }

      fileCreationStart = System.currentTimeMillis();
      LocalFile[] validFiles = validateFiles(sourceFiles);
      if (null == validFiles || 0 == validFiles.length) {
        return;
      }
      ImportTask task =
        new ImportTask(getProject(), getWindow(), validFiles, targetDir);
      String msg = LNG.get(IMPORT_IN_PROGRESS_MSG);
      if (task.execute(getWindow(), LNG.get(TITLE), msg, false, true)) {
        double totalSize = getTotalSizeInKBytes(task.getTotalSize());
        double totalTime = getElapsedTime(fileCreationStart);
        double transferTime = getElapsedTime(task.getTransferStart());
        double transferRate = getTransferRate(totalSize, transferTime);
        int errors = task.getResult();
        if (errors == 0) {
          msg = LNG.get(SUCCESS_ALL);
        }
        else if (errors == -1) {
          /*
           * todas as importaes falharam
           */
          msg = LNG.get(SUCCESS_NONE);
        }
        else {
          /*
           * algumas importaes falharam
           */
          msg = String.format(LNG.get(SUCCESS_PARTIAL), errors);
        }
        String stats =
          String.format(LNG.get("PRJ_PROJECT_FILE_IMPORT_STATS"), totalSize,
            totalTime, transferRate);
        Object[] ids = new Object[] { User.getLoggedUser().getId() };
        NotificationProxy.notifyTo(ids, msg + ' ' + stats, false, false);
        if (errors != 0) {
          showWarning(msg);
        }
      }
    }
    catch (IOException ex) {
      showError(LNG.get(IO_ERROR), ex);
    }
  }

  /**
   * Obtm o volume total da transferncia, em Kilobytes.
   * 
   * @param sizeInBytes volume total da transferncia, em bytes.
   * 
   * @return volume total dos arquivos transferidos (em KBytes).
   */
  private double getTotalSizeInKBytes(long sizeInBytes) {
    return sizeInBytes / 1024;
  }

  /**
   * Obtm o tempo decorrido, em segundos, desde o incio da execuo.
   * 
   * @param start tempo de incio da execuo (milissegundos desde 1970).
   * 
   * @return tempo decorrido desde o incio da execuo (em segundos).
   */
  private double getElapsedTime(long start) {
    return (double) (System.currentTimeMillis() - start) / 1000;
  }

  /**
   * Obtm a taxa de transferncia, em Kbytes por segundo.
   * 
   * @param totalSize volume tranferido (em KBytes).
   * @param elapsedTime tempo decorrido (em segundos).
   * 
   * @return taxa de transferncia (em Kbytes/segundo).
   */
  private double getTransferRate(double totalSize, double elapsedTime) {
    return totalSize / elapsedTime;
  }

  /**
   * Verifica se os arquivos de origem so vlidos, isto , se possuem nomes
   * aceitos pelas regras de nomenclatura dos arquivos de projeto e se j no
   * existem arquivos de mesmo nome no projeto.
   * 
   * @param sourceFiles arquivos de origem.
   * 
   * @return uma array com os arquivos vlidos.
   * 
   * @throws IOException em caso de erro de I/O.
   */
  private LocalFile[] validateFiles(LocalFile[] sourceFiles) throws IOException {

    List<LocalFile> validFiles = new ArrayList<LocalFile>();

    int lastReplaceOption = -1;
    final int yesOption, yesToAllOption, noToAllOption;
    String[] replaceOptions;
    if (1 == sourceFiles.length) {
      yesOption = yesToAllOption = 0;
      noToAllOption = 1;
      replaceOptions = new String[] { LNG.get("UTIL_YES"), LNG.get("UTIL_NO") };
    }
    else {
      yesOption = 0;
      yesToAllOption = 1;
      noToAllOption = 3;
      replaceOptions =
        new String[] { LNG.get("UTIL_YES"), LNG.get("UTIL_YES_TO_ALL"),
            LNG.get("UTIL_NO"), LNG.get("UTIL_NO_TO_ALL") };
    }

    for (LocalFile file : sourceFiles) {
      String fileName = file.getName();

      if (!ClientUtilities.isValidFileName(fileName)) {
        showError(LNG.get(NAME_CHARACTER_ERROR));
        return new LocalFile[0];
      }
      if (!file.exists()) {
        showError(MessageFormat.format(LNG.get(FILE_NOT_FOUND), file.getName()));
        return new LocalFile[0];
      }
      if (targetDir.getChild(fileName) != null) {

        if (lastReplaceOption == noToAllOption) {
          continue;
        }
        if (lastReplaceOption == yesToAllOption) {
          validFiles.add(file);
          continue;
        }
        // Sim / No
        String question =
          MessageFormat.format(LNG.get(FILE_EXISTS), new Object[] { fileName,
              targetDir.getName() });

        lastReplaceOption =
          StandardDialogs.showOptionDialog(getWindow(), LNG.get(TITLE),
            question, replaceOptions);

        if (yesOption == lastReplaceOption
          || yesToAllOption == lastReplaceOption) {

          validFiles.add(file);
        }
      }
      else {
        validFiles.add(file);
      }
    }

    return validFiles.toArray(new LocalFile[validFiles.size()]);
  }

  /**
   * Mtodo de aviso de erros de sistema.
   * 
   * @param msg mensagem a ser exibida.
   */
  @Override
  protected void showError(String msg) {
    showError(LNG.get(TITLE), msg);
  }

  /**
   * Mtodo para exibir avisos ao usurio.
   * 
   * @param msg mensagem a ser exibida.
   */
  private void showWarning(String msg) {
    StandardDialogs.showWarningDialog(getWindow(), LNG.get(TITLE), msg);
  }

  /**
   * Mtodo de aviso de erros de sistema.
   * 
   * @param msg mensagem a ser exibida.
   * @param ex exceo detectada.
   */
  private void showError(String msg, Exception ex) {
    StandardErrorDialogs.showErrorDialog(getWindow(), LNG.get(TITLE), msg, ex);
  }

  /**
   * Obtm o nome da ao.
   * 
   * @return nome da ao.
   */
  @Override
  public String getName() {
    return LNG.get("PRJ_PROJECT_FILE_IMPORT");
  }
}
