/*
 * $Id$
 */
package csbase.client.algorithms.commands.cache;

import java.awt.Window;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.algorithms.commands.cache.events.AbstractCacheUpdatedEventListener;
import csbase.client.algorithms.commands.cache.events.AbstractCommandUpdatedEventListener;
import csbase.client.algorithms.commands.cache.events.CacheUpdatedEvent;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent.Type;
import csbase.client.algorithms.commands.cache.events.FinishedOrSystemFailureCommandsLoadingEvent;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.client.util.CodeBlockLog;
import csbase.client.util.event.EventListener;
import csbase.client.util.event.EventManager;
import csbase.client.util.event.IEvent;
import csbase.logic.CommandInfo;
import csbase.logic.CommandNotification;
import csbase.logic.CommandStatus;
import csbase.logic.CommonClientProject;
import csbase.logic.Priority;
import csbase.logic.algorithms.commands.CommandPersistenceNotification;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.CommandPersistenceServiceInterface.CommandInfosRetrived;
import csbase.remote.SchedulerServiceInterface;
import csbase.util.TaskScheduler;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;

/**
 * Representa uma cache contendo todos os comandos a que o usurio tenha acesso.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class CommandsCache {

  /**
   * Logger da classe.
   */
  private static Logger LOGGER = Logger
    .getLogger(CommandsCache.class.getName());

  /**
   * Valor padro do {@link CommandsCache#reloadInterval}.
   */
  private static final long DEFAULT_RELOAD_INTERVAL_S = 30;

  /**
   * Valor desejado para se obter resposta do CommandPersistenceService para a
   * carga de comandos terminados - Ver
   * CommandPersistenceService#requestCommandInfos(Object, long) .
   */
  private static final long LOAD_FINISHEDS_DESIRED_RESPONSE_TIME_MILLIS = 3000;

  /**
   * Instncia nica da {@link CommandsCache}.
   */
  private static CommandsCache instance = new CommandsCache();

  /**
   * Lock utilizado na escrita e leitura da {@link CommandsCache#commandsById}.
   */
  private final ReentrantReadWriteLock commandsLock;

  /**
   * Estrutura de dados que armazena os comandos dado seus identificadores.
   */
  private final Map<String, CommandInfo> commandsById;

  /**
   * Comandos agendados.
   */
  private HashSet<CommandInfo> scheduledCommands;

  /**
   * Comandos terminados.
   */
  private HashSet<CommandInfo> finishedOrSystemFailureCommands;

  /**
   * Lock utilizado na carga dos comandos terminados ou com falha de leitura
   * pela persistncia.
   */
  private Lock finishedOrSystemFailureCommandsLock;

  /**
   * Mantm a conta de quantos ouvintes de comandos terminados existem.
   */
  private AtomicInteger finishedOrSystemFailureCommandListenersCounter;

  /**
   * Identificador do projeto que teve carregado seus comandos terminados.
   */
  private Object projectId;

  /**
   * Lock utilizado na insero e remoo de {@link EventListener} do
   * {@link CommandsCache#eventManager}.
   */
  private final Lock eventManagerLock;
  /** Gerente de eventos. */
  private final EventManager eventManager;

  /**
   * Perodo de cada recarga em {@link CommandsCache#reloadIntervalUnit}.<br>
   * Tambm  usado para verificar se os dados expiraram desde a ltima
   * atualizao, caso o sistema de recarga automtica no esteja ligado por
   * falta de observadores na cache.<br>
   * Este valor ser obtido atravs do mtodo
   * {@link CommandsCache#getReloadInterval()}.
   */
  private final long reloadInterval;

  /**
   * Unidade de tempo de {@link CommandsCache#reloadInterval}.
   */
  private final TimeUnit reloadIntervalUnit;

  /**
   * Data da ltima atualizao em milissegundos.
   */
  private long lastUpdateInMilliseconds;

  /**
   * Flag que indica se uma exceo foi lanada durante a ltima atualizao.
   */
  private AtomicBoolean exceptionThrownDuringLastUpdate;

  /**
   * Objeto que gerencia a task de recarga de dados.
   */
  private TaskScheduler reloader;

  /**
   * Construtor.
   */
  private CommandsCache() {
    commandsLock = new ReentrantReadWriteLock();
    commandsById = new HashMap<String, CommandInfo>();
    scheduledCommands = new HashSet<CommandInfo>();
    finishedOrSystemFailureCommands = new HashSet<CommandInfo>();
    finishedOrSystemFailureCommandsLock = new ReentrantLock();
    finishedOrSystemFailureCommandListenersCounter = new AtomicInteger(0);
    eventManagerLock = new ReentrantLock();
    eventManager = new EventManager();
    reloadInterval = getReloadInterval();
    reloadIntervalUnit = TimeUnit.SECONDS;
    lastUpdateInMilliseconds = 0;
    exceptionThrownDuringLastUpdate = new AtomicBoolean(false);

    // Descobre o identificador do projeto corrente.
    CommonClientProject project = DesktopFrame.getInstance().getProject();
    if (null != project) {
      projectId = project.getId();
    }

    reloader = createReloader();

    initializeListeners();
  }

  /**
   * Obtm a instncia nica da cache.
   * 
   * @return a instncia nica da cache ou <tt>null</tt> caso tenha ocorrido um
   *         erro durante a sua criao.
   */
  public static CommandsCache getInstance() {
    return instance;
  }

  /**
   * Retorna true se existir um comando com o dado identificador, false caso
   * contrrio.
   * 
   * @param commandId - identificador de um comando.
   * @return true se existir um comando com o dado identificador, false caso
   *         contrrio.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public boolean hasCommand(String commandId) throws RemoteException {
    return getCommandFromCache(commandId) != null;
  }

  /**
   * Obtm todos os comandos na cache.
   * 
   * @return todos os comandos.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public Collection<CommandInfo> getCommands() throws RemoteException {
    runManualReload(true);
    return getSnapshot();
  }

  /**
   * Obtm uma lista contendo todos os comandos que sejam aceitos pelo filtro.
   * 
   * @param filter filtro utilizado para indicar que tipo de comando se est
   *        interessado.
   * 
   * @return uma lista de comandos que foram aceitos pelo filtro.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public final Collection<CommandInfo> getCommands(CommandsFilter filter)
    throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "obtendo comandos nos estados %s",
        Arrays
          .toString(filter.getAllowedStatus().toArray(new CommandStatus[0])));

    LOGGER.log(methodLog.begin());

    boolean loadFinishedCommands =
      filter.getAllowedStatus().contains(CommandStatus.FINISHED);
    runManualReload(loadFinishedCommands);

    CodeBlockLog filterLog =
      new CodeBlockLog(Level.FINER, "filtragem de comando");

    LOGGER.log(filterLog.begin());

    commandsLock.readLock().lock();
    try {

      Collection<CommandInfo> filtereds = new ArrayList<CommandInfo>();
      for (CommandInfo command : commandsById.values()) {
        if (filter.accept(command)) {
          filtereds.add(command);
        }
      }

      LOGGER.log(filterLog.finished());

      return filtereds;
    }
    finally {
      commandsLock.readLock().unlock();

      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Obtm um comando qualquer que seja aceito pelo filtro. <br>
   * Este mtodo costuma ser utilizado quando s existe um comando que passe
   * pelo filtro.
   * 
   * @param filter filtro utilizado para indicar que tipo de comando se est
   *        interessado.
   * 
   * @return um comando qualquer que seja aceito pelo filtro ou <tt>null</tt>
   *         caso nenhum comando passe no teste do filtro.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   * @throws RemoteException em caso de falha de chamada remota.
   */
  public final CommandInfo getCommand(CommandsFilter filter)
    throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "obtendo 1 comando em um dos estados %s",
        Arrays
          .toString(filter.getAllowedStatus().toArray(new CommandStatus[0])));
    LOGGER.log(methodLog.begin());

    boolean loadFinishedOrSystemFailureCommands =
      filter.getAllowedStatus().contains(CommandStatus.FINISHED)
        || filter.getAllowedStatus().contains(CommandStatus.SYSTEM_FAILURE);
    runManualReload(loadFinishedOrSystemFailureCommands);

    CodeBlockLog filterLog =
      new CodeBlockLog(Level.FINER, "filtragem de comando");
    LOGGER.log(filterLog.begin());

    commandsLock.readLock().lock();
    try {
      for (CommandInfo command : commandsById.values()) {
        if (filter.accept(command)) {
          return command;
        }
      }
      return null;
    }
    finally {
      commandsLock.readLock().unlock();

      LOGGER.log(filterLog.finished());
      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Obtm um comando dado seu projeto e identificador.
   * 
   * @param prjId identificador nico do projeto dono do comando.
   * @param cmdId identificador nico do comando.
   * 
   * @return o comando representado pelo identificador passado.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public final CommandInfo getCommand(final Object prjId, final String cmdId)
    throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "obtendo comando " + cmdId + " do projeto "
        + prjId);
    LOGGER.log(methodLog.begin());

    commandsLock.readLock().lock();

    CommandInfo command;

    try {
      // Tenta recuperar o comando da cache.
      command = getCommandFromCache(cmdId);
      // Caso no encontre, procura o comando no servidor.
      if (null == command) {

        CodeBlockLog searchServerLog =
          new CodeBlockLog(Level.FINER, "busca do comando " + cmdId
            + " do projeto " + prjId + " no servidor");
        LOGGER.log(searchServerLog.begin());

        command =
          ClientRemoteLocator.commandPersistenceService.getCommandInfo(prjId,
            cmdId);

        LOGGER.log(searchServerLog.finished());

        if (null != command) {
          // Salva o resultado na cache.
          saveInCache(command);

          CodeBlockLog fireEventsLog =
            new CodeBlockLog(Level.FINER, "disparo de eventos");
          LOGGER.log(fireEventsLog.begin());

          Type type =
            CommandUpdatedEvent.Type.valueOf(command.getFinalizationType());
          eventManager.fireEvent(new CommandUpdatedEvent(command, type));
          eventManager.fireEvent(new CacheUpdatedEvent(getSnapshot()));

          LOGGER.log(fireEventsLog.finished());
        }
      }
    }
    finally {
      commandsLock.readLock().unlock();
    }

    LOGGER.log(methodLog.finished());
    return command;
  }

  /**
   * Remove comandos do servidor.<br>
   * Este mtodo s funciona para comandos terminados.
   * 
   * @param commands os comandos a serem removidos.
   * 
   * @return um array de booleanos indicando que comandos foram removidos com
   *         sucesso.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public boolean[] removeCommands(List<CommandInfo> commands)
    throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "remoo de comandos");
    LOGGER.log(methodLog.begin());

    // Flag indicando que devemos atualizar a cache e disparar eventos.
    boolean updateCache = false;
    /*
     * Cria um array para armazenar o resultado da remoo de cada comando.
     */
    boolean[] result = new boolean[commands.size()];
    Arrays.fill(result, false);

    /*
     * Faz 3 listas. Uma com o identificador dos comandos que tem permisso para
     * serem removidos, uma com o identificador do projeto em que estes comandos
     * foram salvos e uma para relacionar os ndices dessa lista com os ndices
     * da lista de comandos<br> Os comandos que no tiverem permisso para serem
     * removidos, tero atribuidos o valor {@code false} como resultado.<br> A
     * permisso  dada somente a comandos que terminaram.
     */
    List<String> removeAlloweds = new ArrayList<String>();
    List<Object> projectIds = new ArrayList<Object>();
    List<Integer> allowedsInx2commandsInx = new ArrayList<Integer>();
    /*
     * Verifica os comandos permitidos.
     */
    for (int cmdInx = 0; cmdInx < commands.size(); cmdInx++) {
      CommandInfo command = commands.get(cmdInx);
      if (command.getStatus() == CommandStatus.FINISHED
        || command.getStatus() == CommandStatus.SYSTEM_FAILURE) {
        removeAlloweds.add(command.getId());
        projectIds.add(command.getProjectId());
        allowedsInx2commandsInx.add(cmdInx);
      }
    }
    // Remove os comandos com permisso para serem removidos.
    boolean[] allowedsResult =
      ClientRemoteLocator.commandPersistenceService.removeCommandInfos(
        projectIds, removeAlloweds);
    /*
     * Joga o resultado dos comandos permitidos para o array de resultado de
     * todos os comandos.
     */
    for (int inx = 0; inx < removeAlloweds.size(); inx++) {
      int cmdInx = allowedsInx2commandsInx.get(inx);
      if (-1 < cmdInx) {
        result[cmdInx] = allowedsResult[inx];
        updateCache |= allowedsResult[inx];
      }
    }

    if (updateCache) {
      boolean cacheUpdated = false;
      /*
       * Remove da cache os comandos que tiveram sucesso a serem removidos do
       * servidor.
       */
      Collection<CommandInfo> cacheSnapshot;
      commandsLock.writeLock().lock();
      try {
        for (int inx = 0; inx < commands.size(); inx++) {
          if (result[inx]) {
            /*
             * Verifica se ele j havia sido removido da cache para evitar
             * mandar eventos de atualizao a toa. O comando poderia j ter
             * sido removido da cache atravs dos eventos emitidos pelo servio
             * de persistncia.
             */
            cacheUpdated |= null != removeFromCache(commands.get(inx).getId());
          }
        }
        /*
         * Cria uma imagen instantnea da cache para posteriormente disparar o
         * evento de que houve mudanas na cache.
         */
        cacheSnapshot = getSnapshot();
      }
      finally {
        commandsLock.writeLock().unlock();
      }

      for (int inx = 0; inx < commands.size(); inx++) {
        if (result[inx]) {
          eventManager.fireEvent(new CommandUpdatedEvent(commands.get(inx),
            Type.removed));
        }
      }
      if (cacheUpdated) {
        eventManager.fireEvent(new CacheUpdatedEvent(cacheSnapshot));
      }
    }

    LOGGER.log(methodLog.finished());

    return result;
  }

  /**
   * Atualiza a prioridade de um comando no servidor.<br>
   * Somente comandos no estado {@link CommandStatus#SCHEDULED} podem ter sua
   * prioridade atualizada.
   * 
   * @param command o comando a ser atualizado.
   * 
   * @return <tt>true</tt> se o comando estava no estado
   *         {@link CommandStatus#SCHEDULED} e a prioridade foi alterada com
   *         sucesso.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public boolean updatePriority(CommandInfo command) throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "atualizao de prioridade do comando "
        + command.getId());
    LOGGER.log(methodLog.begin());

    try {
      final CommandStatus status = command.getStatus();
      if (status == CommandStatus.SCHEDULED) {
        final String cmdId = command.getId();
        final Priority priority = command.getPriority();
        final SchedulerServiceInterface schedulerService =
          ClientRemoteLocator.schedulerService;
        boolean set = schedulerService.setPriority(cmdId, priority);
        if (set) {
          eventManager
            .fireEvent(new CommandUpdatedEvent(command, Type.updated));
          final Collection<CommandInfo> snapshot = getSnapshot();
          eventManager.fireEvent(new CacheUpdatedEvent(snapshot));
          return true;
        }
      }
      return false;
    }
    finally {
      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Atualiza a posio de um comando no servidor.<br>
   * Somente comandos no estado {@link CommandStatus#SCHEDULED} podem ter sua
   * posio atualizada.
   * 
   * @param command o comando a ser atualizado.
   * 
   * @return <tt>true</tt> se o comando estava no estado
   *         {@link CommandStatus#SCHEDULED} e a posio foi alterada com
   *         sucesso.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public boolean updatePosition(CommandInfo command) throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "atualizao de posio do comando "
        + command.getId());
    LOGGER.log(methodLog.begin());

    try {
      if (command.getStatus() == CommandStatus.SCHEDULED) {
        final String cmdId = command.getId();
        final int position = command.getGlobalPosition();
        boolean set =
          ClientRemoteLocator.schedulerService.setPosition(cmdId, position);
        if (set) {
          /*
           * Como a alterao de posio de um comando influencia em outros, no
           *  suficiente enviar um evento indicando que ele foi alterado.
           * Deve-se recarregar os dados e assim atualizar a posio de todos.
           */
          reload();
          return true;
        }
      }
      return false;
    }
    finally {
      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Atualiza a descrio de um comando no servidor.
   * 
   * @param command o comando a ser atualizado.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public void updateDescription(CommandInfo command) throws RemoteException {
    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "atualizao de descrio do comando "
        + command.getId());
    LOGGER.log(methodLog.begin());

    try {
      switch (command.getStatus()) {
        case SCHEDULED: {
          ClientRemoteLocator.schedulerService.updateCommandDescription(
            command.getId(), command.getDescription());
          break;
        }
        case FINISHED: {
          ClientRemoteLocator.commandPersistenceService
            .updateCommandDescription(command.getProjectId(), command.getId(),
              command.getDescription());
          break;
        }
        case SYSTEM_FAILURE: {
          /**
           * No  permitido atualizar a descrio de um comando com falha no
           * sistema. Isos por que se a falha for a no existncia do arquivo de
           * propriedades ou a falta de espao em disco, a tentativa de
           * atualizao geraria uma exceo.
           */
          break;
        }
        default: {
          ClientRemoteLocator.sgaService.updateCommandDescription(
            command.getId(), command.getDescription());
        }
      }
      eventManager.fireEvent(new CommandUpdatedEvent(command, Type.updated));
      eventManager.fireEvent(new CacheUpdatedEvent(getSnapshot()));
    }
    finally {
      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Adiciona um observador para eventos de atualizao da cache toda. <br>
   * Liga a recarga automtica caso esta j no esteja ligado.
   * 
   * @param listener observador do evento.
   */
  public void addEventListener(AbstractCacheUpdatedEventListener listener) {
    addEventListener(listener, CacheUpdatedEvent.class);
    final CommandsFilter filter = listener.getFilter();
    final Collection<CommandStatus> allowedStatus = filter.getAllowedStatus();
    if (allowedStatus.contains(CommandStatus.FINISHED)
      || allowedStatus.contains(CommandStatus.SYSTEM_FAILURE)) {
      finishedOrSystemFailureCommandListenersCounter.incrementAndGet();
    }
  }

  /**
   * Adiciona um observador para eventos de atualizao de comandos. <br>
   * Liga a recarga automtica caso esta j no esteja ligado.
   * 
   * @param listener observador do evento.
   */
  public void addEventListener(AbstractCommandUpdatedEventListener listener) {
    addEventListener(listener, CommandUpdatedEvent.class);
  }

  /**
   * Adicona listener.
   * 
   * @param listener listener
   */
  public void addEventListener(
    EventListener<FinishedOrSystemFailureCommandsLoadingEvent> listener) {
    addEventListener(listener,
      FinishedOrSystemFailureCommandsLoadingEvent.class);
  }

  /**
   * Remove um observador de eventos. <br>
   * Desliga a recarga automtica caso no haja mais nenhum observador.
   * 
   * @param <E> tipo do evento a sendo observado.
   * @param <L> tipo do observador do evento.
   * @param listener observador do evento.
   */
  public <E extends IEvent, L extends EventListener<E>> void removeEventListener(
    L listener) {
    eventManagerLock.lock();
    try {
      eventManager.removeEventListener(listener);
      if (listener instanceof AbstractCacheUpdatedEventListener) {
        final AbstractCacheUpdatedEventListener cnvListener =
          (AbstractCacheUpdatedEventListener) listener;
        final CommandsFilter filter = cnvListener.getFilter();
        final Collection<CommandStatus> allowedStatus =
          filter.getAllowedStatus();
        if (allowedStatus.contains(CommandStatus.FINISHED)
          || allowedStatus.contains(CommandStatus.SYSTEM_FAILURE)) {
          finishedOrSystemFailureCommandListenersCounter.decrementAndGet();
        }
      }
      if (0 == eventManager.countListeners()) {
        reloader.stop();
      }
    }
    finally {
      eventManagerLock.unlock();
    }
  }

  /**
   * Salva - grava/atualiza - um comando na cache.
   * 
   * @param cmd comando a ser salvo.
   * 
   * @return <tt>false</tt> caso no tenha sido inserido na cahce por estar no
   *         estado {@link CommandStatus#FINISHED} e no ser do projeto
   *         corrente.
   */
  private boolean saveInCache(CommandInfo cmd) {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINER, "persistindo comando " + cmd.getId()
        + " no estado " + cmd.getStatus() + " na cache");
    LOGGER.log(methodLog.begin());

    commandsLock.writeLock().lock();
    try {
      /*
       * Remove ele do conjunto de agendados para o caso dele ter deixado de ser
       * agendado para estar em execuo ou terminado. Se ele continua agendado,
       * no tem problema pois no <i>case SCHEDULED</i> ele volta para o
       * conjunto.
       */
      scheduledCommands.remove(cmd);
      final CommandStatus status = cmd.getStatus();
      final String cmdId = cmd.getId();
      switch (status) {
        case SYSTEM_FAILURE:
          CommandInfo cachedCmd = commandsById.get(cmdId);
          /*
           * S adiciona SYSTEM_FAILURE se o comando tiver terminado ou no
           * estiver presente na cache. Assim, se for obtido um comando com
           * falha de leitura (system failure) do servio de persistncia, mas
           * este comando estiver agendado ou em execuo, estes status vo
           * prevalecer pois ser possvel obt-lo pelo servio de agendamento
           * ou sga.
           */
          if (cachedCmd == null) {
            finishedOrSystemFailureCommands.add(cmd);
          }
          else if (cachedCmd.getStatus() == CommandStatus.FINISHED) {
            /*
             * Se terminou a execuo de um comando que no seja do projeto
             * corrente, ele deve ser removido da cache.
             */
            if (null == projectId || !projectId.equals(cmd.getProjectId())) {
              // Remove o comando da cache.
              commandsById.remove(cmdId);
              return false;
            }
            finishedOrSystemFailureCommands.add(cmd);
          }
          break;
        case FINISHED:
          /*
           * Se terminou a execuo de um comando que no seja do projeto
           * corrente, ele deve ser removido da cache.
           */
          if (null == projectId || !projectId.equals(cmd.getProjectId())) {
            // Remove o comando da cache.
            commandsById.remove(cmdId);
            return false;
          }
          finishedOrSystemFailureCommands.add(cmd);
          break;
        case SCHEDULED:
          scheduledCommands.add(cmd);
          break;

        // Casos no tratados (no fazem nada).
        case DOWNLOADING:
        case EXECUTING:
        case INIT:
        case UPLOADING:
          break;

      }
      commandsById.put(cmdId, cmd);
    }
    finally {
      commandsLock.writeLock().unlock();

      LOGGER.log(methodLog.finished());
    }

    return true;
  }

  /**
   * Remove um comando da cache.
   * 
   * @param commandId Identificador do comando a ser removido da cache.
   * @return comando removido ou {@code null} caso ele no estivesse na cache.
   */
  private CommandInfo removeFromCache(String commandId) {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINER, "removendo comando " + commandId
        + " da cache");
    LOGGER.log(methodLog.begin());

    commandsLock.writeLock().lock();
    try {
      CommandInfo cmd = commandsById.remove(commandId);
      if (cmd == null) {
        return null;
      }
      final CommandStatus status = cmd.getStatus();
      switch (status) {
        case SCHEDULED:
          scheduledCommands.remove(cmd);
          break;
        case SYSTEM_FAILURE:
        case FINISHED:
          finishedOrSystemFailureCommands.remove(cmd);
          break;

        // Casos no tratados (no fazem nada).
        case DOWNLOADING:
        case EXECUTING:
        case INIT:
        case UPLOADING:
          break;
      }
      return cmd;
    }
    finally {
      commandsLock.writeLock().unlock();

      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Cria os ouvintes para verificar o trmino e a remoco de comandos e
   * atualizar o cache.<br>
   */
  private void initializeListeners() {
    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "inicializao de ouvintes");
    LOGGER.log(methodLog.begin());

    CodeBlockLog messageLog =
      new CodeBlockLog(Level.FINER,
        "criando IMessageListener para fim de comandos");
    LOGGER.log(messageLog.begin());

    MessageProxy.addListener(new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        long missing = Integer.MAX_VALUE;
        long total = 0;
        // Salva os comandos terminados na cache.
        commandsLock.writeLock().lock();
        try {
          for (Message aMessage : messages) {
            CommandInfosRetrived data =
              (CommandInfosRetrived) aMessage.getBody();

            for (CommandInfo aCommand : data.getCommandInfos()) {
              LOGGER.log(Level.FINEST,
                "IMessageListener: CmdId:" + aCommand.getId() + ", Status:"
                  + aCommand.getStatus());
              if (aCommand.getStatus() == CommandStatus.FINISHED
                || aCommand.getStatus() == CommandStatus.SYSTEM_FAILURE) {
                saveInCache(aCommand);
              }
            }

            total = data.getTotal(); // O total no muda.
            missing = Math.min(missing, data.getMissing());
          }
        }
        finally {
          commandsLock.writeLock().unlock();
        }
        missing = Math.min(missing, total); // Nunca vai faltar mais que o
                                            // total.
        eventManager.fireEvent(new FinishedOrSystemFailureCommandsLoadingEvent(
          missing, total));
        eventManager.fireEvent(new CacheUpdatedEvent(getSnapshot()));
      }
    }, CommandInfosRetrived.class);
    LOGGER.log(messageLog.finished());

    CodeBlockLog schedulerLog =
      new CodeBlockLog(Level.FINER, "criando SchedulerObserver");
    LOGGER.log(schedulerLog.begin());

    LOGGER.log(schedulerLog.finished());

    CodeBlockLog projectTreeLog =
      new CodeBlockLog(Level.FINER, "criando ProjectTreeListener");
    LOGGER.log(projectTreeLog.begin());

    DesktopFrame.getInstance().getTree()
      .addProjectTreeListener(new ProjectTreeAdapter() {
        @Override
        public void projectChanged(CommonClientProject project) {
          /*
           * Retorna se no h projeto aberto e nada havia sido carregado ou se
           * o projeto aberto j est carregado.
           */
          if ((null == project && null == projectId)
            || (null != project && project.getId().equals(projectId))) {
            return;
          }
          projectId = null == project ? null : project.getId();

          /*
           * Se no h ouvintes de comandos terminados, no h sentido em
           * carreg-los. Neste caso, deve-se apenas se livrar dos que esto em
           * memria.
           */
          if (finishedOrSystemFailureCommandListenersCounter.get() == 0) {
            cleanFinishedCommands();
            return;
          }

          final Runnable runnable = new Runnable() {
            @Override
            public void run() {
              try {
                loadFinishedOrSystemFailureCommands();
                eventManager.fireEvent(new CacheUpdatedEvent(getSnapshot()));
              }
              catch (Exception e) {
                throwExceptionAsEvent(e);
              }
            }
          };
          ThreadFactory threadFactory = Executors.defaultThreadFactory();
          Thread thread = threadFactory.newThread(runnable);
          thread.start();
        }

        @Override
        public void projectClosed(final CommonClientProject project) {
          cleanFinishedCommands();
          projectId = null;
          if (0 < finishedOrSystemFailureCommandListenersCounter.get()) {
            final Collection<CommandInfo> snapshot = getSnapshot();
            eventManager.fireEvent(new CacheUpdatedEvent(snapshot));
          }
        }
      });
    LOGGER.log(projectTreeLog.finished());

    CodeBlockLog cmdLog =
      new CodeBlockLog(Level.FINER, "criando CommandListener");
    LOGGER.log(cmdLog.begin());

    MessageProxy.addListener(new IMessageListener() {
      @Override
      public void onMessagesReceived(final Message... messages)
        throws Exception {

        Runnable runnable = new Runnable() {
          @Override
          public void run() {
            try {
              /*
               * Independente do comando estar na cache ou no, recupera ele
               * pelo servio de persistncia, mantendo assim, seus dados
               * consistentes com o do servidor. Ele poderia no estar na cache
               * aso fosse criado e terminado antes que o reloader pudesse
               * obt-lo.
               */
              for (Message message : messages) {
                CommandNotification notification =
                  (CommandNotification) message.getBody();

                Object prjId = notification.getProjectId();
                String cmdId = notification.getCommandId().toString();
                LOGGER.log(Level.FINEST, "CommandListener: CmdId:" + cmdId
                  + ", project:" + prjId);

                CodeBlockLog log =
                  new CodeBlockLog(Level.FINEST, "obtendo dados do comando "
                    + cmdId + " do servio de persistncia");
                LOGGER.log(log.begin());
                CommandInfo command =
                  ClientRemoteLocator.commandPersistenceService.getCommandInfo(
                    prjId, cmdId);
                LOGGER.log(log.finished());
                if (command == null) {
                  return;
                }

                CommandUpdatedEvent.Type type =
                  CommandUpdatedEvent.Type.valueOf(command
                    .getFinalizationType());
                eventManager.fireEvent(new CommandUpdatedEvent(command, type));
                saveInCache(command);
                eventManager.fireEvent(new CacheUpdatedEvent(getSnapshot()));
              }
            }
            catch (Exception e) {
              throwExceptionAsEvent(e);
            }
          }
        };
        ThreadFactory threadFactory = Executors.defaultThreadFactory();
        Thread thread = threadFactory.newThread(runnable);
        thread.start();
      }
    }, CommandNotification.class);
    LOGGER.log(cmdLog.finished());

    CodeBlockLog cmdPersistLog =
      new CodeBlockLog(Level.FINER, "criando CommandPersistenceListener");
    LOGGER.log(cmdPersistLog.begin());

    MessageProxy.addListener(new IMessageListener() {

      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message message : messages) {
          final CommandPersistenceNotification notification =
            (CommandPersistenceNotification) message.getBody();

          Runnable processNotification = new Runnable() {
            @Override
            public void run() {
              Object prjId = notification.getProjectId();
              String cmdId = notification.getCommandId();

              Collection<CommandInfo> cacheSnapshot = null;
              CommandInfo command = null;

              switch (notification.getType()) {
                case REMOVED:
                  LOGGER.log(Level.FINEST,
                    "CommandPersistenceListener: REMOVED CmdId:" + cmdId,
                    "project:" + prjId);
                  commandsLock.writeLock().lock();
                  try {
                    command = removeFromCache(cmdId);
                    /*
                     * Cria uma imagen instantnea da cache para posteriormente
                     * disparar o evento de que houve mudanas na cache.
                     */
                    if (command != null) {
                      cacheSnapshot = getSnapshot();
                    }
                  }
                  finally {
                    commandsLock.writeLock().unlock();
                  }

                  if (command != null) {
                    eventManager.fireEvent(new CommandUpdatedEvent(command,
                      Type.removed));
                    eventManager
                      .fireEvent(new CacheUpdatedEvent(cacheSnapshot));
                  }
                  break;
                /*
                 * No era necessrio tratar esse evento, uma vez que o reloader
                 * busca comandos novos de tempos em tempos. No entanto,
                 * dependendo do intervalo de atualizao do reloader, um novo
                 * comando demora a ser detectado. Esse mtodo foi implementado
                 * para agilizar essa deteco e garantir um feedback mais
                 * rpido para o usurio ao submeter comandos [CSBASE-1204].
                 */
                case SAVED:
                  LOGGER.log(Level.FINEST,
                    "CommandPersistenceListener: SAVED CmdId:" + cmdId,
                    "project:" + prjId);
                  commandsLock.writeLock().lock();
                  try {
                    if (commandsById.containsKey(cmdId)) {
                      return;
                    }

                    // Caso no encontre, procura o comando no servidor.
                    /*
                     * O comando foi criado mas o reloader ainda no o pegou.
                     * Ele s pode ser obtido antecipadamente atravs do servio
                     * de persistncia.
                     */
                    CodeBlockLog log =
                      new CodeBlockLog(Level.FINEST,
                        "obtendo dados do comando " + cmdId
                          + " do servio de persistncia");
                    LOGGER.log(log.begin());
                    command =
                      ClientRemoteLocator.commandPersistenceService
                        .getCommandInfo(prjId, cmdId);
                    LOGGER.log(log.finished());

                    if (command != null) {
                      saveInCache(command);
                      /*
                       * Cria uma imagen instantnea da cache para
                       * posteriormente disparar o evento de que houve mudanas
                       * na cache.
                       */
                      cacheSnapshot = getSnapshot();
                    }
                  }
                  catch (Exception e) {
                    throwExceptionAsEvent(e);
                  }
                  finally {
                    commandsLock.writeLock().unlock();
                  }

                  if (command != null) {
                    eventManager.fireEvent(new CommandUpdatedEvent(command,
                      Type.created));
                    eventManager
                      .fireEvent(new CacheUpdatedEvent(cacheSnapshot));
                  }
                  break;
                case UPDATED:
                  LOGGER.log(Level.FINEST,
                    "CommandPersistenceListener: UPDATED CmdId:" + cmdId,
                    "project:" + prjId);
                  commandsLock.writeLock().lock();
                  try {
                    CodeBlockLog log =
                      new CodeBlockLog(Level.FINEST,
                        "obtendo dados do comando " + cmdId
                          + " do servio de persistncia");
                    LOGGER.log(log.begin());
                    command =
                      ClientRemoteLocator.commandPersistenceService
                        .getCommandInfo(prjId, cmdId);
                    LOGGER.log(log.finished());

                    if (command != null) {
                      saveInCache(command);
                      /*
                       * Cria uma imagen instantnea da cache para
                       * posteriormente disparar o evento de que houve mudanas
                       * na cache.
                       */
                      cacheSnapshot = getSnapshot();
                    }
                  }
                  catch (Exception e) {
                    throwExceptionAsEvent(e);
                  }
                  finally {
                    commandsLock.writeLock().unlock();
                  }

                  if (command != null) {
                    eventManager.fireEvent(new CommandUpdatedEvent(command,
                      Type.updated));
                    eventManager
                      .fireEvent(new CacheUpdatedEvent(cacheSnapshot));
                  }
                  break;
                default:
                  String errorMessage =
                    String
                      .format(
                        "Notificao de tipo desconhecido.\nProjeto: %s.\nComando: %s.\nTipo de notificao: %s.\n",
                        projectId, cmdId, notification.getType());
                  throw new IllegalArgumentException(errorMessage);
              }
            }
          };

          ThreadFactory threadFactory = Executors.defaultThreadFactory();
          Thread thread = threadFactory.newThread(processNotification);
          thread.start();
        }
      }
    }, CommandPersistenceNotification.class);

    LOGGER.log(cmdPersistLog.finished());

    LOGGER.log(methodLog.finished());

  }

  /**
   * Devo limpar a estrutura de antigas instncias de comandos agendados pois o
   * comando pode ter sido cancelado e o servidor no notifica a cache neste
   * caso.
   */
  private void cleanScheduledCommands() {
    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINER, "limpando comandos agendados");
    LOGGER.log(methodLog.begin());

    for (CommandInfo scheduledCommand : scheduledCommands) {
      commandsById.remove(scheduledCommand.getId());
    }
    scheduledCommands.clear();

    LOGGER.log(methodLog.finished());
  }

  /**
   * Carrega os comandos agendados.
   * 
   * @param accumulator lista que ir armazenar eventos gerados por este mtodo
   *        e que devem ser lanados aps sua chamada.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  private void loadScheduledCommands(Set<CommandUpdatedEvent> accumulator)
    throws RemoteException {

    scheduledCommands.clear();

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINER, "carregando comandos agendados");
    LOGGER.log(methodLog.begin());

    CommandInfo[] queueds =
      ClientRemoteLocator.schedulerService.getQueuedCommands();

    LOGGER.log(methodLog.finished());

    CodeBlockLog eventLog =
      new CodeBlockLog(Level.FINER, "notificacao de comandos agendados");
    LOGGER.log(eventLog.begin());

    for (CommandInfo aScheduledCommand : queueds) {
      if (null != accumulator) {
        boolean updated = commandsById.containsKey(aScheduledCommand.getId());
        Type eventType = updated ? Type.updated : Type.created;
        accumulator.add(new CommandUpdatedEvent(aScheduledCommand, eventType));
      }
      saveInCache(aScheduledCommand);
    }

    LOGGER.log(eventLog.finished());
  }

  /**
   * Carrega os comandos em execuo.
   * 
   * @param accumulator lista que ir armazenar eventos gerados por este mtodo
   *        e que devem ser lanados aps sua chamada.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  private void loadRunningCommands(Set<CommandUpdatedEvent> accumulator)
    throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINER, "carregando comandos em execuo");
    LOGGER.log(methodLog.begin());

    Vector<CommandInfo> runningCommands =
      ClientRemoteLocator.sgaService.getAllSGACommands();

    LOGGER.log(methodLog.finished());

    CodeBlockLog eventLog =
      new CodeBlockLog(Level.FINER, "notificacao de comandos em execuo");
    LOGGER.log(eventLog.begin());

    for (CommandInfo runningCommand : runningCommands) {
      saveInCache(runningCommand);
      if (null != accumulator) {
        accumulator.add(new CommandUpdatedEvent(runningCommand, Type.updated));
      }
    }

    LOGGER.log(eventLog.finished());
  }

  /**
   * Devo limpar a estrutura de antigas instncias de comandos agendados pois o
   * comando pode ter sido cancelado e o servidor no notifica a cache neste
   * caso.
   */
  private void cleanFinishedCommands() {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINER, "limpando comandos terminados");
    LOGGER.log(methodLog.begin());

    finishedOrSystemFailureCommandsLock.lock();
    try {
      if (finishedOrSystemFailureCommands.size() == 0) {
        return;
      }
      commandsLock.writeLock().lock();
      try {
        for (CommandInfo scheduledCommand : finishedOrSystemFailureCommands) {
          commandsById.remove(scheduledCommand.getId());
        }
        finishedOrSystemFailureCommands.clear();
      }
      finally {
        commandsLock.writeLock().unlock();
      }
    }
    finally {
      finishedOrSystemFailureCommandsLock.unlock();

      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Recarrega os comandos terminados.
   * 
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  private void loadFinishedOrSystemFailureCommands() throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINER, "carregando comandos terminados");
    LOGGER.log(methodLog.begin());

    // Salva os comandos terminados na cache.
    finishedOrSystemFailureCommandsLock.lock();
    try {
      if (null == projectId) {
        return;
      }

      /*
       * Requisita que o servio de persistncia retorne os comandos persistidos
       * no projeto corrente atravs do servio de mensagens.
       */
      long loading =
        ClientRemoteLocator.commandPersistenceService.requestCommandInfos(
          projectId, LOAD_FINISHEDS_DESIRED_RESPONSE_TIME_MILLIS);
      // Envia um evento indicando quantos comandos seo carregados.
      eventManager.fireEvent(new FinishedOrSystemFailureCommandsLoadingEvent(
        loading));
    }
    finally {
      finishedOrSystemFailureCommandsLock.unlock();
    }

    LOGGER.log(methodLog.finished());
  }

  /**
   * Apaga e recarrega os comandos de um projeto na cache, forando a releitura
   * das informaes persistidas. Esse mtodo somente deve ser utilizado quando
   * os arquivos de persistncia dos comandos de um projeto forem
   * *comprometidos* no sistema de arquivos. Em outros casos, para atualizar as
   * informaes da cache, deve ser utilizado o mtodo
   * {@link #runManualReload(boolean)}.
   * 
   * @param prjId Identificador do projeto a ter seus comandos recarregados.
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public void hardReload(Object prjId) throws RemoteException {
    commandsLock.writeLock().lock();
    try {
      // Usando iterator para remover itens de maneira segura enquanto
      // itero pela coleo.
      Iterator<CommandInfo> iterator = commandsById.values().iterator();
      while (iterator.hasNext()) {
        CommandInfo command = iterator.next();
        if (command.getProjectId().equals(prjId)) {
          iterator.remove();
        }
      }
    }
    finally {
      commandsLock.writeLock().unlock();
    }

    // Faz a cache a ler novamente os comandos persistidos deste projeto.
    runManualReload(0 < finishedOrSystemFailureCommandListenersCounter.get()
      && prjId.equals(this.projectId));
  }

  /**
   * Recarrega os comandos armazenados e em execuo no mapa de comandos.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  private void reload() throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINEST, "recarga de comandos");
    LOGGER.log(methodLog.begin());

    Collection<CommandInfo> commands;
    Set<CommandUpdatedEvent> events = new HashSet<CommandUpdatedEvent>();

    /*
     * Se a ltima atualizao falhou deve-se garantir que, caso esta obtenha
     * sucesso, todos os ouvintes de comandos recebam um evento indicando que o
     * erro foi sanado.
     */
    commandsLock.readLock().lock();
    try {
      if (exceptionThrownDuringLastUpdate.compareAndSet(true, false)) {
        for (CommandInfo command : commandsById.values()) {
          CommandUpdatedEvent event =
            new CommandUpdatedEvent(command, Type.updated);
          events.add(event);
        }
      }
    }
    finally {
      commandsLock.readLock().unlock();
    }
    commandsLock.writeLock().lock();
    try {
      cleanScheduledCommands();
      /**
       * A ordem aqui faz diferena. <br>
       * Como o ciclo de vida de um comando  "agendado", "em execuo" e
       * "terminado", se mudar a ordem da carga pode-se deixar de carregar algum
       * comando que mude de estado entre as cargas. <br>
       * Ex.: Existe um comando A agendado. Inverte-se a ordem e carregamos
       * primeiro os comandos em execuo e depois os agendados. Se entre as
       * cargas o comando mudar do estado "agendado" para o estado "em
       * execuo", o comando no ser carregado. <br>
       */
      loadScheduledCommands(events);
      loadRunningCommands(events);

      lastUpdateInMilliseconds = System.currentTimeMillis();
      commands = getSnapshot();
    }
    finally {
      commandsLock.writeLock().unlock();
    }
    LOGGER.log(methodLog.finished());

    CodeBlockLog eventLog =
      new CodeBlockLog(Level.FINEST, "notificao de comandos");
    LOGGER.log(eventLog.begin());

    for (CommandUpdatedEvent event : events) {
      eventManager.fireEvent(event);
    }
    eventManager.fireEvent(new CacheUpdatedEvent(commands));

    LOGGER.log(eventLog.finished());
  }

  /**
   * Caso o {@link CommandsCache#reloader} esteja desligado, este mtodo ir
   * forar uma recarga dos dados.
   * 
   * @param loadFinishedCommands indica se deve carregar os comandos terminados
   *        do projeto corrente caso estes ainda no tenham sido carregados.
   * 
   * @return <tt>false</tt> se ocorreu um erro durante a recarga, se os dados
   *         atuais ainda so vlidos ou se a recarga automtica est ligada.
   * 
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  private boolean runManualReload(boolean loadFinishedCommands)
    throws RemoteException {
    boolean status = false;

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "recarga manual de comandos");
    LOGGER.log(methodLog.begin());

    if (!reloader.isRunning()) {
      long expirationDate =
        lastUpdateInMilliseconds + reloadIntervalUnit.toMillis(reloadInterval);
      if (System.currentTimeMillis() > expirationDate) {
        reload();
        status = true;
      }
    }

    if (loadFinishedCommands) {
      loadFinishedOrSystemFailureCommands();
    }

    LOGGER.log(methodLog.begin());

    return status;
  }

  /**
   * Cria uma instncia de {@link TaskScheduler} que ir executar o mtodo
   * {@link #reload()} a cada {@link CommandsCache#reloadInterval}
   * {@link CommandsCache#reloadIntervalUnit}.
   * 
   * @return uma instncia de {@link TaskScheduler} que ir executar o mtodo
   *         {@link #reload()} a cada {@link CommandsCache#reloadInterval}
   *         {@link CommandsCache#reloadIntervalUnit}.
   */
  private TaskScheduler createReloader() {
    Runnable reloadTask = new Runnable() {
      @Override
      public void run() {
        try {
          reload();
        }
        catch (Exception e) {
          throwExceptionAsEvent(e);
        }
      }
    };
    return new TaskScheduler(reloadTask, reloadInterval, reloadIntervalUnit);
  }

  /**
   * Obtm o intervalo de atualizao da cache em segundos.
   * 
   * @return o intervalo de atualizao da cache em segundos.
   */
  private long getReloadInterval() {
    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "obtendo intervalo de recarga de comandos");
    LOGGER.log(methodLog.begin());

    try {
      RemoteTask<Long> task = new RemoteTask<Long>() {
        @Override
        protected void performTask() throws Exception {
          long interval =
            ClientRemoteLocator.sgaService.getCommandsUpdateInterval();
          setResult(interval);
        }
      };
      Window window = DesktopFrame.getInstance().getDesktopFrame();
      String title = DesktopFrame.getInstance().getTitle();
      String message = LNG.get("CommandsCache.task.message");
      if (task.execute(window, title, message)) {
        return task.getResult();
      }
      return DEFAULT_RELOAD_INTERVAL_S;
    }
    finally {
      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Obtm um comando da cache.<br>
   * A cache ser inicializada caso ela ainda no tenha sido.
   * 
   * @param commandId identificador nico do comando.
   * @return o comando cujo identificador  <i>paramId</i> ou <tt>null</tt> caso
   *         ele no exista ou no esteja na cache.
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  private CommandInfo getCommandFromCache(String commandId)
    throws RemoteException {

    CodeBlockLog methodLog =
      new CodeBlockLog(Level.FINE, "pegando comando " + commandId + " da cache");
    LOGGER.log(methodLog.begin());

    runManualReload(false);

    commandsLock.readLock().lock();
    try {
      return commandsById.get(commandId);
    }
    finally {
      commandsLock.readLock().unlock();

      LOGGER.log(methodLog.finished());
    }
  }

  /**
   * Obtm uma lista dos comandos que esto na cache no momento da chamada deste
   * mtodo.
   * 
   * @return uma lista dos comandos que esto na cache no momento da chamada
   *         deste mtodo.
   */
  private Collection<CommandInfo> getSnapshot() {
    CodeBlockLog methodLog = new CodeBlockLog(Level.FINEST, "obtendo snapshot");
    LOGGER.log(methodLog.begin());

    Collection<CommandInfo> values = new ArrayList<CommandInfo>();
    commandsLock.readLock().lock();
    try {
      values.addAll(commandsById.values());
    }
    finally {
      commandsLock.readLock().unlock();

      LOGGER.log(methodLog.finished());
    }
    return values;
  }

  /**
   * Sinaliza os listeners de que uma exceo ocorreu durante a atualizao dos
   * comandos.
   * 
   * @param e exceo que interrompeu o processo de atualizao dos comandos.
   */
  private void throwExceptionAsEvent(Exception e) {
    String updateCmdExceptionMsg, updateCacheExceptionMsg;
    if (e instanceof RemoteException) {
      ClientRemoteMonitor.getInstance().invalidate();
      updateCacheExceptionMsg =
        updateCmdExceptionMsg =
          LNG.get("CommandsCache.exception.RemoteException");
    }
    else {
      updateCacheExceptionMsg = LNG.get("CommandsCache.exception.cache.update");
      updateCmdExceptionMsg = LNG.get("CommandsCache.exception.command.update");
    }
    eventManager.fireEvent(new CommandUpdatedEvent(e, updateCmdExceptionMsg));
    eventManager.fireEvent(new CacheUpdatedEvent(e, updateCacheExceptionMsg));
    exceptionThrownDuringLastUpdate.set(true);
  }

  /**
   * Adiciona um observador de eventos. <br>
   * Liga a recarga automtica caso esta j no esteja ligado.
   * 
   * @param <E> tipo do evento a ser observado.
   * @param <L> tipo do observador do evento.
   * @param listener observador do evento.
   * @param clazz classe do evento a ser observado.
   */
  private <E extends IEvent, L extends EventListener<E>> void addEventListener(
    L listener, Class<E> clazz) {
    eventManagerLock.lock();
    try {
      eventManager.addEventListener(listener, clazz);
      reloader.start();
    }
    finally {
      eventManagerLock.unlock();
    }
  }
}
