package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.ProjectStatusModificationDialog;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.StatusModificationTask;

/**
 * Ao para alterao do status de alocao de "Sem alocao" para "Alocado".
 * 
 * @author jnlopes
 * 
 */
public class ChangeStatusAllocateAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public ChangeStatusAllocateAction(ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    int allocatedSpace =
      ProjectStatusModificationDialog.openDialog(projectsManager, projects);
    if (allocatedSpace <= 0) {
      return;
    }

    boolean ret = runTask(projects, allocatedSpace);
    if (ret) {
      updateStatusOfLocalProjects(projects, allocatedSpace);
      projectsManager.refreshProjectsTable();
    }
    else {
      String errMsg =
        getString("ChangeStatusAllocateAction.error.allocation.message");
      String errTitle =
        getString("ChangeStatusAllocateAction.error.allocation.title");
      StandardDialogs.showErrorDialog(projectsManager.getApplicationFrame(),
        errTitle, errMsg);
    }
  }

  /**
   * Define o status de alocao dos projetos validados como
   * <code>ProjectSpaceAllocation.ALLOCATED</code> e o espao de alocao.
   * 
   * @param projects Projetos validados.
   * @param allocatedSpace Espao a ser definido como alocado.
   */
  private void updateStatusOfLocalProjects(List<ProjectsManagerData> projects,
    int allocatedSpace) {
    for (final ProjectsManagerData pmd : projects) {
      pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.ALLOCATED);
      pmd.setRequestedSpace(allocatedSpace);
      pmd.setAllocatedSpace(allocatedSpace);
    }
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @param allocatedSpace Espao a ser definido como alocado.
   * @return Boolean indicando se a operao foi bem sucedida.
   */
  private boolean runTask(List<ProjectsManagerData> projects, int allocatedSpace) {
    ProjectsManager projectsManager = getProjectsManager();
    ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    StatusModificationTask smt =
      new StatusModificationTask(projectsManager, projects,
        ProjectSpaceAllocation.ALLOCATED, allocatedSpace);
    String taskTitle = getString("ChangeStatusAllocateAction.running.title");
    String taskMsg = getString("ChangeStatusAllocateAction.running.message");
    smt.execute(appFrame, taskTitle, taskMsg);
    if (smt.wasCancelled()) {
      String err = getString("ChangeStatusAllocateAction.cancelled.message");
      smt.showError(err);
      return false;
    }
    if (smt.getStatus() != true) {
      Exception exception = smt.getError();
      String err =
        getString("ChangeStatusAllocateAction.allocation.status.error.message");
      String msg = err + "\n" + exception.getMessage();
      StandardDialogs.showErrorDialog(appFrame, msg, "");
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    List<ProjectsManagerData> prjList = new ArrayList<ProjectsManagerData>();

    prjList.addAll(projectList);
    List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    int i = 0;
    while (i < prjList.size()) {
      ProjectsManagerData pmd = prjList.get(i);
      ProjectSpaceAllocation spaceAllocation = pmd.getProjectSpaceAllocation();
      if (spaceAllocation != ProjectSpaceAllocation.NO_ALLOCATION) {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
      else {
        i++;
      }
    }

    ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      String deniedMessage =
        getString("ChangeStatusAllocateAction.project.selection.denied.message")
          + getString("ChangeStatusAllocateAction.project.requirements.message");
      StatusDialog
        .showNoneOkDialog(
          projectsManager,
          deniedMessage,
          failedProjects,
          getString("ChangeStatusAllocateAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      String someDeniedMessage =
        getString("ChangeStatusAllocateAction.project.selection.some.denied.message")
          + getString("ChangeStatusAllocateAction.project.requirements.message");
      int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("ChangeStatusAllocateAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return null;
  }
}
