/**
 * $Id$
 */

package csbase.client.project.tasks;

import java.awt.Window;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;

/**
 * Tarefa responsvel por obter o contedo de um arquivo de projeto [que fica no
 * servidor] e o armazenar em um string no cliente. Observe que todo o contedo
 * do arquivo  carregado em memria, e em forma de String, ento essa tarefa 
 * adequada apenas para arquivos de texto e com um tamanho 'razovel'.
 */
public class GetFileContentTask extends RemoteTask<String> {

  /** Tamanho mximo para o buffer. */
  private static final int MAX_BUFFER_SIZE = 100 * 1024; // 100K 

  /** o arquivo em questo */
  private final ClientProjectFile file;

  /**
   * Constri a tarefa. Chame 'execute()' para inici-la efetivamente.
   * 
   * @param file - o arquivo a ser obtido.
   */
  public GetFileContentTask(ClientProjectFile file) {
    this.file = file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    int fileSize = (int) file.size();
    BufferedReader reader = null;
    try {
      // buffer para armazenar o contedo do arquivo:
      final StringBuilder stringBuffer = new StringBuilder(fileSize);
      final char[] buffer = new char[MAX_BUFFER_SIZE];
      final InputStream stream = file.getInputStream();
      reader = new BufferedReader(new InputStreamReader(stream));
      int nbytes = reader.read(buffer);
      while (nbytes > 0) {
        stringBuffer.append(buffer, 0, nbytes);
        nbytes = reader.read(buffer);
      }
      setResult(stringBuffer.toString());
    }
    catch (OutOfMemoryError e) {
      StandardErrorDialogs.showErrorDialog(
        parentWindow,
        LNG.get("GetFileContentTask.title"),
        String.format(LNG.get("GetFileContentTask.outOfMemoryMessage"),
          file.getName(), (fileSize / (1024.0 * 1024.0))));
    }
    finally {
      if (reader != null) {
        reader.close();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String[] getAdditionalInfo() {
    return new String[] { String.format(LNG.get("GetFileContentTask.fileInfo"),
      FileUtils.joinPath('/', file.getPath())) };
  }

  /**
   * Executa a tarefa efetivamente.
   * 
   * @param taskDialogParent - janela de referncia para o dilogo da tarefa.
   * @return true se a tarefa foi executada at o final (h um resultado) false
   *         se a ocorrncia de um erro impediu a execuo da tarefa (no h
   *         resultado associado)
   */
  public boolean execute(Window taskDialogParent) {
    return super.execute(taskDialogParent, LNG.get("GetFileContentTask.title"),
      String.format(LNG.get("GetFileContentTask.msg"), file.getName()));
  }

  /**
   * Executa a tarefa efetivamente, utilizando o DeskTopFrame como janela pai.
   * 
   * @return true se a tarefa foi executada at o final (h um resultado) false
   *         se a ocorrncia de um erro impediu a execuo da tarefa (no h
   *         resultado associado)
   */
  public boolean execute() {
    return execute(DesktopFrame.getInstance().getDesktopFrame());
  }
}
