package csbase.client.util.csvpanel.table;

import java.awt.Component;

import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableColumnModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import csbase.client.util.csvpanel.columns.ColumnGenerator;

/**
 * Tabela de visualizao de CSVs.
 */
public class CSVTable extends JTable {

  /**
   * Construtor.
   */
  public CSVTable() {
    this(null);
  }

  /**
   * Construtor.
   * 
   * @param data dados a serem visualizados.
   */
  public CSVTable(String[][] data) {
    this(data, false, 0);
  }

  /**
   * Construtor.
   * 
   * @param data dados a serem visualizados.
   * @param editable Indica se os dados podero ser editados na tabela.
   * @param numRowsAsHeader Indica o nmero de linhas no incio do arquivo CSV
   *        que devem ser consideradas como cabealho da tabela.
   */
  public CSVTable(String[][] data, boolean editable, int numRowsAsHeader) {
    super(new CSVTableModel(data, editable, numRowsAsHeader));
    configureTableUI();
  }

  /**
   * Ajusta as propriedades da tabela.
   */
  private void configureTableUI() {
    setFillsViewportHeight(true);
    setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
    setColumnSelectionAllowed(true);
    setRowSelectionAllowed(true);
    setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
    setAutoCreateRowSorter(true);
    getColumnModel().addColumnModelListener(new TableColumnModelListener() {
      @Override
      public void columnSelectionChanged(ListSelectionEvent arg0) {
        //Ignora evento
      }

      @Override
      public void columnRemoved(TableColumnModelEvent arg0) {
        //Ignora evento
      }

      @Override
      public void columnMoved(TableColumnModelEvent arg0) {
        //Ignora evento
      }

      @Override
      public void columnMarginChanged(ChangeEvent arg0) {
        //Ignora evento
      }

      @Override
      public void columnAdded(TableColumnModelEvent e) {
        adjustColumnPreferredWidths(e.getFromIndex(), e.getToIndex());
      }
    });
  }

  /**
   * Gera automaticamente os valores de uma determinada coluna a partir de uma
   * linha.
   * 
   * @param generator o gerador de valores.
   * @param rows linhas selecionadas.
   * @param col o ndice da coluna.
   */
  public void generateColum(ColumnGenerator<?> generator, int[] rows, int col) {
    for (int i = 0; i < rows.length; i++) {
      int index = rows[i];
      Object valueAt = generator.getValueAt(i);
      String stringValue = null;
      if (valueAt != null) {
        stringValue = String.valueOf(valueAt);
      }
      setValueAt(stringValue, index, col);
    }
  }

  /**
   * Ajusta as larguras das colunas.
   * 
   * @param from ndice da coluna inicial
   * @param to ndice da coluna final
   */
  public void adjustColumnPreferredWidths(int from, int to) {
    TableColumnModel columns = getColumnModel();
    for (int col = from; col <= to; col++) {
      TableColumn column = columns.getColumn(col);
      TableCellRenderer headerRenderer = column.getHeaderRenderer();
      if (headerRenderer == null) {
        headerRenderer = new DefaultTableCellRenderer();
      }
      Object headerValue = column.getHeaderValue();
      if (headerValue != null) {
        Component headerComp =
          headerRenderer.getTableCellRendererComponent(this, headerValue,
            false, false, 0, col);
        int width = headerComp.getPreferredSize().width;
        column.setPreferredWidth(Math.max(width, 100));
      }
    }
  }

  /**
   * Carrega novos dados na tabela.
   * 
   * @param data a tabela da dados.
   * @param editable Indica se os dados podero ser editados na tabela.
   */
  public void setData(String[][] data, boolean editable) {
    CSVTableModel tableModel = getModel();
    tableModel.setEditable(editable);
    tableModel.setData(data);
  }

  /**
   * Indica se a tabela aceita edies.
   * 
   * @return verdadeiro se a tabela aceita edies ou falso, caso contrrio.
   */
  public boolean isEditable() {
    CSVTableModel tableModel = getModel();
    return tableModel.isEditable();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CSVTableModel getModel() {
    return (CSVTableModel) super.getModel();
  }

}
