package csbase.client.util.sga;

import java.rmi.RemoteException;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import csbase.logic.SharedObject;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SharedObjectServiceInterface;

/**
 * Classe que representa o histrico de descries dos comandos executados. As
 * ltimas {@link #MAX_DESCRIPTIONS} descries podem ser consultadas e novas
 * descries podem ser salvas no histrico para que possam ser reutilizadas.
 */
public class CommandDescriptionHistory {

  /**
   * Nmero mximo de descries que devem ser armazenadas no
   * {@link SharedObject}.
   */
  private static final int MAX_DESCRIPTIONS = 10;

  /**
   * Nome do {@link SharedObject} que armazena as descries dos comandos do
   * usurio.
   */
  private static final String COMMAND_DESCRIPTION_OBJ =
    "command_description_history";

  /**
   * Prefixo da chave utilizada no mapa de informaes gravado no
   * {@link SharedObject}.
   */
  private static final String COMMAND_DESCRIPTION_KEY = "description_";

  /**
   * Nome da categoria para o {@link SharedObject} que armazena as descries
   * dos comandos do usurio.
   */
  private static final String HISTORY_CATEGORY = "History";

  /**
   * Obtm as descries utilizadas em execues anteriores.
   * 
   * @return a lista de descries utilizadas anteriormente.
   * @throws RemoteException em caso de falha na comunicao com o servidor.
   */
  public String[] loadPreviousDescriptions() throws RemoteException {
    SharedObject sharedObject = readSharedObject();
    if (sharedObject != null) {
      Collection<String> descriptions =
        sharedObjectContentToHistory(sharedObject.getContents());
      return descriptions.toArray(new String[descriptions.size()]);
    }
    else {
      return new String[0];
    }
  }

  /**
   * Retorna o {@link SharedObject} que armazena o histrico de descries dos
   * comandos.
   * 
   * @return o {@link SharedObject}
   * @throws RemoteException em caso de falha na comunicao com o servidor.
   */
  private SharedObject readSharedObject() throws RemoteException {
    SharedObjectServiceInterface sharedObjectService =
      ClientRemoteLocator.sharedObjectService;
    try {
      SharedObject sharedObject =
        sharedObjectService.getSharedObject(HISTORY_CATEGORY, User
          .getLoggedUser().getId(), COMMAND_DESCRIPTION_OBJ);
      return sharedObject;
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Transforma o contedo do {@link SharedObject} no histrico de descries de
   * comandos.
   * 
   * @param contents contedo do {@link SharedObject}.
   * @return o histrico de descries.
   */
  @SuppressWarnings("unchecked")
  private Deque<String> sharedObjectContentToHistory(Object contents) {
    Deque<String> descriptions = new LinkedList<String>();
    if (contents != null) {
      Map<String, Object> historyMap = (Map<String, Object>) contents;
      for (int i = 0; i < MAX_DESCRIPTIONS; i++) {
        Object object = historyMap.get(COMMAND_DESCRIPTION_KEY + i);
        if (object != null) {
          descriptions.addLast(object.toString());
        }
        else {
          break;
        }
      }
    }
    return descriptions;
  }

  /**
   * Transforma o histrico de descries de comandos no contedo do
   * {@link SharedObject}.
   * 
   * @param history histrico de descries.
   * @return o contedo do {@link SharedObject}.
   */
  private Object historyToSharedObjectContent(Deque<String> history) {
    Map<String, Object> contents = new HashMap<String, Object>();
    for (int i = 0; i < MAX_DESCRIPTIONS; i++) {
      if (!history.isEmpty()) {
        contents.put(COMMAND_DESCRIPTION_KEY + i, history.removeFirst());
      }
      else {
        break;
      }
    }
    return contents;
  }

  /**
   * Salva a descrio do comando para poder reutilizar futuramente.
   * 
   * @param description a descrio a ser salva.
   * @throws RemoteException em caso de falha na comunicao com o servidor.
   */
  public void saveDescription(final String description) throws RemoteException {
    if (description == null) {
      return;
    }
    SharedObject sharedObject = readSharedObject();
    Deque<String> descriptions;
    if (sharedObject != null) {
      descriptions = sharedObjectContentToHistory(sharedObject.getContents());
      if (descriptions.contains(description)) {
        // Remove para evitar duplicatas no histrico.
        descriptions.remove(description);
      }
    }
    else {
      descriptions = new LinkedList<String>();
      sharedObject =
        new SharedObject(HISTORY_CATEGORY, User.getLoggedUser().getId(),
          COMMAND_DESCRIPTION_OBJ, false, null);
    }

    // Coloca o ltimo utilizado sempre no incio.
    descriptions.addFirst(description);

    Object content = historyToSharedObjectContent(descriptions);
    sharedObject.setContents(content);

    SharedObjectServiceInterface sharedObjectService =
      ClientRemoteLocator.sharedObjectService;
    sharedObjectService.saveSharedObject(sharedObject);
  }

}
