package csbase.client.applications.flowapplication.graph;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import csbase.exception.BugException;
import csbase.exception.algorithms.ParameterNotFoundException;

/**
 * Parmetros (transferable)
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ParametersTransferable implements Transferable {
  public static final DataFlavor DATA_FLAVOR;

  private static final String PARAMETER_LABEL = "LABEL";
  private static final String PARAMETER_TYPE = "TYPE";
  private static final String PARAMETER_VALUE = "VALUE";

  static {
    try {
      DATA_FLAVOR =
        new DataFlavor(DataFlavor.javaJVMLocalObjectMimeType + ";class="
          + ParametersTransferable.class.getName());
    }
    catch (final ClassNotFoundException e) {
      throw new BugException(e);
    }
  }

  /**
   * Mapa de parmetros.
   */
  private final Map<String, Map<String, String>> parameterMapsByName;

  /**
   * Construtor
   * 
   * @param node n
   */
  public ParametersTransferable(final GraphNode node) {
    if (node == null) {
      throw new IllegalArgumentException("O parmetro node est nulo.");
    }
    this.parameterMapsByName = new HashMap<String, Map<String, String>>();
    final Set<String> parameterNames = node.getParameterNames();
    for (final String parameterName : parameterNames) {
      String parameterLabel;
      String parameterValue;
      String parameterType;
      try {
        parameterLabel = node.getParameterLabel(parameterName);
        parameterValue = node.getParameterValue(parameterName);
        parameterType = node.getParameterType(parameterName);
      }
      catch (final ParameterNotFoundException e) {
        final IllegalStateException illegalStateException =
          new IllegalStateException();
        illegalStateException.initCause(e);
        throw illegalStateException;
      }
      final Map<String, String> parameterMap = new HashMap<String, String>();
      parameterMap.put(PARAMETER_LABEL, parameterLabel);
      parameterMap.put(PARAMETER_VALUE, parameterValue);
      parameterMap.put(PARAMETER_TYPE, parameterType);
      this.parameterMapsByName.put(parameterName, parameterMap);
    }
  }

  public String getParameterLabel(final String parameterName) {
    final Map<String, String> parameterMap =
      this.parameterMapsByName.get(parameterName);
    if (parameterMap == null) {
      return null;
    }
    return parameterMap.get(PARAMETER_LABEL);
  }

  public Set<String> getParameterNames() {
    return Collections.unmodifiableSet(this.parameterMapsByName.keySet());
  }

  public String getParameterType(final String parameterName) {
    final Map<String, String> parameterMap =
      this.parameterMapsByName.get(parameterName);
    if (parameterMap == null) {
      return null;
    }
    return parameterMap.get(PARAMETER_TYPE);
  }

  public String getParameterValue(final String parameterName) {
    final Map<String, String> parameterMap =
      this.parameterMapsByName.get(parameterName);
    if (parameterMap == null) {
      return null;
    }
    return parameterMap.get(PARAMETER_VALUE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getTransferData(final DataFlavor flavor)
    throws UnsupportedFlavorException {
    if (!isDataFlavorSupported(flavor)) {
      throw new UnsupportedFlavorException(flavor);
    }
    return Collections.unmodifiableMap(this.parameterMapsByName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataFlavor[] getTransferDataFlavors() {
    return new DataFlavor[] { DATA_FLAVOR };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isDataFlavorSupported(final DataFlavor flavor) {
    return DATA_FLAVOR.equals(flavor);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return getClass().getName() + "\n" + this.parameterMapsByName;
  }
}
