/*
 * $Id:$
 */
package csbase.client.util.iostring;

import java.awt.Window;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.logic.ClientFile;

/**
 * Classe auxiliar para leitura de um arquivo completo como texto.
 * 
 * @author Tecgraf
 */
public class TextReader {

  /**
   * Leitura de arquivo e obteno da string completa de seu contedo (<b> no 
   * criada task para esse mtodo</b> - ver
   * {@link #readAllWithTask(Window, ClientFile, Charset)}.
   * 
   * @param file arquivo
   * @param charset charset
   * @param dealerStatus objeto para controle de estado da leitura
   * @return texto
   * @throws IOException em caso de erro de I/O
   */
  final public static String readAllWithoutTask(final ClientFile file,
    final Charset charset, final TextDealerStatusInterface dealerStatus)
    throws IOException {
    final Charset readCharset;
    if (charset == null) {
      final Client client = Client.getInstance();
      readCharset = client.getSystemDefaultCharset();
    }
    else {
      readCharset = charset;
    }

    if ((file == null) || file.isDirectory()) {
      return null;
    }

    BufferedReader reader = null;
    try {
      final long fileSize = file.size();
      if (fileSize > Integer.MAX_VALUE) {
        throw new IOException("file too big");
      }
      final int totalBytes = (int) fileSize;
      final StringBuilder stringBuilder = new StringBuilder(totalBytes);
      final InputStream stream = file.getInputStream();
      final InputStreamReader rd = new InputStreamReader(stream, readCharset);
      reader = new BufferedReader(rd);
      final char[] buffer = new char[1024 * 10];
      int readBytes = 0;
      int nbytes = reader.read(buffer);
      readBytes += nbytes;
      while (nbytes > 0) {
        if (dealerStatus != null && dealerStatus.isTextOperationInterrupted()) {
          return null;
        }
        stringBuilder.append(buffer, 0, nbytes);
        nbytes = reader.read(buffer);
        readBytes += nbytes;
        final double perc = ((double) readBytes) / totalBytes * 100.;
        final int percentage = (int) Math.round(perc);
        if (dealerStatus != null) {
          dealerStatus.setTextOperationPercentage(percentage);
        }
      }
      if (dealerStatus != null) {
        dealerStatus.setTextOperationPercentage(100);
      }
      reader.close();
      return stringBuilder.toString();
    }
    finally {
      if (reader != null) {
        reader.close();
      }
    }
  }

  /**
   * Obtm os dados do arquivo como uma string.
   * 
   * @param owner janela-me da task a ser criada para leitura
   * @param file arquivo
   * @param charset encoding a ser usado para montagem do texto em funo dos
   *        bytes lidos pelo stream (se ajustado para {@code null}, ser usado o
   *        default do sistema conforme
   *        {@code Client#getSystemDefaultCharset())}.
   * @return string representando o texto ou {@code null}.
   * @throws Exception em caso de falha.
   */
  public static String readAllWithTask(final Window owner,
    final ClientFile file, final Charset charset) throws Exception {

    final ReaderTask task = new ReaderTask(file, charset);
    String msg = LNG.get("TextReader.task.message");
    msg = msg + " (" + file.getName() + ")";
    if (!task.execute(owner, "", msg, true, false)) {
      final Exception error = task.getError();
      if (error != null) {
        throw error;
      }
      return "";
    }

    final String str = task.getResult();
    return str;
  }
}
