package csbase.client.algorithms.commands.newview;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;
import java.util.HashMap;
import java.util.Map;

import javax.swing.JSplitPane;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StatusBar;
import csbase.client.algorithms.AlgorithmConfiguratorFactory;
import csbase.client.algorithms.AlgorithmConfiguratorView;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent.Type;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.configurator.FlowAlgorithmConfiguratorView;
import csbase.client.applications.flowapplication.filters.AddNodePopupActionFilter;
import csbase.client.applications.flowapplication.filters.PopupFilter;
import csbase.client.applications.flowapplication.filters.WorkspaceFilter;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.graph.GraphNodeImageDecoration.DecorationType;
import csbase.client.applications.flowapplication.graph.actions.Action;
import csbase.client.applications.flowapplication.graph.actions.GraphElementAction;
import csbase.client.applications.flowapplication.messages.PickNodeMessage;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.kernel.ClientException;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.gui.log.tab.Tab;
import csbase.client.util.gui.log.tab.AbstractTab.TabType;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationInfo.FinalizationInfoType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.logic.ProgressData;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;

/**
 * Viso de um comando do tipo fluxo.
 */
class FlowCommandView implements CommandView {

  /**
   * Comando sendo visualizado.
   */
  protected CommandInfo command;

  /**
   * Configurador do fluxo.
   */
  protected FlowAlgorithmConfigurator configurator;

  /**
   * Painel principal da viso.
   */
  protected JSplitPane mainComponent;

  /**
   * Viso do configurador do fluxo, com todas as suas "caixinhas".
   */
  protected FlowAlgorithmConfiguratorView configuratorView;

  /**
   * Viso detalhada do comando. Pode mostrar o detalhamento do fluxo inteiro ou
   * de um s n (caso o n for selecionado pelo usurio).
   */
  protected CommandView selectedView;

  /**
   * Viso de detalhamento do fluxo.
   */
  protected CommandView detailsView;

  /**
   * Vises dos ns j criadas.
   */
  protected Map<Integer, NodeCommandView> nodeViewCache;

  /**
   * Componente dono da viso.
   */
  protected DesktopComponentFrame owner;

  /**
   * N selecionado atualmente pelo usurio.
   */
  private GraphNode selectedNode;

  /**
   * Construtor.
   * 
   * @param cmd o comando sendo visualizado (No aceita {@code null}).
   * @param configurator o configurador do fluxo (No aceita {@code null}).
   * 
   * @throws ClientException em caso de erro na criao da viso do comando.
   */
  public FlowCommandView(final CommandInfo cmd,
    final FlowAlgorithmConfigurator configurator) throws ClientException {
    if (cmd == null) {
      throw new IllegalArgumentException("O parmetro command est nulo.");
    }
    if (configurator == null) {
      throw new IllegalArgumentException("O parmetro configurator est nulo.");
    }
    this.nodeViewCache = new HashMap<Integer, NodeCommandView>();
    this.command = cmd;
    this.configurator = configurator;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void commandUpdated(final Type type, final CommandInfo newCmdInfo) {
    this.command = newCmdInfo;
    // atualiza as vises na cache
    this.detailsView.commandUpdated(type, newCmdInfo);
    for (NodeCommandView nodeView : nodeViewCache.values()) {
      nodeView.commandUpdated(type, newCmdInfo);
    }
    try {
      this.addCommandStatusDecoration(configuratorView.getGraph());
    }
    catch (ClientException e) {
      StatusBar statusBar = selectedView.getStatusBar();
      statusBar.setError(e.getMessage());
    }
  }

  /**
   * Adiciona as decoraes que representam o estado do comando s caixas do
   * fluxo.
   * 
   * @param graph O grafo que contm as caixas do fluxo.
   * 
   * @throws ClientException em caso de falha.
   */
  private void addCommandStatusDecoration(final Graph graph)
    throws ClientException {
    if (command != null) {
      if (command.getStatus() == CommandStatus.FINISHED) {
        addFinishedCommandStatusDecoration(graph);
      }
      else {
        addRunningCommandStatusDecoration(graph);
      }
    }
    else {
      String message =
        LNG.get(FlowCommandView.class.getSimpleName()
          + ".error.command.not.found");
      throw new ClientException(message);
    }
  }

  /**
   * Adiciona as decoraes que representam o estado do comando ainda no
   * finalizado s caixas do fluxo.
   * 
   * @param graph O grafo que contm as caixas do fluxo.
   */
  private void addRunningCommandStatusDecoration(final Graph graph) {
    switch (command.getStatus()) {
      case SYSTEM_FAILURE:
        addImageDecoration(graph, DecorationType.SYSTEM_FAILURE);
        break;
      case SCHEDULED:
        addImageDecoration(graph, DecorationType.SCHEDULED);
        break;
      case EXECUTING:
        if (command.getProgressData() != null) {
          addProgressDecoration(graph, command.getProgressDataMap());
        }
        else {
          addImageDecoration(graph, DecorationType.EXECUTION);
        }
        break;
      case INIT:
      case UPLOADING:
      case DOWNLOADING:
        addImageDecoration(graph, DecorationType.EXECUTION);
        break;
      default:
        addImageDecoration(graph, DecorationType.BLANK);
    }

  }

  /**
   * Adiciona uma decorao do tipo imagem a todos os ns do grafo do fluxo.
   * 
   * @param graph grafo do fluxo
   * @param decoration a decorao.
   */
  private void addImageDecoration(final Graph graph, DecorationType decoration) {
    for (GraphNode graphNode : graph.getNodeCollection()) {
      if (!graphNode.isBypassed()) {
        graphNode.addImageDecoration(decoration);
      }
    }
  }

  /**
   * Adiciona uma decorao textual informando o progresso de cada um dos ns do
   * grafo do fluxo.
   * 
   * @param graph o grafo do fluxo.
   * @param map mapa de progresso indexado pelo identificador do n.
   */
  private void addProgressDecoration(Graph graph, Map<Integer, ProgressData> map) {
    String key =
      FlowCommandView.class.getSimpleName() + ".tooltip.progress.description";
    for (GraphNode graphNode : graph.getNodeCollection()) {
      if (!graphNode.isBypassed()) {
        ProgressData progressData = map.get(graphNode.getId());
        if (progressData != null) {
          String description =
            LNG.get(key, new Object[] { graphNode.getAlgorithmName(),
                progressData.getDescription() });
          String formattedValue = progressData.getFormattedValue();
          graphNode.addTextDecoration(formattedValue, description);
        }
      }
    }
  }

  /**
   * Adiciona as decoraes que representam o estado do comando finalizado s
   * caixas do fluxo.
   * 
   * @param graph O grafo que contm as caixas do fluxo.
   */
  private void addFinishedCommandStatusDecoration(final Graph graph) {
    CommandFinalizationInfo info = command.getFinalizationInfo();
    if (info != null) {
      if (configurator.canBeRunAsSimpleCommand()) {
        GraphNode node = graph.getNodeCollection().iterator().next();
        decorateNode(node, info, false, false);
      }
      else if (info.getInfoType() == FinalizationInfoType.EXTENDED) {
        ExtendedCommandFinalizationInfo finalizationInfo =
          (ExtendedCommandFinalizationInfo) info;
        for (GraphNode node : graph.getNodeCollection()) {
          CommandFinalizationInfo nodeFinalizationInfo =
            finalizationInfo.getFinalizationInfoForNode(node.getId());
          Integer guiltyNodeId = finalizationInfo.getGuiltyNodeId();
          /*
           * Se existir o id do n "culpado" pelo erro, quer dizer que o comando
           * foi interrompido pelo flowmonitor. Neste caso, os algoritmos que
           * no retornaram cdigo de sada podem ser considerados como
           * "interrompidos" ao invs de "faltando cdigo de retorno".
           */
          boolean commandWasInterrupted = false;
          boolean isGuilty = false;
          if (guiltyNodeId != null) {
            commandWasInterrupted = true;
            if (guiltyNodeId == node.getId()) {
              isGuilty = true;
            }
          }
          decorateNode(node, nodeFinalizationInfo, commandWasInterrupted,
            isGuilty);
        }
      }
    }
  }

  /**
   * Adiciona a decorao que representa o estado de uma caixa do fluxo a partir
   * do estado de finalizao do algoritmo.
   * 
   * @param node A caixa a ser decorada.
   * @param finalizationInfo O estado de finalizao da caixa.
   * @param commandWasInterrupted Indica se a execuo do comando foi
   *        interrompida.
   * @param isGuilty Indica se o n foi o responsvel pela interrupo do fluxo.
   */
  private void decorateNode(GraphNode node,
    CommandFinalizationInfo finalizationInfo, boolean commandWasInterrupted,
    boolean isGuilty) {
    DecorationType decoration = DecorationType.BLANK;
    if (!node.isBypassed()) {
      if (commandWasInterrupted && isGuilty) {
        decoration = DecorationType.FATAL_ERROR;
      }
      else {
        switch (finalizationInfo.getFinalizationType()) {
          case EXECUTION_ERROR:
            if (commandWasInterrupted) {
              decoration = DecorationType.NON_FATAL_ERROR;
            }
            else {
              decoration = DecorationType.EXECUTION_ERROR;
            }
            break;
          case NO_EXIT_CODE:
            if (commandWasInterrupted) {
              decoration = DecorationType.INTERRUPTED;
            }
            else {
              decoration = DecorationType.EXECUTION_UNKNOWN;
            }
            break;
          case SUCCESS:
            if (finalizationInfo.hasWarnings()) {
              decoration = DecorationType.EXECUTION_WARNING;
            }
            else {
              decoration = DecorationType.EXECUTION_SUCCESS;
            }
            break;
          default:
            decoration = DecorationType.BLANK;
            break;
        }
      }
    }
    node.addImageDecoration(decoration);
  }

  /**
   * Cria as aes vinculadas  visualizao das "caixinhas" do fluxo.
   * 
   * @param workspace rea de visualizao das "caixinhas" do fluxo.
   */
  private void createFilters(final Workspace workspace) {

    // Cria o menu popup.
    new AddNodePopupActionFilter(workspace) {
      @Override
      protected Action createAction(GraphNode graphNode, Point2D point) {
        if (graphNode != null && !graphNode.isBypassed()) {
          return new ShowAlgorithmCommandViewPopupAction(graphNode);
        }
        else {
          return null;
        }
      }
    }.attach();

    new PopupFilter(workspace).attach();

    // Cria a ao que ser chamada quando houver um clique em algum n.
    new ShowAlgorithmDetailsClickAction(workspace).attach();
  }

  /**
   * Ao de que exibe a viso de um n do fluxo no painel de detalhes.
   */
  private class ShowAlgorithmDetailsClickAction extends WorkspaceFilter {

    /**
     * Construtor.
     * 
     * @param workspace rea de visualizao das "caixinhas" do fluxo.
     */
    public ShowAlgorithmDetailsClickAction(final Workspace workspace) {
      super(workspace);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void callbackButton(Point2D pt, MouseEvent ev) {
      if (ev.getButton() == MouseEvent.BUTTON1
        && ev.getID() == MouseEvent.MOUSE_CLICKED) {
        PickNodeMessage pickNodeMessage = new PickNodeMessage(pt);
        pickNodeMessage.sendVO(this);
        GraphNode node = pickNodeMessage.getNode();
        if (node == selectedNode || (node != null && node.isBypassed())) {
          return;
        }

        try {
          if (selectedNode != null) {
            selectedNode.setSelected(false);
            selectedNode.repaint();
          }
          selectedNode = node;

          if (node != null) {
            NodeCommandView view = nodeViewCache.get(node.getId());
            if (view == null) {
              // S cria a viso se j no estiver na cache.
              AlgorithmConfiguratorView algorithmConfiguratorView =
                node.getAlgorithmConfiguratorView();
              if (algorithmConfiguratorView != null) {
                view =
                  new NodeCommandView(node.getId(), command,
                    algorithmConfiguratorView.getConfigurator());
                nodeViewCache.put(node.getId(), view);
              }
              else {
                showErrorView(node);
              }
            }
            if (view != null) {
              node.setSelected(true);
              selectedNode.repaint();
              setSelectedView(view);
            }
          }
          else {
            setSelectedView(detailsView);
          }
        }
        catch (Exception ex) {
          StandardErrorDialogs.showErrorDialog(owner, getTitle(), ex);
        }
      }
      super.callbackButton(pt, ev);
    }

    /**
     * Mostra a viso de erro.
     * 
     * @param node o n selecionado quando o erro ocorreu,
     * @throws ClientException em caso de erro ao selecionar a viso.
     */
    private void showErrorView(GraphNode node) throws ClientException {
      String errorMessage =
        LNG.get(FlowCommandView.class.getSimpleName() + ".error.loading.view",
          new Object[] { node.getAlgorithmName() });
      CommandView errorView = new ErrorView(errorMessage);
      setSelectedView(errorView);
    }
  }

  /**
   * Ao de menu popup que exibe a viso de um n do fluxo em uma nova janela.
   */
  private final class ShowAlgorithmCommandViewPopupAction extends
    GraphElementAction {

    /**
     * Cria a ao.
     * 
     * @param node O n (No aceita {@code null}).
     */
    public ShowAlgorithmCommandViewPopupAction(final GraphNode node) {
      super(node, FlowCommandView.class.getSimpleName() + ".action.show.node",
        ApplicationImages.ICON_INFORMATION_16);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      GraphNode node = (GraphNode) getElement();
      if (node != null && !node.isBypassed()) {
        try {
          Tab selected = selectedView.getSelected();
          TabType type;
          if (selected != null) {
            type = selected.getType();
          }
          else {
            type = TabType.STATIC;
          }
          CommandViewFactory.showNodeView(command, owner, type, node);
        }
        catch (Exception ex) {
          StandardErrorDialogs.showErrorDialog(owner, getTitle(), ex);
        }
      }
    }
  }

  /**
   * Determina a viso a ser mostrada no painel de detalhes do fluxo.
   * 
   * @param view a visualizao a ser mostrado.
   * 
   * @throws ClientException em caso de erro na criao da viso do n.
   */
  private void setSelectedView(CommandView view) throws ClientException {
    if (mainComponent == null) {
      mainComponent = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
      mainComponent.setOneTouchExpandable(true);
      AlgorithmConfiguratorFactory configuratorFactory =
        AlgorithmConfiguratorFactory.getInstance();
      configuratorView =
        (FlowAlgorithmConfiguratorView) configuratorFactory.createReportView(
          owner, configurator);
      createFilters(configuratorView.getWorkspace());
      addCommandStatusDecoration(configuratorView.getGraph());
      mainComponent.setLeftComponent(configuratorView.getMainComponent());
    }
    String title = null;
    TabType type = null;
    Integer dividerLocation = null;
    if (selectedView != null) {
      Tab selected = selectedView.getSelected();
      if (selected != null) {
        selected.setSelected(false);
        title = selected.getTitle();
        type = selected.getType();
      }
      dividerLocation = mainComponent.getDividerLocation();
    }
    selectedView = view;
    owner.setTitle(getTitle());
    Component selectedComponent = selectedView.getMainComponent(this.owner);

    if (title != null) {
      selectedView.selectTab(title);
    }
    else if (type != null) {
      selectedView.selectTab(type);
    }
    else {
      selectedView.selectPreferredTab();
    }

    mainComponent.setRightComponent(selectedComponent);
    if (dividerLocation != null) {
      mainComponent.setDividerLocation(dividerLocation);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getMainComponent(DesktopComponentFrame window)
    throws ClientException {
    if (mainComponent == null) {
      this.owner = window;
      detailsView = new FlowDetailsView(command, configurator);
      setSelectedView(detailsView);
    }
    return mainComponent;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTitle() {
    if (selectedNode != null) {
      return LNG.get(FlowCommandView.class.getSimpleName()
        + ".window.title.node",
        new Object[] { command.getId(), selectedNode.getAlgorithmName(),
            selectedNode.getAlgorithmVersionId() });
    }
    else {
      return LNG.get(FlowCommandView.class.getSimpleName() + ".window.title",
        new Object[] { command.getId() });
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void selectTab(TabType tabType) throws ClientException {
    selectedView.selectTab(tabType);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void selectTab(String title) throws ClientException {
    selectedView.selectTab(title);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Tab getSelected() {
    return selectedView.getSelected();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public StatusBar getStatusBar() {
    return selectedView.getStatusBar();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void selectPreferredTab() throws ClientException {
    selectedView.selectPreferredTab();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() {
    this.detailsView.close();
    for (NodeCommandView nodeView : nodeViewCache.values()) {
      nodeView.close();
    }

  }

}
