package csbase.client.applications.flowapplication.multiflow;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;

import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.Application;
import csbase.client.applications.flowapplication.multiflow.tree.AlgorithmNode;
import csbase.client.applications.flowapplication.multiflow.tree.ParameterNode;
import csbase.client.applications.flowapplication.multiflow.tree.ParameterSelection;
import csbase.client.applications.flowapplication.multiflow.tree.ParameterSelectionModel;
import csbase.client.applications.flowapplication.multiflow.tree.ParameterSelectionNode;
import csbase.client.applications.flowapplication.multiflow.tree.ParameterSelectionTree;
import csbase.client.applications.flowapplication.multiflow.tree.RootNode;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.FlowLink;
import csbase.logic.algorithms.flows.FlowNode;
import csbase.logic.algorithms.flows.LinkParameter;
import csbase.logic.algorithms.flows.NodeParameter;

/**
 * Dilogo de seleo de parmetros para gerao de configurao de execuo
 * mltipla de fluxos.
 */
public class ParameterSelectionDialog extends
  AbstractMultipleFlowConfigurationDialog {

  /**
   * A rvore de seleo de parmetros.
   */
  private ParameterSelectionTree parameterSelectionTree;
  /**
   * Caixa de seleo que indica se os parmetros sem valor devem ser
   * selecioandos.
   */
  private JCheckBox selectEmptyParametersCheckBox;

  /**
   * Construtor.
   * 
   * @param application a aplicao dona da janela.
   * @param flow o fluxo para seleo de parmetros.
   */
  public ParameterSelectionDialog(Application application, Flow flow) {
    super(application);
    parameterSelectionTree = createTree(flow);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createMainComponent() {
    JPanel panel = new JPanel(new GridBagLayout());
    GBC gbc = new GBC().insets(5);
    JScrollPane treePane = new JScrollPane(parameterSelectionTree);
    panel.add(treePane, new GBC(gbc).both());
    selectEmptyParametersCheckBox =
      new JCheckBox(
        getString("ParameterSelectionDialog.selectEmptyParamsLabel"));
    selectEmptyParametersCheckBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        boolean selected = selectEmptyParametersCheckBox.isSelected();
        changeEmptyParametersSelection(selected);
      }
    });
    panel.add(selectEmptyParametersCheckBox, new GBC(gbc).gridy(1).west());
    return panel;
  }

  /**
   * Cria a rvore de seleo de parmetros a partir do fluxo especificado.
   * 
   * @param flow o fluxo.
   * @return a rvore j com os ns preenchidos.
   */
  private ParameterSelectionTree createTree(Flow flow) {
    RootNode root = new RootNode(flow);
    Set<FlowNode> nodes = flow.getNodes();
    for (FlowNode flowNode : nodes) {
      Set<NodeParameter> parameters = flowNode.getParameters();
      AlgorithmNode treeNode = new AlgorithmNode(flowNode);
      root.add(treeNode);
      for (NodeParameter parameter : parameters) {
        if (!isLinked(flow, parameter, flowNode.getId())) {
          ParameterNode parameterNode = new ParameterNode(parameter, flowNode);
          treeNode.add(parameterNode);
        }
      }
    }
    ParameterSelectionTree tree = new ParameterSelectionTree(root);
    return tree;
  }

  /**
   * Determina se um parmetro de um n est sendo utilizado como ligao no
   * fluxo.
   * 
   * @param flow o fluxo ao qual o parmetro faz parte.
   * @param parameter o parmetro.
   * @param nodeId o identificador do n do parmetro.
   * @return verdadeiro se o parmetro est numa ligao do fluxo ou falso, caso
   *         contrrio.
   */
  private boolean isLinked(Flow flow, NodeParameter parameter, int nodeId) {
    Set<FlowLink> links = flow.getLinks();
    for (FlowLink link : links) {
      LinkParameter input = link.getInput();
      LinkParameter output = link.getOutput();
      String name = parameter.getName();
      if ((name.equals(input.getName()) && nodeId == input.getNodeId())
        || (name.equals(output.getName())) && nodeId == output.getNodeId()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Obtm a lista de parmetros selecionados na rvore.
   * 
   * @return a lista de parmetros.
   */
  public List<ParameterSelection> getParameterSelection() {
    ParameterSelectionModel parameterSelectionModel =
      parameterSelectionTree.getParameterSelectionModel();
    TreePath[] paths = parameterSelectionModel.getSelectionPaths();
    List<ParameterSelection> selectedParams =
      new ArrayList<ParameterSelection>();
    if (paths != null) {
      for (TreePath path : paths) {
        Object node = path.getLastPathComponent();
        if (node instanceof ParameterSelectionNode) {
          ParameterSelectionNode selection = (ParameterSelectionNode) node;
          selectedParams.addAll(selection.getParameterSelection());
        }
      }
    }
    return selectedParams;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean validateFields() {
    ParameterSelectionModel parameterSelectionModel =
      parameterSelectionTree.getParameterSelectionModel();
    TreePath[] paths = parameterSelectionModel.getSelectionPaths();
    if (paths != null) {
      return true;
    }
    else {
      Application application = getApplication();
      application.showError(this,
        getString("ParameterSelectionDialog.emptySelectionMessage"));
      return false;
    }
  }

  /**
   * Muda o estado da seleo dos parmetros da rvore que no tm valor
   * definido.
   * 
   * @param select verdadeiro se os parmetros devem ser selecionados ou falso,
   *        caso contrrio.
   */
  private void changeEmptyParametersSelection(boolean select) {
    ParameterSelectionModel parameterSelectionModel =
      parameterSelectionTree.getParameterSelectionModel();
    TreeModel model = parameterSelectionTree.getModel();
    RootNode root = (RootNode) model.getRoot();
    Enumeration<?> enumeration = root.depthFirstEnumeration();
    while (enumeration.hasMoreElements()) {
      Object element = enumeration.nextElement();
      if (element instanceof ParameterNode) {
        ParameterNode node = (ParameterNode) element;
        ParameterSelection selection = node.getUserObject();
        NodeParameter parameter = selection.parameter;
        if (parameter.getValue() == null) {
          TreePath path = new TreePath(node.getPath());
          if (select) {
            parameterSelectionModel.addSelectionPath(path);
          }
          else {
            parameterSelectionModel.removeSelectionPath(path);
          }
          parameterSelectionTree.expandPath(path.getParentPath());
        }
      }
    }
  }
}
