/*
 * PlatformInfoDialog.java
 * 
 * $Author: isabella $ $Revision: 175530 $ - $Date: 2010-07-06 16:06:51 -0300
 * (Tue, 06 Jul 2010) $
 */
package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.text.MessageFormat;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.remote.srvproxies.PlatformProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.Platform;
import csbase.logic.PlatformInfo;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * A classe {@code PlatformInfoDialog}  responsvel por incluir ou
 * modificar as informaes de uma plataforma de execuo atravs de uma
 * interface especfica.
 * 
 * @author Anselmo e Ana
 */
public class PlatformInfoDialog {
  /** Janela para cadastro dos dados da plataforma */
  private DesktopComponentDialog window;
  /** Janela "pai" */
  private Window owner;
  /** Campo nome da plataforma */
  private JTextField name;
  /** Campo descrio da plataforma */
  private JTextField description;
  /** Boto de alterao da plataforma */
  private JButton modifyPlatformButton;
  /** Ttulo da janela */
  private String windowLabel;
  /** Objeto para persistncia dos dados de plataforma entrados pelo usurio */
  private Platform platform;

  /**
   * Cria um painel contendo o boto limpar.
   * 
   * @return o painel criado
   */
  private JPanel makeClearButtonPanel() {
    JPanel clearPanel = new JPanel();
    JButton clearButton = new JButton(LNG.get("IAS_CLEAR"));
    clearButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        clearFields();
      }
    });
    clearPanel.add(clearButton);
    return clearPanel;
  }

  /**
   * Cria e mostra a janela de incluso/modificao da plataforma
   */
  protected void display() {
    // Cria o painel com os campos e o boto limpar.
    JPanel mainPane = new JPanel(new BorderLayout());
    mainPane.setBorder(BorderFactory.createTitledBorder(LNG
      .get("IAS_PLATFORM_REGISTRATION")));
    mainPane.add(makeMainPanel(), BorderLayout.NORTH);
    mainPane.add(makeClearButtonPanel(), BorderLayout.SOUTH);
    // Cria o painel de fundo
    JPanel backPane = new JPanel(new BorderLayout());
    backPane.add(mainPane);
    backPane.add(makeButtonPanel(), BorderLayout.SOUTH);
    // Cria a janela    
    window = new DesktopComponentDialog(owner, windowLabel);
    window.getContentPane().add(backPane);
    window.setDefaultCloseOperation(DesktopComponentDialog.DISPOSE_ON_CLOSE);
    window.pack();
    window.center(owner);
    window.setVisible(true);
  }

  /**
   * Inicializa os campos da janela com os dados da plataforma a ser modificada.
   */
  protected void initFields() {
    name.setText(platform.getName());
    name.setEditable(false);
    description.setText(platform.getDescription());
    ChangeListener changeList = new ChangeListener();
    description.addKeyListener(changeList);
  }

  /**
   * Constri o painel para preenchimento ou alterao dos dados da plataforma
   * 
   * @return o painel principal da janela
   */
  protected JPanel makeMainPanel() {
    name = new JTextField(20);
    JLabel nameL = new JLabel(LNG.get("IAS_PLATFORM"));
    description = new JTextField();
    JLabel descriptionL = new JLabel(LNG.get("IAS_PLATFORM_DESCRIPTION"));
    if (platform != null) {
      initFields();
    }
    GridLayout g = new GridLayout(0, 1);
    g.setVgap(10);
    JPanel textP = new JPanel();
    JPanel labels = new JPanel(g);
    JPanel texts = new JPanel(new GridLayout(0, 1));
    labels.add(nameL);
    labels.add(descriptionL);
    texts.add(name);
    texts.add(description);
    textP.add(labels);
    textP.add(texts);
    return textP;
  }

  /**
   * Cria o boto de modificao de Plataforma Esse boto somente  habilitado
   * quando alguma modificao  feita nos campos.
   * 
   * @return o boto de modificao de dados da Plataforma
   */
  protected JButton makeChangeButton() {
    modifyPlatformButton = new JButton(LNG.get("IAS_PLATFORM_UPDATE"));
    modifyPlatformButton.setEnabled(false);
    modifyPlatformButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        if (validateFields()) {
          modifyPlatform();
          window.close();
        }
        else {
          StandardErrorDialogs.showErrorDialog(window, windowLabel,
            LNG.get("IAS_FILL_ALL_FIELDS"));
          modifyPlatformButton.setEnabled(false);
        }
      }
    });
    return modifyPlatformButton;
  }

  /**
   * Cria o boto de incluso de plataforma
   * 
   * @return o boto de incluso
   */
  protected JButton makeAddPlatformButton() {
    JButton addPlatformButton = new JButton(LNG.get("IAS_PLATFORM_INCLUDE"));
    addPlatformButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        if (validateFields()) {
          createPlatform();
        }
        else {
          StandardErrorDialogs.showErrorDialog(window, windowLabel,
            LNG.get("IAS_FILL_ALL_FIELDS"));
        }
      }
    });
    return addPlatformButton;
  }

  /**
   * Cria o boto de cancelamento de operao.
   * 
   * @return o boto de cancelamento.
   */
  protected JButton makeCancelButton() {
    JButton cancelButton = new JButton(LNG.get("IAS_CLOSE"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        window.close();
      }
    });
    return cancelButton;
  }

  /**
   * Cria o boto de fechamento da janela.
   * 
   * @return o boto de fechamento.
   */
  protected JButton makeCloseButton() {
    JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        // Se os campos nao forem alterados ou estiverem em branco, fecha
        // a janela.
        String nameText = name.getText().trim();
        String descText = description.getText().trim();
        if ((nameText.equals(platform.getName()) && descText.equals(platform
          .getDescription())) || ((nameText.equals("")) && descText.equals(""))) {
          window.close();
          return;
        }
        Object[] options = { LNG.get("IAS_UPDATE"), LNG.get("IAS_CANCEL") };
        int selected =
          JOptionPane.showOptionDialog(window,
            LNG.get("IAS_PLATFORM_UPDATE_CONFIRMATION"), windowLabel,
            JOptionPane.DEFAULT_OPTION, JOptionPane.INFORMATION_MESSAGE, null,
            options, options[0]);
        if (options[selected] == LNG.get("IAS_UPDATE") && validateFields()) {
          modifyPlatform();
        }
        window.close();
      }
    });
    return closeButton;
  }

  /**
   * Constri o painel com os botes da janela de incluso/ modificao de
   * plataformas. As aes previstas so para confirmar a operao solicitada ou
   * desistir da operao.
   * 
   * @return o painel de botes
   */
  protected JPanel makeButtonPanel() {
    JPanel buttonPanel = new JPanel();
    if (platform != null) {
      buttonPanel.add(makeChangeButton());
      buttonPanel.add(makeCloseButton());
    }
    else {
      buttonPanel.add(makeAddPlatformButton());
      buttonPanel.add(makeCancelButton());
    }
    return buttonPanel;
  }

  /**
   * Verifica se o campos de nome est preenchido (o campo de descrio no 
   * obrigatrio).
   * 
   * @return true se os campo estiver preenchido ou false se estiver vazio.
   */
  protected boolean validateFields() {
    String nameText = name.getText().trim();
    if ((!nameText.equals(""))) {
      return true;
    }
    return false;
  }

  /**
   * Executa a ao de criar a plataforma, usando os dados preenchidos nos
   * campos da janela.
   */
  private void createPlatform() {
    String nameText = name.getText().trim();
    String descriptionText = description.getText().trim();
    PlatformInfo platformInfo = new PlatformInfo(nameText, descriptionText);
    platform =
      PlatformProxy.createPlatform(window, windowLabel,
        LNG.get("IAS_WAITING_PLATFORM_CREATION"), platformInfo);
    if (platform != null) {
      StandardDialogs.showInfoDialog(window, windowLabel, MessageFormat.format(
        LNG.get("IAS_PLATFORM_INCLUDED_WITH_SUCCESS"),
        new Object[] { platform.getName() }));
      platform = null;
      clearFields();
    }
  }

  /**
   * Executa a ao de modificar a plataforma, usando os dados preenchidos nos
   * campos da janela.
   */
  private void modifyPlatform() {
    String nameText = name.getText().trim();
    String descText = description.getText().trim();
    PlatformInfo platformInfo = new PlatformInfo(nameText, descText);
    Platform modifiedPlatform =
      PlatformProxy.modifyPlatform(window, windowLabel,
        LNG.get("IAS_WAITING_PLATFORM_MODIFICATION"), platform.getId(),
        platformInfo);
    if (modifiedPlatform != null) {
      this.platform = modifiedPlatform;
      StandardDialogs.showInfoDialog(window, windowLabel, MessageFormat.format(
        LNG.get("IAS_PLATFORM_MODIFIED_WITH_SUCCESS"),
        new Object[] { platform.getName() }));
      modifyPlatformButton.setEnabled(false);
    }
  }

  /**
   * Limpa os campos de nome e descrio.
   */
  private void clearFields() {
    if ((name == null) || (description == null)) {
      return;
    }
    if (platform == null) {
      name.setText("");
      name.requestFocus();
    }
    else {
      description.requestFocus();
    }
    description.setText("");
  }

  /**
   * Classe que implementa o listener que habilita ou no boto de modificao.
   */
  private class ChangeListener extends KeyAdapter {
    /*
     * (non-Javadoc)
     * 
     * @see java.awt.event.KeyAdapter#keyReleased(java.awt.event.KeyEvent)
     */
    @Override
    public void keyReleased(KeyEvent kev) {
      checkChange();
    }

    /**
     * Verifica se o usurio efetuou alteraes nos dados da platforma. Em caso
     * positivo, habilita o boto de alteraes. Caso contrrio, este 
     * desabilitado.
     */
    private void checkChange() {
      if (name.getText().trim().equals(platform.getName())
        && description.getText().trim().equals(platform.getDescription())) {
        modifyPlatformButton.setEnabled(false);
      }
      else {
        modifyPlatformButton.setEnabled(true);
      }
    }
  }

  /**
   * Cria uma janela para incluso de dados de uma nova plataforma
   * 
   * @param owner Janela "pai".
   */
  public PlatformInfoDialog(Window owner) {
    this(owner, null);
  }

  /**
   * Cria uma janela para modificao de dados de uma plataforma j cadastrada
   * no sistema, ou de incluso de dados, caso a plataforma passada seja null.
   * 
   * @param owner Janela "pai".
   * @param platform a plataforma que vai ser modificada
   */
  public PlatformInfoDialog(Window owner, Platform platform) {
    this.owner = owner;
    if (platform == null) {
      windowLabel = LNG.get("IAS_PLATFORM_INCLUSION_TITLE");
    }
    else {
      windowLabel = LNG.get("IAS_PLATFORM_UPDATE_TITLE");
      this.platform = platform;
    }
    display();
  }
}
