package csbase.client.ias;

import java.awt.GridBagLayout;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JLabel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.ias.PermissionInfoDialog.FieldsState;
import csbase.logic.Permission;
import csbase.logic.UserPasswordPermission;

/**
 * Painel que contm o grupo de componentes de usurio/senha, somente visveis
 * quando uma classe filha da classe genrica de usurio/senha estiver
 * selecionada.
 */
class UserPasswordPanel extends AbstractPermissionPanel {

  /** Label login (opcional) */
  private JLabel loginL;

  /** Campo login (opcional) */
  private JTextField login;

  /** Label senha (opcional) */
  private JLabel passwordL;

  /** Campo senha (opcional) */
  private JPasswordField password;

  /** Label confirmao de senha (opcional) */
  private JLabel confirmPasswordL;

  /** Campo confirmao de senha (opcional) */
  private JPasswordField confirmPassword;

  /** Indica se uma nova permisso est sendo criada */
  private boolean isNew = true;

  /**
   * Construtor.
   * 
   * @param dialog Dilogo no qual este painel est contido.
   */
  UserPasswordPanel(PermissionInfoDialog dialog) {
    super(new GridBagLayout(), dialog);
    // Cria os componentes
    loginL = new JLabel(LNG.get("IAS_USER") + ":");
    login = new JTextField(20);
    passwordL = new JLabel(LNG.get("IAS_PERMISSION_PASSWORD") + ":");
    password = new JPasswordField(20);
    confirmPasswordL =
      new JLabel(LNG.get("IAS_CONFIRM_PERMISSION_PASSWORD") + ":");
    confirmPassword = new JPasswordField(20);

    // Label usurio
    GBC gbc = new GBC(0, 0).northwest().insets(5, 10, 10, 5);
    add(loginL, gbc);

    // Campo usurio
    gbc = new GBC(1, 0).horizontal().northwest().insets(5, 0, 5, 10).filly();
    add(login, gbc);

    // Label senha
    gbc = new GBC(2, 0).northwest().insets(5, 0, 10, 10);
    add(passwordL, gbc);

    // Campo senha
    gbc = new GBC(3, 0).horizontal().northwest().insets(5, 0, 5, 10).filly();
    add(password, gbc);

    // Label confirmao de senha
    gbc = new GBC(2, 1).northwest().insets(5, 0, 15, 5);
    add(confirmPasswordL, gbc);

    // Campo confirmao de senha
    gbc = new GBC(3, 1).horizontal().northwest().insets(5, 0, 10, 10).filly();
    add(confirmPassword, gbc);

  }

  /**
   * {@inheritDoc}
   */
  @Override
  void importFrom(Permission permission) {
    isNew = false;
    UserPasswordPermission userPwdPermission =
      (UserPasswordPermission) permission;
    login.setText(userPwdPermission.getUser());

    KeyListener keyChangeListener = new KeyAdapter() {
      @Override
      public void keyReleased(KeyEvent e) {
        dialog.checkDataChange();
      }
    };

    login.addKeyListener(keyChangeListener);
    password.addKeyListener(keyChangeListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void exportTo(Permission permission) {
    UserPasswordPermission userPwdPermission =
      (UserPasswordPermission) permission;
    userPwdPermission.setUser(getLogin());
    userPwdPermission.setPassword(new String(getPassword()));
  }

  /**
   * Retorna o login digitado.
   * 
   * @return login
   */
  public String getLogin() {
    return login.getText().trim();
  }

  /**
   * Retorna a senha digitada.
   * 
   * @return password
   */
  public char[] getPassword() {
    return password.getPassword();
  }

  /**
   * Retorna a confirmao da senha.
   * 
   * @return confirmPassword
   */
  public char[] getConfirmPassword() {
    return confirmPassword.getPassword();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void clearFields() {
    login.setText("");
    password.setText("");
    confirmPassword.setText("");
  }

  /**
   * Limpa os campos de senha.
   */
  void clearPasswordFields() {
    password.setText("");
    confirmPassword.setText("");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FieldsState validateFields() {
    String loginText = getLogin();
    if (loginText.equals("")) {
      return FieldsState.EMPTY_FIELDS;
    }
    char[] passwordChars = getPassword();
    String passwordText = new String(passwordChars).trim();
    char[] confirmPasswordChars = getConfirmPassword();
    String confirmPasswordText = new String(confirmPasswordChars).trim();
    if (!isNew && (passwordText.equals("") != confirmPasswordText.equals(""))) {
      return FieldsState.EMPTY_FIELDS;
    }
    if (isNew && (passwordText.equals("") || confirmPasswordText.equals(""))) {
      return FieldsState.EMPTY_FIELDS;
    }
    if (!passwordText.equals(confirmPasswordText)) {
      return FieldsState.PASSWORDS_DONT_MATCH;
    }
    return FieldsState.FIELDS_OK;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  boolean hasChanged(Permission permission) {
    UserPasswordPermission userPwdPermission =
      (UserPasswordPermission) permission;
    if (!getLogin().equals(userPwdPermission.getUser())) {
      return true;
    }
    String passwordText = new String(getPassword()).trim();
    if (!passwordText.equals("")) {
      return true;
    }
    return false;
  }
}
