package csbase.client.rest.resources.v1.project;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import csbase.client.desktop.DesktopFrame;
import csbase.client.rest.resources.v1.application.ApplicationRestService;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import io.jsonwebtoken.impl.Base64Codec;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import io.swagger.annotations.Authorization;

/**
 * Responsvel pelo atendimento REST de projeto do desktop
 *
 * @author Tecgraf/PUC-Rio
 */
@Path("v1/project")
@Api(value = "Project", authorizations = {
    @Authorization(value = "Bearer", scopes = {}) })
public class ProjectRestService {

  /**
   * Codificador Base64
   */
  final private Base64Codec codec = new Base64Codec();

  /**
   * Encapsulador de dono do projeto (em conformidade com as informaes
   * oferecidas na parte servidor.
   *
   * @author Tecgraf/PUC-Rio
   */
  public class OwnerWrapper {
    /**
     * Login
     */
    final private String login;

    /**
     * Id CSBASE
     */
    final private String id;

    /**
     * Nome
     */
    final private String name;

    /**
     * Consulta o login
     *
     * @return login
     */
    public String getLogin() {
      return login;
    }

    /**
     * Consulta o identificador
     *
     * @return id
     */
    public String getId() {
      return id;
    }

    /**
     * Consulta o nome.
     *
     * @return nome
     */
    public String getName() {
      return name;
    }

    /**
     * Construtor
     *
     * @param user usurio dono do projeto.
     */
    public OwnerWrapper(User user) {
      if (user == null) {
        this.id = "--";
        this.name = "--";
        this.login = "--";
      }
      else {
        this.id = user.getId().toString();
        this.name = user.getName();
        this.login = user.getLogin();
      }
    }
  }

  /**
   * Encapsulador do projeto corrente.
   *
   * @author Tecgraf/PUC-Rio
   */
  public class ProjectWapper {
    /**
     * Identificador Base64
     */
    private final String id;

    /**
     * Nome
     */
    private final String name;

    /**
     * Dono do projeto.
     */
    private final OwnerWrapper owner;

    /**
     * Construtor.
     *
     * @param project projeto
     */
    ProjectWapper(CommonClientProject project) {
      this.id = codec.encode(project.getId().toString());
      this.name = project.getName();
      User user;
      try {
        user = User.getUser(project.getUserId());
      }
      catch (Exception ex) {
        user = null;
      }
      this.owner = new OwnerWrapper(user);
    }

    /**
     * Consulta o identificador base 64.
     *
     * @return id (base64)
     */
    public String getId() {
      return id;
    }

    /**
     * Consulta o nome.
     *
     * @return nome
     */
    public String getName() {
      return name;
    }

    /**
     * Consulta o dono
     *
     * @return dono
     */
    public OwnerWrapper getOwner() {
      return owner;
    }

  };

  /**
   * Processamento do GET
   *
   * @return a resposta HTTP
   */
  @GET
  @Produces(MediaType.APPLICATION_JSON)
  @ApiOperation(value = "Get the current project", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation", response = ProjectWapper.class),
      @ApiResponse(code = 404, message = "No project is open.", response = ApplicationRestService.RESTError.class), })
  public Response getCurrentProject() {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    CommonClientProject project = desktop.getProject();
    if (project == null) {
      final String jsonMsg = "{\"error\":\"No project found.\"}";
      return Response.status(Status.NOT_FOUND).entity(jsonMsg).build();
    }
    final ProjectWapper projWrapper = new ProjectWapper(project);
    return Response.status(Status.OK).entity(projWrapper).build();
  }

  /**
   * Processamento do POST
   *
   * @param projectId identificador base64 do projeto.
   * @return response
   */
  @POST
  @Path("{projectId}")
  @Produces(MediaType.APPLICATION_JSON)
  @ApiOperation(value = "Open a project", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation", response = ProjectWapper.class),
      @ApiResponse(code = 403, message = "Failure while opening the required project", response = ApplicationRestService.RESTError.class),
      @ApiResponse(code = 404, message = "The required project cannot be opened.", response = ApplicationRestService.RESTError.class), })
  public Response setCurrentProject(
    @ApiParam(value = "The project Id", required = true) @PathParam("projectId") String projectId) {
    final String decodedProjectId = codec.decodeToString(projectId);
    final DesktopFrame desktop = DesktopFrame.getInstance();
    if (!desktop.checkOpenableProject(decodedProjectId)) {
      final String fmt = "{\"error\":\"Project %s cannot be opened.\"}";
      final String jsonMsg = String.format(fmt, decodedProjectId);
      return Response.status(Status.NOT_FOUND).entity(jsonMsg).build();
    }
    final CommonClientProject project = desktop.openProject(decodedProjectId);
    if (project == null) {
      final String fmt = "{\"error\":\"Project %s openning failure.\"}";
      final String jsonMsg = String.format(fmt, decodedProjectId);
      return Response.status(Status.FORBIDDEN).entity(jsonMsg).build();
    }
    final ProjectWapper projWrapper = new ProjectWapper(project);
    return Response.status(Status.OK).entity(projWrapper).build();
  }

  /**
   * Processamento do DELETE
   *
   * @return response
   */
  @DELETE
  @Produces(MediaType.APPLICATION_JSON)
  @ApiOperation(value = "Closes the current project", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation", response = ProjectWapper.class),
      @ApiResponse(code = 500, message = "Error while closing the current project.", response = ProjectWapper.class), })
  public Response resetCurrentProject() {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    CommonClientProject project = desktop.getProject();
    if (project == null) {
      final String jsonMsg = "{\"warning\":\"No previous project set.\"}";
      return Response.status(Status.OK).entity(jsonMsg).build();
    }
    if (!desktop.closeProject()) {
      final ProjectWapper projWrapper = new ProjectWapper(project);
      return Response.status(Status.INTERNAL_SERVER_ERROR).entity(projWrapper)
        .build();
    }
    return Response.status(Status.OK).build();
  }
}
