package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.EventObject;
import java.util.List;

import javax.swing.DefaultCellEditor;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.event.CellEditorListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.text.Document;
import javax.swing.tree.DefaultTreeCellEditor;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreeCellEditor;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import csbase.client.applications.Application;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.versiontree.AbstractFileInfoNode;
import csbase.client.applications.algorithmsmanager.versiontree.ConfigurationFileNode;
import csbase.client.applications.algorithmsmanager.versiontree.DocumentationFileNode;
import csbase.client.applications.algorithmsmanager.versiontree.ExecutableFileNode;
import csbase.client.applications.algorithmsmanager.versiontree.ReleaseNotesFileNode;
import csbase.client.applications.algorithmsmanager.versiontree.VersionNodeInterface;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTree;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTreeNode;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTreeRootNode;
import csbase.client.applications.algorithmsmanager.versiontree.actions.CommonAlgoFileRenameAction;
import csbase.logic.FileInfo;
import csbase.logic.algorithms.AlgorithmInfo;
import tecgraf.javautils.gui.GBC;

/**
 * Representa uma viso da rvore de verses de um algoritmo e seus respectivos
 * filhos. Para inserir essa viso grfica voc deve obter o painel
 * correspondente.
 * 
 */
public class VersionTreeView {

  /** Painel que criou essa viso de rvore de verses */
  private AlgorithmVersionInfoPanel versionInfoPanel;

  /** rvore com as verses do algoritmo */
  private VersionTree versionTree;

  /** rvore de dados, que pode exibir ou no os filhos dos ns principais */
  private JPanel versionTreePanel;

  /**
   * O editor de ns. Utilizado para renomear os arquivos representados pelos
   * ns da rvore.
   */
  private VersionTreeCellEditor editor;

  /**
   * Indica que as informaes j foram atualizadas. Foi necessrio para tratar
   * os casos de seleo de um item da JList em que o evento de mudana chega,
   * mas  somente para o item aparecer selecionado e no precisa inicializar
   * esse elemento
   */
  private boolean infoAlreadyUpdated;

  /**
   * Indica se a edio de nomes de arquivos est habilitada.
   */
  protected boolean renameEnabled;

  /**
   * Linha do n de verso correntemente selecionado na rvore
   */
  private int currentVersionRow;

  /** Nome da raiz da rvore de verses do algoritmo */
  public static String ROOT_NAME;

  /**
   * Constri o painel da rvore de dados.
   * 
   * @param versionInfoPanel painel que criou essa viso
   * @param algorithmInfo informaes do algoritmo selecionado
   */
  public VersionTreeView(final AlgorithmVersionInfoPanel versionInfoPanel,
    AlgorithmInfo algorithmInfo) {
    this.versionInfoPanel = versionInfoPanel;
    currentVersionRow = -1;
    ROOT_NAME = getApplication().getString(
      "VersionTreeView.root.label.versions");
    getVersionTree().getSelectionModel().addTreeSelectionListener(
      new TreeSelectionListener() {
        @Override
        public void valueChanged(TreeSelectionEvent e) {
          Object node = e.getPath().getLastPathComponent();
          TreeNode treeNode = (TreeNode) node;
          initVersionSelection(treeNode);
        }
      });
  }

  /**
   * Obtm a aplicao.
   * 
   * @return a aplicao
   */
  private Application getApplication() {
    return versionInfoPanel.getApplication();
  }

  /**
   * Inicializa a seleo de uma nova verso na rvore de verses do algoritmo.
   * 
   * @param treeNode n de verso selecionado na rvore
   */
  private void initVersionSelection(TreeNode treeNode) {
    if (infoAlreadyUpdated) {
      versionInfoPanel.verifyAndChangeButtonsState();
    }
    else {
      if (changedSelectedVersion()) {
        if (!versionInfoPanel.confirmSelectionChanged()) {
          setCurrentNode();
          return;
        }
        currentVersionRow = getSelectedRow();
      }
      VersionTreeNode versionNode = getVersionNode(treeNode);
      versionInfoPanel.initVersionNodeEdition(versionNode);
    }
    versionInfoPanel.enableVersionEditPanel(
      treeNode instanceof VersionTreeNode);
  }

  /**
   * Obtm o painel correspondente a viso da rvore de verses.
   * 
   * @return o painel criado
   */
  public JPanel getVersionTreePanel() {
    if (versionTreePanel == null) {
      versionTreePanel = new JPanel(new GridBagLayout());
      versionTreePanel.add(new JScrollPane(getVersionTree()), new GBC(0, 0)
        .both().west().insets(0, 0, 0, 0));
    }
    return versionTreePanel;
  }

  /**
   * Obtm o primeiro n selecionado na rvore de dados.
   * 
   * @return o primeiro n selecionado na rvore de dados
   */
  public VersionNodeInterface getFirstSelectedNode() {
    VersionNodeInterface node = null;
    int selectedCount = versionTree.getSelectionCount();
    if (selectedCount == 1) {
      Object selectionPath = versionTree.getSelectionPath()
        .getLastPathComponent();
      node = (VersionNodeInterface) selectionPath;
    }
    return node;
  }

  /**
   * Obtm o primeiro n selecionado na rvore de dados.
   * 
   * @return o primeiro n selecionado na rvore de dados
   */
  public int getSelectedRow() {
    int[] selectionRows = versionTree.getSelectionRows();
    if (selectionRows != null && selectionRows.length > 0) {
      return selectionRows[0];
    }
    return 0;
  }

  /**
   * Obtm a lista dos dados selecionados na rvore de dados.
   * 
   * @return retorna a lista de dados selecionados na rvore de dados. Se a
   *         seleo
   */
  public List<VersionNodeInterface> getSelectedDataList() {
    List<VersionNodeInterface> selectedDataList = new ArrayList<>();
    TreePath[] selectionPaths = versionTree.getSelectionPaths();
    if (selectionPaths != null) {
      for (TreePath treePath : selectionPaths) {
        Object treeNode = treePath.getLastPathComponent();
        if (treeNode instanceof VersionNodeInterface) {
          selectedDataList.add((VersionNodeInterface) treeNode);
        }
      }
    }
    return selectedDataList;
  }

  /**
   * Obtm os ns referentes ao caminho completo do n selecionado.
   * 
   * @return uma lista com os ns referentes ao caminho completo do n
   *         selecionado
   */
  public List<VersionNodeInterface> getFullPathSelectedNode() {
    List<VersionNodeInterface> fullPath = new ArrayList<>();
    int selectedCount = versionTree.getSelectionCount();
    if (selectedCount == 1) {
      Object[] paths = versionTree.getSelectionPath().getPath();
      VersionTreeRootNode rootNode = (VersionTreeRootNode) paths[0];
      fullPath.add(rootNode);
      for (Object path : paths) {
        if (path instanceof VersionTreeNode) {
          VersionTreeNode node = (VersionTreeNode) path;
          fullPath.add(node);
        }
      }
    }
    return fullPath;
  }

  /**
   * Obtm o n raiz da rvore de dados.
   * 
   * @return retorna o n raiz da rvore de dados, caso contrrio, retorna null
   * 
   */
  public VersionTreeRootNode getRootNode() {
    TreePath rootPath = versionTree.getPathForRow(0);
    Object treeNode = rootPath.getLastPathComponent();
    if (treeNode instanceof VersionTreeRootNode) {
      return (VersionTreeRootNode) treeNode;
    }
    return null;
  }

  /**
   * Obtm um n a partir de um caminho da rvore de dados.
   * 
   * @param path caminho completo de um n da rvore
   * @return o n correspondente a esse caminho
   */
  protected VersionNodeInterface getNode(TreePath path) {
    if (path != null) {
      Object lastPath = path.getLastPathComponent();
      if (VersionNodeInterface.class.isAssignableFrom(lastPath.getClass())) {
        return (VersionNodeInterface) lastPath;
      }
    }
    return null;
  }

  /**
   * Verifica se o n raiz da rvore de dados est selecionado.
   * 
   * @return retorna true, se o n raiz da rvore de dados estiver selecionado,
   *         caso contrrio, retorna false
   * 
   */
  public boolean isRootNodeSelected() {
    return versionTree.isRowSelected(0);
  }

  /**
   * Obtm o nmero de itens selecionados na rvore de dados.
   * 
   * @return o nmero de itens selecionados na rvore de dados
   */
  public int getSelectionCount() {
    return versionTree.getSelectionCount();
  }

  /**
   * Obtm a rvore de verses do algoritmo selecionado.
   * 
   * @return a rvore de verses do algoritmo
   */
  public VersionTree getVersionTree() {
    if (versionTree == null) {
      AlgorithmListItem selectedAlgorithmItem = versionInfoPanel
        .getSelectedAlgorithmItem();
      AlgorithmInfo algoInfo = (selectedAlgorithmItem == null) ? null
        : selectedAlgorithmItem.getItem();
      versionTree = new VersionTree(getApplication().getApplicationFrame(),
        algoInfo);
      this.editor = new VersionTreeCellEditor(new CommonAlgoFileRenameAction(
        versionTree), versionTree);
      versionTree.setCellEditor(editor);
      enableRenameAction(true);
    }
    return versionTree;
  }

  /**
   * Inicializa a rvore de verso para um novo algoritmo.
   * 
   * @param algoInfo informaes do algoritmo
   */
  public void initializeVersionTree(AlgorithmInfo algoInfo) {
    getVersionTree().initializeTree(algoInfo);
    getVersionTreePanel().revalidate();
  }

  /**
   * Obtm o n de verso a partir de um n selecionado na rvore de verses.
   * 
   * @param treeNode n selecionado na rvore
   * @return Se o n selecionado for a raiz da rvore, retorna null. Se o n
   *         selecionado for um n de verso ou um sub-n de verso, ento
   *         retorna o n da verso.
   */
  public VersionTreeNode getVersionNode(TreeNode treeNode) {
    if (treeNode == null) {
      return null;
    }
    if (treeNode instanceof VersionTreeRootNode) {
      return null;
    }
    if (treeNode instanceof VersionTreeNode) {
      return (VersionTreeNode) treeNode;
    }
    else {
      TreeNode parent = treeNode.getParent();
      return getVersionNode(parent);
    }
  }

  /**
   * Verifica se a verso selecinada mudou em relao  verso corrente.
   * 
   * @return retorna true se a verso selecionado for diferente do que estava
   *         correntemente selecionada, caso contrrio, retorna false
   */
  private boolean changedSelectedVersion() {
    return currentVersionRow != getSelectedRow();
  }

  /**
   * Seleciona o n raiz da rvore de dados, inicializando suas informaes.
   */
  private void selectRootNode() {
    versionTree.setSelectionRow(0);
  }

  /**
   * Seleciona um determinado n na rvore, de acordo com o nmero do n
   * especificado. Inicializa as informaes do n selecionado.
   * 
   * @param nodeRow nmero do n a ser selecionado
   * 
   */
  private void selectNode(int nodeRow) {
    versionTree.selectNode(nodeRow);
  }

  /**
   * Seleciona o n corrente, inicializando os valores do item selecionado.
   * 
   */
  private void selectCurrentNode() {
    if (currentVersionRow == -1) {
      selectRootNode();
    }
    selectNode(currentVersionRow);
  }

  /**
   * Estabelece o n corrente para ser o n selecionado, sem fazer nenhum tipo
   * de inicializao dos seus valores, mantendo o estado corrente do item.
   */
  public void setCurrentNode() {
    infoAlreadyUpdated = true;
    selectCurrentNode();
    infoAlreadyUpdated = false;
  }

  /**
   * Realiza uma ao quando um algoritmo  alterado na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo alterado
   */
  public void handleAlgorithmUpdated(AlgorithmInfo algoInfo) {
    infoAlreadyUpdated = true;
    versionTree.replaceAlgoNode(algoInfo);
    infoAlreadyUpdated = false;
    selectCurrentNode();
  }

  /**
   * Habilita/desabilita edio nos ns da rvore. Usado para renomear os
   * arquivos de projeto.
   *
   * @param enable true, para habilitar a edio, ou false, caso contrrio.
   */
  public void enableRenameAction(boolean enable) {
    if (enable) {
      versionTree.setEditable(true);
      versionTree.setCellEditor(this.editor);
    }
    else {
      versionTree.setEditable(false);
      versionTree.setCellEditor(null);
    }
    renameEnabled = enable;
  }

  /**
   * Responsvel pela edio dos ns da rvore (renomear).
   */
  private class VersionTreeCellEditor extends DefaultTreeCellEditor {
    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isCellEditable(EventObject event) {
      VersionTree tree = getVersionTree();
      TreePath path = tree.getSelectionPath();
      if (path == null) {
        return false;
      }
      return super.isCellEditable(event);
    }

    /**
     * Construtor.
     *
     * @param action Classe efetivamente responsvel por renomear o arquivo.
     * @param tree rvore  qual pertence o n sendo editado.
     */
    VersionTreeCellEditor(CommonAlgoFileRenameAction action, JTree tree) {
      this(action, tree, new DefaultTreeCellRenderer(), null);
    }

    /**
     * Construtor. Adiciona listener que vai renomear o arquivo no servidor aps
     * o usurio ter terminado a sua edio.
     *
     * @param action Classe efetivamente responsvel por renomear o arquivo.
     * @param tree rvore  qual pertence o n sendo editado.
     * @param renderer Classe responsvel por desenhar os cones da rvore.
     * @param editor Editor de ns
     */
    VersionTreeCellEditor(final CommonAlgoFileRenameAction action, final JTree tree,
      DefaultTreeCellRenderer renderer, TreeCellEditor editor) {
      super(tree, renderer, editor);
      addCellEditorListener(new CellEditorListener() {
        /**
         * Dispara a ao de renomear o arquivo no servidor.
         */
        private void rename() {
          TreePath path = tree.getSelectionPath();
          if (path == null) {
            return;
          }
          Object component = path.getLastPathComponent();

          if (AbstractFileInfoNode.class.isAssignableFrom(component
            .getClass())) {
            AbstractFileInfoNode node = (AbstractFileInfoNode) component;
            FileInfo fileInfo = node.getFile();
            String oldName = fileInfo.getName();
            String newName = (String) getCellEditorValue();
            if (oldName.equals(newName)) {
              return;
            }
            if (ConfigurationFileNode.class.isAssignableFrom(component
              .getClass())) {
              ConfigurationFileNode confFileNode =
                (ConfigurationFileNode) component;
              action.renameFile(confFileNode.getVersion(), confFileNode
                .getType(), fileInfo, newName);
            }
            else if (DocumentationFileNode.class.isAssignableFrom(component
              .getClass())) {
              DocumentationFileNode docFileNode =
                (DocumentationFileNode) component;
              action.renameFile(docFileNode.getVersion(), docFileNode.getType(),
                fileInfo, newName);
            }
            else if (ReleaseNotesFileNode.class.isAssignableFrom(component
              .getClass())) {
              ReleaseNotesFileNode releaseNotesFileNode =
                (ReleaseNotesFileNode) component;
              action.renameFile(releaseNotesFileNode.getVersion(),
                releaseNotesFileNode.getType(), fileInfo, newName);
            }
            else if (ExecutableFileNode.class.isAssignableFrom(component
              .getClass())) {
              ExecutableFileNode execFileNode = (ExecutableFileNode) component;
              action.renameExecFile(execFileNode.getVersion(), execFileNode
                .getPlatform(), fileInfo, newName);
            }
          }
        }

        /**
         * Disparado se o usurio selecionar outro arquivo sem pressionar a
         * tecla ENTER.
         *
         * @param event .
         */
        @Override
        public void editingCanceled(ChangeEvent event) {
          rename();
        }

        /**
         * Disparado se o usurio pressionar a tecla ENTER.
         *
         * @param event .
         */
        @Override
        public void editingStopped(ChangeEvent event) {
          rename();
        }
      });

    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected TreeCellEditor createTreeCellEditor() {
      Border aBorder = UIManager.getBorder("Tree.editorBorder");
      /*
       * criamos um novo campo de texto onde o mtodo selectAll()  redefinido
       * para selecionar apenas at antes do ltimo '.'
       */
      DefaultTextField textField = new DefaultTextField(aBorder) {
        /**
         * {@inheritDoc}
         */
        @Override
        public void selectAll() {
          Document doc = getDocument();
          if (doc != null) {
            setCaretPosition(0);
            moveCaretPosition(getLastPositionBeforeDot());
          }
        }

        /**
         * Retorna a ltima posio a conter caracter antes da extenso do
         * arquivo.
         */
        private int getLastPositionBeforeDot() {
          String text = getText();
          int pos = text.lastIndexOf('.');
          if (pos == -1 || pos == 0) {
            /*
             * o nome no possui '.' ou este  o 1o caracter (arquivo oculto) e
             * no existe extenso; nestes casos, selecionamos tudo
             */
            return text.length();
          }
          return pos;
        }
      };

      return new DefaultCellEditor(textField);
    }
  }

}
