/*
 * $Id: LogViewer.java 175538 2016-08-24 18:46:56Z isabella $
 */
package csbase.client.applications.logviewer;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.KeyEvent;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.ActionMap;
import javax.swing.ButtonGroup;
import javax.swing.InputMap;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JOptionPane;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JRootPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;

import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationProject;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.util.event.EventListener;
import csbase.client.util.gui.log.LogPanel;
import csbase.client.util.gui.log.LogPanel.FileEvent;
import csbase.client.util.gui.log.LogPanel.ThrowableEvent;
import csbase.client.util.gui.log.LogPanelReloader;
import csbase.client.util.gui.log.LogPanelReloader.StatusChangedEvent;
import csbase.client.util.gui.log.actions.AutoReloadAction;
import csbase.client.util.gui.log.actions.CloseLogAction;
import csbase.client.util.gui.log.actions.DecPageAction;
import csbase.client.util.gui.log.actions.FirstPageAction;
import csbase.client.util.gui.log.actions.IncPageAction;
import csbase.client.util.gui.log.actions.LastPageAction;
import csbase.client.util.gui.log.actions.OpenLogAction;
import csbase.client.util.gui.log.actions.SetAutoReloadTimeAction;
import csbase.client.util.gui.log.actions.SetFontSizeAction;
import csbase.client.util.gui.log.actions.SetFontTypeAction;
import csbase.client.util.gui.log.enums.FontSize;
import csbase.client.util.gui.log.enums.FontType;
import csbase.logic.ClientProjectFile;
import tecgraf.javautils.gui.GBC;

/**
 * <b><u>Comportamentos para o caso do arquivo de log venha a ser movido,
 * apagado, ou substitudo</u></b><br>
 * <br>
 * 
 * Obs. Os smbolos 'a' e 'b' seram utilizados abaixo para identificar arquivos
 * diferentes, porm com o MESMO nome. Quando aberto pelo LogViewer o arquivo
 * 'b' tem a metade de pginas do arquivo 'a'. <br>
 * <ol>
 * <li>auto-recarga: ligada<br>
 * pgina corrente: ltima<br>
 * <br>
 * <u>Fluxo principal:</u>
 * <ol type="a">
 * <li>o arquivo 'a'  movido ou apagado<br>
 * <li>o usurio  informado pela barra de status que o arquivo no foi
 * encontrado.<br>
 * <li>o arquivo 'a'  recriado ou movido de volta para seu diretrio inicial.
 * <br>
 * <li>um dilogo  aberto com o cliente indicando que o arquivo foi encontrado
 * e ser recarregado.<br>
 * <br>
 * <u>Fluxo alternativo:</u><br>
 * c)  criado o arquivo 'b' com o mesmo nome de 'a' e no mesmo diretrio em que
 * 'a' se encontrava antes da etapa 1.a.<br>
 * <li>um dilogo  aberto com o cliente indicando que o arquivo foi encontrado
 * e ser recarregado.<br>
 * <li>como o usurio estava no tail da ltima pgina de 'a', este ir para o
 * tail da ltima pgina de 'b'.<br>
 * <br>
 * <br>
 * </ol>
 * 
 * <li>auto-recarga: desligada<br>
 * pgina corrente: N, onde {@code 0 <= N < nmero de pginas de 'a'}<br>
 * <br>
 * <u>Fluxo principal:</u><br>
 * <ol type = "a">
 * <li>o arquivo 'a'  movido ou apagado<br>
 * <li>o usurio tenta navegar entre as pginas<br>
 * <li>o usurio  informado pela barra de status que o arquivo no foi
 * encontrado.<br>
 * <li>o arquivo 'a'  recriado ou movido de volta para seu diretrio inicial.
 * <br>
 * <li>o usurio tenta navegar entre as pginas<br>
 * <li>
 * o usurio vai para a pgina escolhida<br>
 * <br>
 * <u>Fluxo alternativo 1:</u><br>
 * d)  criado o arquivo 'b' com o mesmo nome de 'a' e no mesmo diretrio em que
 * 'a' se encontrava antes da etapa 2.a.<br>
 * e) o usurio tenta navegar para uma pgina M onde {@code 0 <= M < nmero de pginas
 * e 'b'}<br>
 * f) o total de pginas  alterado para para nmero de pginas do arquivo 'b'<br>
 * g) o usurio vai para a pgina escolhida<br>
 * <br>
 * <u>Fluxo alternativo 2:</u><br>
 * d)  criado o arquivo 'b' com o mesmo nome de 'a' e no mesmo diretrio em que
 * 'a' se encontrava antes da etapa 2.a.<br>
 * e) o usurio tenta navegar para uma pgina M onde {@code M > nmero de pginas e 'b'}
 * <br>
 * f) o usurio  informado de que est tentando acessar uma pgina invlida
 * <br>
 * <li>
 * o total de pginas  alterado para para nmero de pginas do arquivo 'b'<br>
 * <li>o usurio vai para a ltima pgina de 'b'<br>
 * </ol>
 * </ol>
 * 
 * @author Tecgraf/PUC-Rio
 */
public class LogViewer extends ApplicationProject {

  /**
   * Indicativo interno de arquivo no existente
   */
  private final AtomicBoolean fileNotFound;

  /**
   * Painel de texto aonde fica o texto de logging
   */
  private final LogPanel logPanel;

  /**
   * Verifica se o default  ligar a auto-recarga assim que o arquivo for
   * aberto.
   */
  private boolean autoReload = true;

  /**
   * Construtor
   * 
   * @param id identificador da aplicao
   */
  public LogViewer(final String id) {
    super(id);

    fileNotFound = new AtomicBoolean(false);

    // Se a propriedade de auto-recarga no existir, o default  ligar a 
    // auto-recarga assim que o arquivo for aberto. 
    autoReload = this.getBooleanSpecificProperty("autoReload", true);

    /*
     * Testa-se o valor da propriedade do application manager do sistema para o
     * tamanho de pgina. Se no estiver definido, usa-se o valor default
     * exportado pela classe
     */
    final int defPageSizeKb = LogPanel.DEFAULT_PAGE_SIZE_KB;
    final String propName = "pageSizeKb";
    final int pageSizeKb = getIntSpecificProperty(propName, defPageSizeKb);
    logPanel = LogPanel.createLogPanelWithCompleteToolBar(pageSizeKb);

    /*
     * Adiciona a barra de ferramentas uma ajuda com dados sobre a aplicao que
     * detm este painel.
     */
    final JToolBar toolBar = logPanel.getToolBar();
    toolBar.addSeparator();
    toolBar.add(new ApplicationAboutAction(this));

    /* Captura os eventos de excees gerados pelo painel de log. */
    logPanel
      .addThrowableEventListener(new EventListener<LogPanel.ThrowableEvent>() {
        @Override
        public void eventFired(ThrowableEvent event) {
          getApplicationFrame().getStatusBar().setError(event.getMessage());
        }
      });

    /*
     * Captura os eventos de arquivo para notificar na barra de status caso o
     * arquivo no tenha sido encontrado ou caso tenha reaparecido. Tambm
     * inicializa o sistema de auto-recarga assim que um arquivo  aberto e o
     * interrompe sempre que o arquivo corrente for fechado
     */
    logPanel.addFileEventListener(new EventListener<LogPanel.FileEvent>() {
      @Override
      public void eventFired(FileEvent event) {
        getApplicationFrame().getStatusBar().clearStatus();
        if (LogPanel.FileEvent.Type.NOT_FOUND == event.getType()) {
          fileNotFound.set(true);
          /*
           * Caso o arquivo no tenha sido encontrado, o usurio  informado com
           * uma mensagem de erro na barra de status.
           * 
           * Esse erro ocorre quando o arquivo  movido ou apagado.
           */
          String warningMsg =
            LogPanel.getString("statusbar.warning.file.notFound", event
              .getFile().getName());
          getApplicationFrame().getStatusBar().setWarning(warningMsg);
        }
        else {
          boolean fileWasNotFound = fileNotFound.get();
          fileNotFound.set(false);
          getApplicationFrame().getStatusBar().clearStatus();

          if (LogPanel.FileEvent.Type.RELOADED == event.getType()
            && fileWasNotFound) {
            /**
             * Se a recarga foi feita com sucesso e antes dela o estado era de
             * arquivo no encontrado, devemos avisar ao usurio ento que o
             * arquivo voltou.
             */
            final String warningMsg =
              LogPanel.getString("dialog.info.file.isBack",
                new Object[] { event.getFile().getName() });
            final ApplicationFrame frame = getApplicationFrame();
            final String title = getName();

            String[] buttons = new String[] { LogPanel.getString("button.ok") };
            JOptionPane.showOptionDialog(frame, warningMsg, title,
              JOptionPane.DEFAULT_OPTION, JOptionPane.PLAIN_MESSAGE, null,
              buttons, buttons[0]);
          }

          // Verifica se o default  ligar a auto-recarga.
          if (autoReload == true) {
            /*
             * Inicializa o sistema de auto-recarga assim que um arquivo 
             * aberto e o interrompe sempre que o arquivo corrente for fechado.
             */
            if (LogPanel.FileEvent.Type.CLOSED == event.getType()) {
              logPanel.getReloader().stop();
            }
            else if (LogPanel.FileEvent.Type.OPENED == event.getType()) {
              logPanel.getReloader().start();
            }
          }
        }
      }
    });

    /*
     * Passa o LogPanel como parmetro demonstrando assim que  necessria a sua
     * criao antes da execuo dos mtodos de construo do frame e de
     * configurao das hot keys.
     */
    buildFrame(logPanel);
    configHotKeys(logPanel);
  }

  /**
   * Abre um arquivo de log com a auto-recarga ligada.
   * 
   * @param file o arquivo.
   */
  public final void openFile(final ClientProjectFile file) {
    logPanel.openFile(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(String name, Object value, String senderId) {
    if (value == null) {
      return;
    }
    if (name.equals(PROJECT_FILE_MESSAGE)) {
      ClientProjectFile file = (ClientProjectFile) value;
      openFile(file);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    try {
      logPanel.closeFile();
    }
    catch (Exception e) {
      handleError(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * Montagem do dilogo principal.
   * 
   * @param logPanel o panel de log.
   */
  private void buildFrame(final LogPanel logPanel) {
    final DesktopComponentFrame mainFrame = getApplicationFrame();
    mainFrame.getContentPane().setLayout(new GridBagLayout());
    mainFrame.getContentPane().add(logPanel, new GBC(0, 0).both());
    mainFrame.setJMenuBar(new LogViewerMenuBar(logPanel));
    mainFrame.pack();
    mainFrame.setSize(new Dimension(800, 600));
    mainFrame.getStatusBar().showStatusBar();
  }

  /**
   * Inicializa teclas de atalho para o mecanismo de busca.
   * 
   * @param logPanel o panel de log.
   */
  private void configHotKeys(final LogPanel logPanel) {
    final ApplicationFrame frame = getApplicationFrame();
    final JRootPane rootPane = frame.getRootPane();
    final int mode = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    final InputMap inputMap = rootPane.getInputMap(mode);
    final ActionMap aMap = rootPane.getActionMap();

    // CTRL + O : open
    final KeyStroke crtlO =
      KeyStroke.getKeyStroke(KeyEvent.VK_O, KeyEvent.CTRL_DOWN_MASK);
    inputMap.put(crtlO, crtlO.toString());
    aMap.put(crtlO.toString(), new OpenLogAction(logPanel));
  }

  /**
   * Esta classe representa a barra de menu da aplicao de log contendo os
   * menus: Arquivo, Preferncias, Pgina e Ajuda.
   * 
   * @author Tecgraf / PUC-Rio
   */
  class LogViewerMenuBar extends JMenuBar {

    /**
     * Construtor
     * 
     * @param logPanel o painel de log
     */
    LogViewerMenuBar(final LogPanel logPanel) {
      add(buildFileMenu(logPanel));
      add(buildPrefsMenu(logPanel));
      add(buildPagingMenu(logPanel));
      add(buildHelpMenu(logPanel));
    }

    /**
     * @param logPanel o painel de log
     * @return o menu de arquivos.
     */
    private JMenu buildFileMenu(final LogPanel logPanel) {
      final JMenu menu = new JMenu();
      menu.setText(getString("file.menu"));
      menu.add(new OpenLogAction(logPanel));
      menu.add(new CloseLogAction(logPanel));
      menu.addSeparator();
      menu.add(new ApplicationExitAction(LogViewer.this));
      return menu;
    }

    /**
     * @param logPanel o painel de log
     * @return menu de preferncias.
     */
    private JMenu buildPrefsMenu(final LogPanel logPanel) {
      final JMenu menu = new JMenu();
      menu.setText(getString("preferences.menu"));

      /** Item de menu de auto-reload. */
      final JCheckBoxMenuItem autoItem = new JCheckBoxMenuItem();
      autoItem.setAction(new AutoReloadAction(logPanel));
      autoItem.setSelected(logPanel.getReloader().isRunning());
      logPanel.getReloader().addStatusChangedEventListener(
        new EventListener<LogPanelReloader.StatusChangedEvent>() {
          @Override
          public void eventFired(StatusChangedEvent event) {
            autoItem.setSelected(event.isRunning());
          }
        });
      menu.add(autoItem);

      JMenu timeReloadMenu = buildPrefsTimeMenu(logPanel);
      menu.add(timeReloadMenu);

      final JMenu fontMenu = new JMenu();
      fontMenu.setText(getString("preferences.font.menu"));
      fontMenu.add(buildFontTypeMenu(logPanel));
      fontMenu.add(buildFontSizeMenu(logPanel));

      menu.add(fontMenu);

      return menu;
    }

    /**
     * @param logPanel o painel de log
     * @return menu de troca de fontes, sub-menu de preferncias.
     */
    private JMenu buildFontTypeMenu(final LogPanel logPanel) {
      final JMenu menu = new JMenu();
      menu.setText(getString("preferences.font.type.menu"));
      final ButtonGroup grp = new ButtonGroup();
      for (FontType tp : FontType.values()) {
        final SetFontTypeAction action = new SetFontTypeAction(logPanel, tp);
        final JRadioButtonMenuItem it = new JRadioButtonMenuItem(action);
        menu.add(it);
        grp.add(it);
        if (tp == logPanel.getTextArea().getFontType()) {
          it.setSelected(true);
        }
      }
      return menu;
    }

    /**
     * @param logPanel o painel de log
     * @return menu de troca de tamanho de fonte, sub-menu de preferncias.
     */
    private JMenu buildFontSizeMenu(final LogPanel logPanel) {
      final JMenu menu = new JMenu();
      menu.setText(getString("preferences.font.size.menu"));
      final ButtonGroup grp = new ButtonGroup();
      for (FontSize sz : FontSize.values()) {
        final SetFontSizeAction action = new SetFontSizeAction(logPanel, sz);
        final JRadioButtonMenuItem it = new JRadioButtonMenuItem(action);
        menu.add(it);
        grp.add(it);
        if (sz == logPanel.getTextArea().getFontSize()) {
          it.setSelected(true);
        }
      }
      return menu;
    }

    /**
     * @param logPanel o painel de log
     * @return menu de tempos de reload, sub-menu de preferncias.
     */
    private JMenu buildPrefsTimeMenu(final LogPanel logPanel) {
      final JMenu timeReloadMenu = new JMenu();

      timeReloadMenu.setEnabled(logPanel.getReloader().isRunning());
      logPanel.getReloader().addStatusChangedEventListener(
        new EventListener<LogPanelReloader.StatusChangedEvent>() {
          @Override
          public void eventFired(StatusChangedEvent event) {
            timeReloadMenu.setEnabled(event.isRunning());
          }
        });

      final String reloadPrefix = "preferences.reload.";
      final String timeMenuText = getString(reloadPrefix + "menu");
      timeReloadMenu.setText(timeMenuText);
      final SetAutoReloadTimeAction r10 =
        new SetAutoReloadTimeAction(logPanel, 10);
      final SetAutoReloadTimeAction r20 =
        new SetAutoReloadTimeAction(logPanel, 20);
      final SetAutoReloadTimeAction r60 =
        new SetAutoReloadTimeAction(logPanel, 60);

      final String t10 = getString(reloadPrefix + "10.menu");
      final String t20 = getString(reloadPrefix + "20.menu");
      final String t60 = getString(reloadPrefix + "60.menu");
      final JRadioButtonMenuItem mn10 = new JRadioButtonMenuItem(r10);
      final JRadioButtonMenuItem mn20 = new JRadioButtonMenuItem(r20);
      final JRadioButtonMenuItem mn60 = new JRadioButtonMenuItem(r60);
      mn10.setText(t10);
      mn20.setText(t20);
      mn60.setText(t60);
      final ButtonGroup grp = new ButtonGroup();
      grp.add(mn10);
      grp.add(mn20);
      grp.add(mn60);
      mn10.setSelected(true);

      timeReloadMenu.add(mn10);
      timeReloadMenu.add(mn20);
      timeReloadMenu.add(mn60);

      return timeReloadMenu;
    }

    /**
     * @param logPanel o painel de log
     * @return o menu de paginao.
     */
    private JMenu buildPagingMenu(final LogPanel logPanel) {
      final JMenu menu = new JMenu();
      menu.setText(getString("page.menu"));
      menu.add(new IncPageAction(logPanel));
      menu.add(new DecPageAction(logPanel));
      menu.addSeparator();
      menu.add(new FirstPageAction(logPanel));
      menu.add(new LastPageAction(logPanel));

      return menu;
    }

    /**
     * @param logPanel o painel de log
     * @return o menu de ajuda.
     */
    private JMenu buildHelpMenu(final LogPanel logPanel) {
      final JMenu menu = new JMenu();
      menu.setText(getString("help.menu"));
      menu.add(new ApplicationAboutAction(LogViewer.this));
      return menu;
    }
  }
}
