package csbase.client.algorithms.commands.view;

import java.io.IOException;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandInfo;
import csbase.logic.CommonClientProject;

/**
 * Task que busca os logs de um comando a partir de padres de nomes de arquivo
 * configurados no {@link ConfigurationManager}.
 */
public class GetLogTabConfigurationsTask extends
  RemoteTask<List<LogTabConfiguration>> {

  /**
   * Nome da propriedade do {@link ConfigurationManager} que determina o
   * identificador de uma aba de log.
   */
  private static final String LOG_TAB_ID_PROPERTY_NAME = "tab.{0}.id";

  /**
   * Nome da propriedade do {@link ConfigurationManager} que determina o ttulo
   * de uma aba de log.
   */
  private static final String LOG_TAB_TITLE_PROPERTY_NAME = "tab.{0}.title.{1}";

  /**
   * Nome da propriedade do {@link ConfigurationManager} que determina o padro
   * de nome de arquivo a ser mostrado na aba de log.
   */
  private static final String LOG_TAB_FILE_PATTERN_PROPERTY_NAME =
    "tab.{0}.log.file.pattern.{1}";

  /**
   * Tag que pode ser utilizada na propriedade
   * {@link #LOG_TAB_FILE_PATTERN_PROPERTY_NAME} para adicionar um identificador
   * no padro de nome de arquivo de log.
   */
  private static final String ID_REPLACEMENT_TAG = "$ID";

  /**
   * Tipo de log a ser buscado.
   */
  private CommandViewType logViewType;

  /**
   * Identificador que ir substituir o {@link #ID_REPLACEMENT_TAG} (se
   * existente) na {@link #LOG_TAB_FILE_PATTERN_PROPERTY_NAME}.
   */
  private String idForLogPattern;

  /**
   * Informaes sobre o comando cujo log deve ser apresentado.
   */
  protected CommandInfo commandInfo;

  /**
   * Construtor
   * 
   * @param commandInfo Informaes sobre o comando cujo log deve ser
   *        apresentado.
   * @param logViewType Tipo de log a ser buscado.
   * @param idForLogPattern Identificador que ir substituir o
   *        {@link #ID_REPLACEMENT_TAG} (se existente) na
   *        {@link #LOG_TAB_FILE_PATTERN_PROPERTY_NAME}.
   */
  public GetLogTabConfigurationsTask(CommandInfo commandInfo,
    CommandViewType logViewType, String idForLogPattern) {
    this.commandInfo = commandInfo;
    this.logViewType = logViewType;
    this.idForLogPattern = idForLogPattern;
  }

  /**
   * Construtor
   * 
   * @param commandInfo Informaes sobre o comando cujo log deve ser
   *        apresentado.
   * @param logViewType Tipo de log a ser buscado.
   */
  public GetLogTabConfigurationsTask(CommandInfo commandInfo,
    CommandViewType logViewType) {
    this(commandInfo, logViewType, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws IOException,
    ConfigurationManagerException {
    setResult(getLogTabConfigurations());
  }

  /**
   * L a lista de configuraes de abas de log do {@link ConfigurationManager}.
   * 
   * @return A lista de configuraes.
   * @throws ConfigurationManagerException casa ocorra erro na leitura das
   *         configuraes.
   * @throws RemoteException caso ocorra erra na leitura dos diretrios de log.
   */
  private List<LogTabConfiguration> getLogTabConfigurations()
    throws ConfigurationManagerException, RemoteException {
    List<LogTabConfiguration> configurations =
      new ArrayList<LogTabConfiguration>();
    ConfigurationManager cnfManager = ConfigurationManager.getInstance();
    if (cnfManager != null) {
      Class<ConsolidatedLogsTab> propClass = ConsolidatedLogsTab.class;
      Configuration cnf = cnfManager.getConfiguration(propClass);
      ClientProjectFile logsBaseDir = getLogsSearchPath();
      if (logsBaseDir != null) {
        ClientProjectFile[] children = logsBaseDir.getChildren();
        boolean continueReading = true;
        for (int i = 1; continueReading; i++) {
          String tabId =
            cnf.getOptionalProperty(MessageFormat.format(
              LOG_TAB_ID_PROPERTY_NAME, i));
          if (tabId != null) {
            String title =
              cnf.getMandatoryProperty(MessageFormat.format(
                LOG_TAB_TITLE_PROPERTY_NAME, tabId, LNG.getLocale()));
            List<String> filePatterns =
              cnf.getMandatoryListProperty(MessageFormat.format(
                LOG_TAB_FILE_PATTERN_PROPERTY_NAME, tabId,
                logViewType.getPropertyName()));
            Set<ClientProjectFile> matchingFiles =
              findLogFiles(children, filePatterns);
            if (!matchingFiles.isEmpty()) {
              configurations.add(new LogTabConfiguration(tabId, title,
                matchingFiles));
            }
          }
          else {
            continueReading = false;
          }
        }
      }
    }
    return configurations;
  }

  /**
   * Retorna o caminho para o diretrio de onde os logs devero ser buscados.
   * 
   * @return o caminho para o diretrio de onde os logs devero ser buscados.
   * @throws RemoteException Se no for possvel obter o caminho.
   */
  private ClientProjectFile getLogsSearchPath() throws RemoteException {
    ClientProjectFile logsBaseDir = null;
    String[] logsPath = commandInfo.getPersistencyPath();
    if (logsPath != null) {
      final DesktopFrame desktop = DesktopFrame.getInstance();
      final CommonClientProject project = desktop.getProject();
      if (project != null) {
        logsBaseDir = project.getFile(logsPath);
      }
    }
    return logsBaseDir;
  }

  /**
   * Filtra um array de arquivos e retorna um subconjunto com os arquivos de
   * log. A filtragem  feita de acordo com o padres de nomeclatura
   * configurados para arquivos que devem ser mostrados em cada aba do sistema.
   * 
   * @param files O array de arquivos a ser filtrado.
   * @param patterns A lista de padres de nomenclatura de arquivos
   *        configurados.
   * @return O conjunto de arquivos de log.
   * @throws RemoteException Se no for possvel obter os arquivos.
   */
  private Set<ClientProjectFile> findLogFiles(final ClientProjectFile[] files,
    final List<String> patterns) throws RemoteException {
    final Set<ClientProjectFile> logFiles = new HashSet<ClientProjectFile>();
    for (final ClientProjectFile file : files) {
      if (file.isDirectory()) {
        logFiles.addAll(findLogFiles(file.getChildren(), patterns));
      }
      else if (matchesPatterns(file, patterns)) {
        logFiles.add(file);
      }
    }
    return logFiles;
  }

  /**
   * Testa se um determinado arquivo satisfaz algum dos padres de nomenclatura
   * para arquivos de log definidos para uma aba deste sistema. Caso positivo, o
   * arquivo deve ser mostrado na aba.
   * 
   * @param file O arquivo a ser testado.
   * @param logFilePatterns A lista de padres de nomenclatura para arquivos de
   *        log de uma aba.
   * @return Verdadeiro se o arquivo pode ser considerado um log de acordo com o
   *         padro de nomenclatura definido ou falso, caso contrrio.
   */
  private boolean matchesPatterns(final ClientProjectFile file,
    final List<String> logFilePatterns) {
    if (logFilePatterns != null) {
      for (String pattern : logFilePatterns) {
        if (idForLogPattern != null) {
          pattern = pattern.replace(ID_REPLACEMENT_TAG, idForLogPattern);
        }
        if (file.getName().matches(pattern.trim())) {
          return true;
        }
      }
    }
    return false;
  }

}
