package csbase.client.applicationmanager;

import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import csbase.client.Client;
import tecgraf.javautils.core.timestamp.TStamp32;

import javax.ws.rs.core.MediaType;

/**
 * Controla as aplicaes externas conhecidas por este desktop
 * 
 *
 * @author Tecgraf/PUC-Rio
 */
public class ExternalApplicationManager {

  /** Pool de threads para chamadas assncronas */

  private static final ExecutorService threadPool = new ThreadPoolExecutor(1,
          Integer.MAX_VALUE, 60L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>());

  /** Serializador JSON */
  private static final ObjectMapper objectMapper = new ObjectMapper();

  /** Mapa de applications e suas localizaes */
  private Map<String, String> applications = new HashMap<>();


  private final static String USER_AGENT = "CSBase REST Client";

  public ExternalApplicationManager() {
  // URL para testes externos
//   applications.put("test","http://127.0.0.1:8081/");
  }

  private static String sendPost(String url, String data) {
    try {
      URL obj = new URL(url);
      HttpURLConnection con = (HttpURLConnection) obj.openConnection();

      con.setRequestMethod("POST");

      //con.setRequestProperty("Authentication", "Bearer $token$"); //setar cabecalho, se for o caso

      con.setRequestProperty("User-Agent", USER_AGENT+" "+Client.getInstance().getVersion());
      con.setRequestProperty("Content-Type", MediaType.APPLICATION_JSON);
      con.setRequestProperty("Accept", MediaType.APPLICATION_JSON);

      con.setDoOutput(true);
      DataOutputStream wr = new DataOutputStream(con.getOutputStream());

      wr.writeBytes(data);
      wr.flush();
      wr.close();

      int responseCode = con.getResponseCode();

      //TODO remover esses logs
      System.out.println("\nSending 'POST' request to URL: " + url);
      System.out.println("Post data: " + data);
      System.out.println("Response Code: " + responseCode);

      BufferedReader in = new BufferedReader(
              new InputStreamReader(con.getInputStream()));
      String inputLine;
      StringBuffer response = new StringBuffer();

      while ((inputLine = in.readLine()) != null) {
        response.append(inputLine);
      }
      in.close();

      //TODO remover esse log
      System.out.println("Response Data:\n" + response.toString());

      return response.toString();

    } catch (MalformedURLException e) {
      e.printStackTrace();
    } catch (IOException e) {
      e.printStackTrace();
    }
    //TODO lanar exceo se der erro
    return "";
  }

  /**
   * Registar uma aplicao externa associando-a  um id que pode ser utilizado
   * no envio e recebimento de mensagens
   * 
   * @param url O caminho onde est a aplicao externa
   * @param type O tipo da aplicao
   * @return O id associado a aplicao
   */
  public String registerApplication(String url, String type) {
    TStamp32 idSuffix = new TStamp32();
    String id = type + "_" + idSuffix;
    applications.put(id, url);
    return id;
  }

  /**
   * Desregistar uma aplicao externa associanda  um id
   *
   * @param id O id da aplicao externa
   * @return true se a aplicao estava previamente registrada, false caso contrario
   */
  public boolean unregisterApplication(String id) {
    String removed = applications.remove(id);
    return removed!=null;
  }


  /**
   * Indica se existe aplicao com tal id;
   * 
   * @param id
   * @return Verdadeiro se existe
   */
  public boolean hasId(String id) {
    return applications.containsKey(id);

  }

  /**
   * Envia uma mensagem assncrona para todas as aplicaes.
   *
   * @param type tipo da mensagem.
   * @param value valor da mensagem.
   * @param senderId id da aplicao de origem da mensagem.
   */
  public void broadcastMessage(String type, Object value, String senderId) {

    if (senderId == null) {
      throw new IllegalArgumentException("senderID can't be null");
    }

    applications.keySet().stream().filter(key -> !(key.equals(senderId)))
      .forEach(key -> sendAsyncMessage(key, type, value, senderId));
  }


  private class RequestMessage {
    private String type;
    private String senderId;
    private Object value;

    public RequestMessage(String type, String senderId, Object value) {
      this.type = type;
      this.senderId = senderId;
      this.value = value;
    }

    public String getType() {
      return type;
    }

    public String getSenderId() {
      return senderId;
    }

    public Object getValue() {
      return value;
    }
  }

  /**
   * Envia uma mensagem assncrona para a aplicao.
   *
   * @param appInstanceId id da instncia da aplicao.
   * @param type tipo da mensagem.
   * @param value valor da mensagem.
   * @param senderId id da aplicao de origem da mensagem.
   */
  public void sendAsyncMessage(String appInstanceId, String type, Object value,
    String senderId) {
      String url=applications.get(appInstanceId);
      if(url==null) {
        throw new IllegalArgumentException("instance ID not registered");
      }

      //processar a saida JSON
      try {
        RequestMessage message=new RequestMessage(type,senderId,value);
        String data = objectMapper.writeValueAsString(message);

        threadPool.submit(()-> {
          //TODO tra
          sendPost(url,data);
        });

      } catch (JsonProcessingException e) {
        e.printStackTrace();
      }
  }

  /**
   * Envia uma mensagem sncrona para a aplicao.
   *
   * @param appInstanceId id da instncia da aplicao.
   * @param type tipo da mensagem.
   * @param value valor da mensagem.
   * @param senderId id da aplicao de origem da mensagem.
   * @return A resposta dapa pela aplicao
   */
  public Object sendSyncMessage(String appInstanceId, String type, Object value,
    String senderId) {
    String url=applications.get(appInstanceId);
    if(url==null) {
      throw new IllegalArgumentException("instance ID not registered");
    }

    //processar a saida JSON
    try {
      RequestMessage message=new RequestMessage(type,senderId,value);
      String data = objectMapper.writeValueAsString(message);

      String response=sendPost(url,data);
      HashMap<String, Object> map;
      map = objectMapper.readValue(response, HashMap.class);

      return map;

    } catch (JsonProcessingException e) {
      e.printStackTrace();
    } catch (IOException e) {
      e.printStackTrace();
    }
    return null;
  }
}
