package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.applications.projectsmanager.proxy.SetProjectSharingStatsTask;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;

/**
 * Ao que altera o status de compartilhamento de projetos para modo "privado".
 * 
 * @author jnlopes
 * 
 */
public class SetProjectPrivateAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public SetProjectPrivateAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc} Mtodo que invoca a remote task que efetivamente altera o
   * status de compartilhamento de projetos para modo "privado".
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final boolean result = runTask(projects);

    if (result) {
      for (ProjectsManagerData pmd : projects) {
        pmd.setSharingType(SharingType.PRIVATE);
      }
      projectsManager.refreshProjectsTable();
      projectsManager.refreshInfoPanel();
    }
    else {
      final String errMsg =
        getString("SetProjectPrivateAction.set.private.error.message");
      final String errTitle =
        getString("SetProjectPrivateAction.set.private.error.title");
      StandardDialogs.showErrorDialog(frame, errTitle, errMsg);
    }
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @return Boolean indicando se a operao foi bem sucedida.
   * @throws Exception em caso de erro.
   */
  private boolean runTask(final List<ProjectsManagerData> projects)
    throws Exception {
    final ProjectsManager projectsManager = getProjectsManager();
    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    final SetProjectSharingStatsTask fptt =
      new SetProjectSharingStatsTask(projectsManager, projects,
        ProjectsManagerScope.PRIVATE, SharingType.PRIVATE);
    fptt.execute(frame, getString("SetProjectPrivateAction.title"),
      getString("SetProjectPrivateAction.message"));
    if (fptt.wasCancelled()) {
      final String err = getString("SetProjectPrivateAction.cancelled.message");
      fptt.showError(err);
      return false;
    }
    if (fptt.getStatus() != true) {
      final Exception exception = fptt.getError();
      throw exception;
    }
    return fptt.getResult();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ApplicationImages.ICON_PRIVATE_16;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    final List<ProjectsManagerData> prjList =
      new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    final List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    final User loggedUser = User.getLoggedUser();
    final Object loggedUserId = loggedUser.getId();

    int i = 0;
    while (i < prjList.size()) {
      final ProjectsManagerData pmd = prjList.get(i);
      final ProjectSpaceAllocation spaceAllocation =
        pmd.getProjectSpaceAllocation();

      final boolean isAdmin = loggedUser.isAdmin();
      final boolean isOwner = pmd.getOwnerId().equals(loggedUserId);

      /*
       * Para que um usurio possa tornar um projeto privado  preciso que: - O
       * projeto possa ser aberto (no esteja aguardando alocao /
       * desalocao). - O usurio seja o administrador ou o dono do projeto. -
       * O projeto j no tenha seu escopo no modo privado.
       */
      if (spaceAllocation.isOpenable() && (isAdmin || isOwner)) {
        i++;
      }
      else {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
    }

    final ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      final String deniedMessage =
        getString("SetProjectPrivateAction.project.selection.denied.message")
          + getString("SetProjectPrivateAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("SetProjectPrivateAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      final String someDeniedMessage =
        getString("SetProjectPrivateAction.project.selection.some.denied.message")
          + getString("SetProjectPrivateAction.project.requirements.message");
      final int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("SetProjectPrivateAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }

}