/*
 * $Id: CommonProjectSelectDialog.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.project;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.UserProjectInfo;

/**
 * A classe <code>CommonProjectSelectDialog</code> mostra um dilogo modal para
 * seleo um projeto de um usurio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class CommonProjectSelectDialog extends DesktopComponentDialog {

  /**
   * A lista com os nomes de projetos do usurio
   */
  private JList projectList;

  /**
   * O boto default da janela
   */
  final private JButton defaultButton = new JButton();

  /**
   * O array com os nomes dos projetos do usurio
   */
  final private List<UserProjectInfo> projectInfos;

  /**
   * Indica se o seleciona um ou mais projetos.
   */
  final private boolean singleSelection;

  /**
   * Cria um dilogo que aguarda a seleo de um projeto. Solicita ao servidor
   * que o projeto selecionado seja aberto.
   */
  private void makeDialog() {
    // Cria o painel principal da janela
    JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.add(makeProjectListPanel(), BorderLayout.CENTER);
    mainPanel.add(makeButtonPanel(), BorderLayout.SOUTH);

    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        close();
      }
    });
    getContentPane().add(mainPanel);
    pack();
    center(getOwner());
    getRootPane().setDefaultButton(defaultButton);

  }

  /**
   * Obtm a lista de projetos de um usurio.
   * 
   * @return A lista completa de projetos do usurio.
   */
  private List<UserProjectInfo> getProjects() {
    RemoteTask<List<UserProjectInfo>> task =
      new RemoteTask<List<UserProjectInfo>>() {
        @Override
        public void performTask() throws Exception {
          setResult(getAllProjectNames(User.getLoggedUser().getId()));
        }
      };
    final String msg = LNG.get("CommonProjectSelectDialog.info.get.projects");
    if (task.execute(getOwner(), getTitle(), msg)) {
      return task.getResult();
    }
    return new ArrayList<UserProjectInfo>();
  }

  /**
   * Obtem a lista de projetos selecionados.
   * 
   * @return a lista
   */
  public List<UserProjectInfo> getSelectedProjects() {
    final List<UserProjectInfo> list = new ArrayList<UserProjectInfo>();
    final Object[] selectedProjects = projectList.getSelectedValues();
    for (Object object : selectedProjects) {
      list.add((UserProjectInfo) object);
    }
    return list;
  }

  /**
   * Cria um painel com a lista dos projetos apresentados no dilogo. O contedo
   * da lista  requisitado ao servidor.
   * 
   * @return painel.
   */
  private JPanel makeProjectListPanel() {
    final String projectString = LNG.get("CommonProjectSelectDialog.projects");
    final JLabel projectLabel = new JLabel(projectString);

    projectList = new JList(projectInfos.toArray());
    final ListSelectionModel selectionModel = projectList.getSelectionModel();
    if (singleSelection) {
      final int selectionMode = ListSelectionModel.SINGLE_SELECTION;
      selectionModel.setSelectionMode(selectionMode);
    }
    else {
      final int selectionMode = ListSelectionModel.MULTIPLE_INTERVAL_SELECTION;
      selectionModel.setSelectionMode(selectionMode);
    }

    // Associa o duplo clique a ao principal do dilogo
    final MouseListener mouseListener = new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
          handleAction();
        }
      }
    };
    projectList.addMouseListener(mouseListener);
    projectList.setSelectedIndex(0);

    final JScrollPane usersScrollPane =
      new JScrollPane(projectList,
        ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
        ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);

    int width;
    int height;
    final int defaultWidth = 270;
    final int defaultHeight = 130;
    try {
      final ConfigurationManager mng = ConfigurationManager.getInstance();
      final Configuration cnf = mng.getConfiguration(this.getClass());
      width = cnf.getOptionalIntegerProperty("list.width", defaultWidth);
      height = cnf.getOptionalIntegerProperty("list.height", defaultHeight);
    }
    catch (Exception e) {
      width = defaultWidth;
      height = defaultHeight;
    }
    usersScrollPane.setPreferredSize(new Dimension(width, height));

    final JPanel panel = new JPanel(new GridBagLayout());
    panel.add(projectLabel, new GBC(0, 0).west().insets(10, 10, 5, 10));
    panel
      .add(usersScrollPane, new GBC(0, 1).both().west().insets(0, 10, 5, 10));

    return panel;
  }

  /**
   * Retorna os projetos existentes de um dado usurio. Este mtodo pode ser
   * sobrescrito para abrir projeto especficos de cada sistema.
   * 
   * @param userId Identificador do usurio.
   * @return Lista com informaes sobre cada projeto.
   * @throws Exception em caso de falha.
   */
  protected List<UserProjectInfo> getAllProjectNames(Object userId)
    throws Exception {
    return CommonClientProject.getAllProjects(userId);
  }

  /**
   * Ajusta texto do boto de ao.
   * 
   * @param text texto
   */
  protected void setDefaultTextButtonText(String text) {
    defaultButton.setText(text);
  }

  /**
   * Cria um painel com os botes que vo ser usados no dilogo requisitado.
   * Esse mtodo deve ser sobrescrito dependendo acao da selecao.
   * 
   * @return O painel recm-criado.
   */
  protected JPanel makeButtonPanel() {
    String defaultText = LNG.get("CommonProjectSelectDialog.open");
    defaultButton.setText(defaultText);
    defaultButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        handleAction();
      }
    });

    String cancelText = LNG.get("CommonProjectSelectDialog.cancel");
    JButton cancelButton = new JButton(cancelText);
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        close();
      }
    });

    JComponent[] components = new JComponent[] { defaultButton, cancelButton };
    GUIUtils.matchPreferredSizes(components);

    JPanel panel = new JPanel();
    panel.add(cancelButton);
    panel.add(defaultButton);
    return panel;
  }

  /**
   * Executa a ao de abrir o projeto selecionado e instalar um observador para
   * ele.
   */
  abstract protected void handleAction();

  /**
   * Constri uma viso de dilogo para seleo de projetos de um um usurio.
   * 
   * @param owner A janela a partir da qual esse dilogo foi chamado.
   * @param singleSelection indica se o dilogo  de seleo simples ou
   *        mltipla.
   */
  protected CommonProjectSelectDialog(Window owner, boolean singleSelection) {
    super(owner);
    this.projectInfos = getProjects();
    this.singleSelection = singleSelection;
    Collections.sort(projectInfos);
    makeDialog();
  }
}
