package csbase.client.algorithms.parameters;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.ListModel;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import csbase.client.applications.ApplicationImages;
import csbase.client.util.ClientUtilities;
import csbase.logic.algorithms.parameters.FileListParameter;
import csbase.logic.algorithms.parameters.FileURLValue;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;

/**
 * A Viso para {@link FileListParameter Parmetro do Tipo Lista de Arquivos}.
 *
 * @author Tecgraf/PUC-Rio
 */
public abstract class FileListParameterView extends
SimpleParameterView<List<FileURLValue>> {

  /**
   * Prefixo para internacionalizao de idioma.
   */
  protected static final String LNGPREFIX = FileListParameterView.class
    .getName()
    + ".";

  /**
   * Cria uma viso.
   *
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   *        {@link ParameterView.Mode#REPORT}.
   */
  public FileListParameterView(FileListParameter parameter, Mode mode) {
    super(parameter, mode);
    updateViewContents();
    updateCapabilityView();
    updateVisibilyView();
  }

  /**
   * Interface do componente
   *
   * @author Tecgraf/PUC-Rio
   */
  protected interface FileListParameterComponent {
    /**
     * Atualiza o contedo exibido pela viso.
     */
    void updateViewContents();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createConfigurationComponent(Object... componentArgs) {
    return new FileListConfigurationParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createReportComponent(Object... componentArgs) {
    return new FileListReportParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateViewContents() {
    ((FileListParameterComponent) getComponent()).updateViewContents();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileListParameter getParameter() {
    return (FileListParameter) super.getParameter();
  }

  /**
   * Solicita um arquivo da usurio.
   *
   * @return O arquivo ou {@code null} se o usurio cancelar ou houver um erro.
   */
  protected abstract Set<FileURLValue> askForFile();

  /**
   * Classe de implementao do modo report.
   *
   * @author Tecgraf/PUC-Rio
   */
  final class FileListReportParameter extends JPanel implements
  FileListParameterComponent {

    /**
     * rea de texto de exibio
     */
    private JTextArea textArea;

    /**
     * Construtor
     */
    FileListReportParameter() {

      setLayout(new GridLayout());

      textArea = new JTextArea();
      textArea.setToolTipText(getParameter().getDescription());
      ComponentProperties.setProperties(textArea, Mode.REPORT, false);

      JScrollPane scroll = new JScrollPane(textArea);
      scroll
      .setBorder(ComponentProperties.getInstance(Mode.REPORT).getBorder());
      updateViewContents();
      add(scroll, new GBC(0, 0).both());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      List<FileURLValue> values = getParameter().getValue();
      if (values == null) {
        textArea.setRows(1);
      }
      else {
        textArea.setRows(Math.max(1, Math.min(values.size(), 6)));

        StringBuffer sb = new StringBuffer();
        Iterator<FileURLValue> valueIterator = values.iterator();
        while (valueIterator.hasNext()) {
          FileURLValue value = valueIterator.next();
          sb.append(value.getPath()).append('\n');
        }
        // remove o ltimo breakLine
        String text = sb.substring(0, sb.length() - 1);
        textArea.setText(text);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean enabled) {
      super.setEnabled(enabled);
      textArea.setEnabled(enabled);
    }
  }

  /**
   * Classe do parmetro
   *
   * @author Tecgraf/PUC-Rio
   */
  final class FileListConfigurationParameter extends JPanel implements
  FileListParameterComponent {

    /**
     * O Boto Adicionar.
     */
    final private JButton addButton = new JButton();

    /**
     * Indica se a viso est habilitada.
     */
    private boolean isEnabledView;

    /**
     * A lista de arquivos.
     */
    private JList list;

    /**
     * O modelo da lista.
     */
    private DefaultListModel model;

    /**
     * O Boto Remover.
     */
    final private JButton removeButton = new JButton();

    /**
     * O Boto up.
     */
    final private JButton upButton = new JButton();

    /**
     * O Boto down.
     */
    final private JButton dnButton = new JButton();

    /**
     * Construtor
     */
    FileListConfigurationParameter() {
      setLayout(new BorderLayout());

      model = new DefaultListModel();
      list = new JList(model);
      list.getSelectionModel().setSelectionMode(
        ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
      list.setToolTipText(getParameter().getDescription());
      list.addKeyListener(new KeyAdapter() {
        @Override
        public void keyPressed(KeyEvent e) {
          int keyCode = e.getKeyCode();
          switch (keyCode) {
            case KeyEvent.VK_BACK_SPACE:
            case KeyEvent.VK_DELETE:
              removeFiles();
          }
        }
      });
      list.addListSelectionListener(new ListSelectionListener() {
        @Override
        public void valueChanged(ListSelectionEvent e) {
          updateRemoveButton();
          updateUpDownButton();
        }
      });
      list.setToolTipText(getParameter().getDescription());
      JPanel editPanel = new JPanel();
      editPanel.setLayout(new BorderLayout());

      add(new JScrollPane(list), BorderLayout.CENTER);
      add(createButtonPanel(), BorderLayout.EAST);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      model.removeAllElements();
      List<FileURLValue> values = getParameter().getValue();
      if (values != null) {
        Iterator<FileURLValue> valueIterator = values.iterator();
        while (valueIterator.hasNext()) {
          FileURLValue value = valueIterator.next();
          model.addElement(value);
        }
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean isEnabled) {
      isEnabledView = isEnabled;
      list.setEnabled(isEnabled);
      addButton.setEnabled(isEnabled);
      updateRemoveButton();
      updateUpDownButton();
    }

    /**
     * Cria o painel com os botes Adicionar e Remover.
     *
     * @return .
     */
    private JPanel createButtonPanel() {
      final FileListParameter parameter = getParameter();

      addButton.setIcon(ApplicationImages.ICON_BROWSEFILE_16);
      addButton.setText(LNG.get(LNGPREFIX + "add.button.text"));
      addButton.setToolTipText(LNG.get(LNGPREFIX + "add.button.tooltip"));
      addButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          Set<FileURLValue> files = askForFile();
          if (files != null) {
            for (FileURLValue file : files) {
              parameter.addElement(file);
            }
          }
        }
      });

      removeButton.setIcon(ApplicationImages.ICON_DELETE_16);
      removeButton.setText(LNG.get(LNGPREFIX + "del.button.text"));
      removeButton.setToolTipText(LNG.get(LNGPREFIX + "del.button.tooltip"));
      removeButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          removeFiles();
        }
      });
      removeButton.setEnabled(false);

      upButton.setIcon(ApplicationImages.ICON_UP_16);
      upButton.setText(LNG.get(LNGPREFIX + "up.button.text"));
      upButton.setToolTipText(LNG.get(LNGPREFIX + "up.button.tooltip"));
      upButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          moveUpFile();
        }
      });
      upButton.setEnabled(false);

      dnButton.setIcon(ApplicationImages.ICON_DOWN_16);
      dnButton.setText(LNG.get(LNGPREFIX + "down.button.text"));
      dnButton.setToolTipText(LNG.get(LNGPREFIX + "down.button.tooltip"));
      dnButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          moveDownFile();
        }
      });
      dnButton.setEnabled(false);

      ClientUtilities.adjustEqualSizes(addButton, removeButton, upButton,
        dnButton);

      final JPanel buttonPanel = new JPanel();
      buttonPanel.setLayout(new GridBagLayout());
      buttonPanel.add(addButton, new GBC(0, 0).center().none().insets(2));
      buttonPanel.add(removeButton, new GBC(0, 1).center().none().insets(2));
      if (!parameter.mustSort()) {
        buttonPanel.add(new JLabel(), new GBC(0, 2).center().vertical().insets(
          2));
        buttonPanel.add(upButton, new GBC(0, 3).center().none().insets(2));
        buttonPanel.add(dnButton, new GBC(0, 4).center().none().insets(2));
      }
      return buttonPanel;
    }

    /**
     * Remove os arquivos selecionados.
     */
    private void removeFiles() {
      int[] selIdxs = list.getSelectedIndices();
      FileURLValue[] files = new FileURLValue[selIdxs.length];
      int i = 0;
      for (int selIdx : selIdxs) {
        final ListModel listModel = list.getModel();
        final FileURLValue file = (FileURLValue) listModel.getElementAt(selIdx);
        files[i++] = file;
      }

      final FileListParameter parameter = getParameter();
      for (FileURLValue file : files) {
        parameter.removeElementObj(file);
      }
    }

    /**
     * Remove os arquivos selecionados.
     */
    private void moveUpFile() {
      final int[] indexes = list.getSelectedIndices();
      final int n = indexes.length;
      if (n != 1) {
        return;
      }
      final int index = indexes[0];
      final FileListParameter parameter = getParameter();
      parameter.moveUp(index);
      list.setSelectedIndex(index - 1);
    }

    /**
     * Remove os arquivos selecionados.
     */
    private void moveDownFile() {
      final int[] indexes = list.getSelectedIndices();
      final int n = indexes.length;
      if (n != 1) {
        return;
      }
      final int index = indexes[0];
      final FileListParameter parameter = getParameter();
      parameter.moveDown(index);
      list.setSelectedIndex(index + 1);
    }

    /**
     * Atualiza o boto Remover.
     */
    private void updateRemoveButton() {
      removeButton.setEnabled(isEnabledView && !list.isSelectionEmpty());
    }

    /**
     * Atualiza os botes Up/Down.
     */
    private void updateUpDownButton() {
      final int[] selection = list.getSelectedIndices();
      final boolean flag =
        isEnabledView && !list.isSelectionEmpty() && (selection.length == 1);
      upButton.setEnabled(flag);
      dnButton.setEnabled(flag);
    }
  }
}