package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.List;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.AbstractSimpleApplicationPanel;
import csbase.client.applications.algorithmsmanager.AlgorithmsManager;
import csbase.client.applications.algorithmsmanager.actions.CommonManagementAction;
import csbase.client.applications.algorithmsmanager.models.DataInterface;

/**
 * Classe abstrata que define um painel com uma estrutura comum s
 * funcionalidade de gerenciamento de categorias e gerenciamento de algoritmos,
 * na aplicao Gerenciador de Algoritmos.
 * 
 * Esse painel possum um divisor (split) que separa o painel de seleo de dados
 *  esquerda do divisor, e o painel de edio do dado selecionado  direita do
 * divisor.
 * 
 * Essa classe deve ser especializada para que construa seu prprio painel de
 * seleo de dados e de edio do dado selecionado. As operaes que podem ser
 * realizadas sobre os dados selecionados so definidas na criao do prprio
 * painel de seleo.
 * 
 */
public abstract class CommonManagementPanel extends
  AbstractSimpleApplicationPanel<AlgorithmsManager> {

  /** Ao que criou esse painel */
  private CommonManagementAction action;

  /** Componente split do painel de gerncia de categorias */
  private AlgorithmsManagerSplitComponent splitComponent;

  /** Painel de gerncia de categorias de algoritmos */
  private CommonSelectionPanel selectionPanel;

  /**
   * Painel para manipulao das informaes do dado selecionado no painel de
   * seleo
   */
  private CommonEditTabbedPanel editPanel;

  /**
   * Painel para criao de um novo dado.  um painel de edio, mas que no
   * est associado a um item selecionado no painel de seleo.
   */
  private CommonEditTabbedPanel createPanel;

  /**
   * Constri o painel.
   * 
   * @param action ao que criou esse painel
   */
  public CommonManagementPanel(CommonManagementAction action) {
    super(action.getApplication());
    this.action = action;
    buildPanel();
  }

  /**
   * Obtm a ao de gerenciamento que criou esse painel.
   * 
   * @return a ao de gerenciamento
   */
  public CommonManagementAction getManagementAction() {
    return action;
  }

  /**
   * Obtm o painel de seleo dos dados a serem manipulados.
   * 
   * Considerando que h uma diviso (split) no painel de gerenciamento, o
   * painel de seleo se localiza  esquerda.
   * 
   * @return o painel de seleo de dados
   */
  public CommonSelectionPanel getSelectionPanel() {
    return (CommonSelectionPanel) selectionPanel;
  }

  /**
   * Obtm o painel de edio com as informaes do dado sobre o qual vai ser
   * realizada alguma operao de edio (por exemplo, incluso, atualizao ou
   * remoo).
   * 
   * Considerando que h uma diviso (split) no painel de gerenciamento, o
   * painel de seleo se localiza  esquerda.
   * 
   * @return o painel de edio do dado selecionado, ou null, caso no seja uma
   *         instncia da classe <code> CommonEditPanel </code>
   */
  public CommonEditTabbedPanel getEditPanel() {
    if (createPanel != null) {
      return createPanel;
    }
    else if (editPanel instanceof CommonEditTabbedPanel) {
      return (CommonEditTabbedPanel) editPanel;
    }
    return null;
  }

  /**
   * Constri um painel vazio referente a edio de nenhum dado vlido
   * selecionado no painel de seleo. Por exemplo, se for feita a seleo da
   * raiz de uma rvore de dados.
   * 
   * @return o painel vazio
   * 
   */
  protected JPanel buildEmptyPanel() {
    return new JPanel();
  }

  /**
   * Constri o painel para seleo dos dados a serem manipulados.
   * 
   * @return o painel para seleo dos dados
   */
  protected abstract CommonSelectionPanel buildSelectionPanel();

  /**
   * Constri o painel de edio com as informaes do dado sobre o qual vai ser
   * realizada alguma operao de edio (por exemplo, incluso, atualizao ou
   * remoo).
   * 
   * @return o painel para edio do dado
   */
  protected abstract CommonEditTabbedPanel buildEditPanel();

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.applications.AbstractSimpleApplicationPanel#buildPanel()
   */
  @Override
  protected void buildPanel() {
    setLayout(new GridBagLayout());
    setBorder(BorderFactory.createTitledBorder((String) action
      .getValue(Action.SHORT_DESCRIPTION)));

    splitComponent = getManageSplitComponent();
    add(splitComponent.getSplitPanel(), new GBC(0, 0).both().west().insets(5,
      5, 5, 5));
  }

  /**
   * Obtm o componente com diviso (split) entre os painis de seleo de dados
   * ( esquerda da diviso)). e o de edio do dado selecionado ( direita da
   * diviso
   * 
   * @return o componente com diviso dos painis de seleo e edio do dado
   */
  private AlgorithmsManagerSplitComponent getManageSplitComponent() {
    if (splitComponent == null) {
      selectionPanel = buildSelectionPanel();
      editPanel = buildEditPanel();
      splitComponent = new AlgorithmsManagerSplitComponent(getApplication(),
        selectionPanel, editPanel);
    }

    return splitComponent;
  }

  /**
   * Atualiza o painel de edio com as informaes do dado sobre o qual vai ser
   * realizada alguma operao de edio (por exemplo, incluso, atualizao ou
   * remoo).
   * 
   * Considerando que h uma diviso (split) no painel de gerenciamento, o
   * painel de edio se localiza  direita.
   * 
   * @param showInfo se true, indica que o painel de edio deve exibir as
   *        informaes pertinentes; se false, indica que nenhuma informao
   *        deve ser exibida (caso da exibio de um n raiz de uma rvore de
   *        categorias)
   */
  public void updateEditPanel(boolean showInfo) {
    createPanel = null;
    if (splitComponent != null) {
      if (!showInfo) {
        splitComponent.updateEditPanel(buildEmptyPanel());
      }
      else {
        // No reconstruir o painel sCommonEditTabbedPanelempre, evita o refresh
        // da tela de edio,
        // que d a sensao de piscar
        CommonEditTabbedPanel editPanel = getEditPanel();
        splitComponent.updateEditPanel(editPanel);
      }
    }
  }

  /**
   * Modifica o painel de edio do split para que contenha um determinado
   * painel de criao de um dado.
   * 
   * @param createPanel painel de criao de um dado
   */
  public void setCreatePanel(CommonEditTabbedPanel createPanel) {
    if (splitComponent != null) {
      if (createPanel == null) {
        this.createPanel = null;
        splitComponent.updateEditPanel(buildEmptyPanel());
      }
      else {
        this.createPanel = createPanel;
        splitComponent.updateEditPanel(createPanel);
        createPanel.initializeData();
      }
    }
  }

  /**
   * Obtm uma lista com os dados selecionados no painel de seleo de dados.
   * 
   * Considerando que h uma diviso no painel de gerenciamento, o painel de
   * seleo se localiza  esquerda.
   * 
   * @return uma lista com os dados selecionados para realizar alguma operao
   *         de edio, caso contrrio, retorna uma lista vazia
   */
  public List<DataInterface> getSelectedDataList() {
    return getSelectionPanel().getSelectedDataList();
  }

  /**
   * Obtm o primeiro dado selecionado no painel de seleo de dados.
   * 
   * Considerando que h uma diviso no painel de gerenciamento, o painel de
   * seleo se localiza  esquerda.
   * 
   * @return o dado selecionado para realizar alguma operao de edio, caso
   *         contrrio, retorna null
   */
  public DataInterface getFirstSelectedData() {
    return getSelectionPanel().getSelectedData();
  }

  /**
   * Reconstri o painel de seleo de dados. Em geral, isso precisa ser feito
   * quando chega um novo evento sobre o dado (por exemplo, dado criado,
   * removido ou alterado), que requer uma alterao no painel de seleo.
   * 
   * Considerando que h uma diviso (split) no painel de gerenciamento, o
   * painel de edio se localiza  direita.
   * 
   * @param panel painel que vai substituir o painel de seleo atual
   * 
   */
  public void updateSelectionPanel(JPanel panel) {
    splitComponent.updateSelectionPanel(panel);
  }
}
