package csbase.client.applications.flowapplication.graph.utils;

import java.util.ArrayList;
import java.util.List;

/**
 * Representa o resultado de uma operao de colar parmetros em um n. Essa
 * operao pode gerar avisos {@link OperationStatus#warnings} e erros
 * {@link OperationStatus#errors} e pode ser cancelada pelo usurio
 * {@link OperationStatus#confirmed} .
 * 
 * @author isabella
 */
public class OperationStatus {

  /**
   * Lista de avisos geradas durante a operao.
   */
  private List<String> warnings;
  /**
   * Lista de erros gerados durante a operao.
   */
  private List<String> errors;
  /**
   * Sinal de confirmao da operao
   */
  private boolean confirmed;

  /**
   * Construtor
   */
  public OperationStatus() {
    this.errors = new ArrayList<String>();
    this.warnings = new ArrayList<String>();
    this.confirmed = false;
  }

  /**
   * Retorna a lista de avisos geradas durante a operao.
   * 
   * @return warnings Lista de avisos
   */
  public List<String> getWarnings() {
    return warnings;
  }

  /**
   * Retorna a lista de erros gerados durante a operao.
   * 
   * @return errors Lista de erros
   */
  public List<String> getErrors() {
    return errors;
  }

  /**
   * Retorna o sinal de confirmao da operao. Verdadeiro se a operao no
   * gerou avisos e erros ou se o usurio confirmou a operao mesmo com os
   * avisos e erros. Falso caso o usurio tenha cancelado a operaco.
   * 
   * @return confirmPaste O Sinal de confirmao.
   */
  public boolean isConfirmed() {
    return confirmed;
  }

  /**
   * Indica se foi gerado algum erro durante a operao.
   * 
   * @return verdadeiro se houve algumerro ou falso, caso contrrio.
   */
  public boolean hasErrors() {
    return !errors.isEmpty();
  }

  /**
   * Indica se foi gerado algum aviso durante a operao.
   * 
   * @return verdadeiro se houve algum aviso ou falso, caso contrrio.
   */
  public boolean hasWarnings() {
    return !warnings.isEmpty();
  }

  /**
   * Adiciona um novo aviso em relao  operao.
   * 
   * @param warning Mensagem do aviso.
   */
  public void addWarning(String warning) {
    this.warnings.add(warning);
  }

  /**
   * Adiciona um novo erro em relao  operao.
   * 
   * @param error Mensagem de erro.
   */
  public void addError(String error) {
    this.errors.add(error);
  }

  /**
   * Atribui o sinal de confirmao da operao. Verdadeiro se a operao no
   * gerou avisos e erros ou se o usurio confirmou a operao mesmo com os
   * avisos e erros. Falso caso o usurio tenha cancelado a operaco.
   * 
   * @param confirmed O Sinal de confirmao.
   */
  public void setConfirmed(boolean confirmed) {
    this.confirmed = confirmed;
  }

}