package csbase.client.openbus;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;

import org.omg.CORBA.Any;
import org.omg.CORBA.ORB;
import org.omg.CORBA.StringHolder;

import csbase.exception.CSBaseException;
import scs.core.ComponentContext;
import scs.core.ComponentId;
import scs.core.IComponent;
import scs.core.IComponentHelper;
import scs.core.IComponentHolder;
import scs.core.exception.SCSException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.openbus.session_service.v1_05.ISession;
import tecgraf.openbus.session_service.v1_05.ISessionHelper;
import tecgraf.openbus.session_service.v1_05.ISessionService;
import tecgraf.openbus.session_service.v1_05.ISessionServiceHelper;
import tecgraf.openbus.session_service.v1_05.SessionEvent;
import tecgraf.openbus.session_service.v1_05.SessionEventSink;
import tecgraf.openbus.session_service.v1_05.SessionEventSinkHelper;
import tecgraf.openbus.session_service.v1_05.SessionEventSinkPOA;

public class OpenBusSession {
  /**
   * Separador
   */
  final static private String SEP = "___@@___";

  /**
   * Lista de recebedores de eventos.
   */
  final private List<OpenBusEventReceiverInterface> eventReceivers =
    new ArrayList<OpenBusEventReceiverInterface>();
  /**
   * O componente membro de uma sesso do OpenBus (opcional).
   */
  private IComponent sessionMember;

  /**
   * O identificador do membro da sesso do usurio.
   */
  private String sessionMemberIdentifier;

  /**
   * A sesso do usurio no OpenBus.
   */
  private ISession session;

  /**
   * O canal de eventos da sesso.
   */
  private SessionEventSink sessionEventSink;

  /**
   * O servant do tratador de eventos de sesso.
   */
  private SessionEventSinkPOA sessionMemberEventSinkServant;

  /**
   * Construtor.
   * 
   * @param sessionEventSinkClass o tratador de eventos da sesso.
   * 
   * @throws CSBaseException em caso de erro.
   */
  public OpenBusSession(
    Class<? extends SessionEventSinkPOA> sessionEventSinkClass)
    throws CSBaseException {
    if (sessionEventSinkClass != null) {
      try {
        this.sessionMemberEventSinkServant =
          sessionEventSinkClass.newInstance();
      }
      catch (InstantiationException e) {
        throw new CSBaseException(e);
      }
      catch (IllegalAccessException e) {
        throw new CSBaseException(e);
      }
      ComponentId componentId =
        new ComponentId("SessionEventSink", (byte) 1, (byte) 0, (byte) 0,
          "Java");
      ComponentContext context;
      try {
        context =
          new ComponentContext(OpenBusAccessPoint.getInstance().getORB(),
            OpenBusAccessPoint.getInstance().getRootPOA(), componentId);
        context.addFacet("SessionEventSink", SessionEventSinkHelper.id(),
          sessionMemberEventSinkServant);
        org.omg.CORBA.Object obj = context.getIComponent();
        this.sessionMember = IComponentHelper.narrow(obj);
      }
      catch (SCSException e) {
        throw new CSBaseException(e);
      }
    }

    ISessionService ss = getSessionService();
    if (ss == null) {
      throw new CSBaseException();
    }

    if (this.sessionMember != null) {
      org.omg.CORBA.Object obj;

      IComponent sessionComponent = ss.getSession();
      if (sessionComponent == null) {
        IComponentHolder sessionComponentHolder = new IComponentHolder();
        StringHolder memberIdentifierHolder = new StringHolder();
        final boolean created =
          ss.createSession(this.sessionMember, sessionComponentHolder,
            memberIdentifierHolder);
        if (created) {
          sessionComponent = sessionComponentHolder.value;
          obj = sessionComponent.getFacet(ISessionHelper.id());
          this.session = ISessionHelper.narrow(obj);
          this.sessionMemberIdentifier = memberIdentifierHolder.value;
        }
        else {
          throw new IllegalStateException("Cannot create OpenBus session!");
        }
      }
      else {
        obj = sessionComponent.getFacet(ISessionHelper.id());
        this.session = ISessionHelper.narrow(obj);
        this.sessionMemberIdentifier = session.addMember(this.sessionMember);
      }

      obj = sessionComponent.getFacet(SessionEventSinkHelper.id());
      this.sessionEventSink = SessionEventSinkHelper.narrow(obj);
    }
  }

  /**
   * Recupera o servio de sesso que est disponvel do barramento. Para buscar
   * esse servio de sesso usa-se o servio de registros do barramento.
   * 
   * @return o objeto que implementa a interface <code>ISessionService</code>.
   * 
   * @throws CSBaseException em caso de erro no OpenBus.
   */
  private ISessionService getSessionService() throws CSBaseException {
    Properties props = new Properties();
    props.put("openbus.component.interface", ISessionServiceHelper.id());
    IComponent component = OpenBusAccessPoint.getInstance().findService(props);
    if (component == null) {
      throw new CSBaseException(getString("service.error"));
    }
    org.omg.CORBA.Object facet = component.getFacet(ISessionServiceHelper.id());
    ISessionService sessionService = ISessionServiceHelper.narrow(facet);
    return sessionService;
  }

  /**
   * Obtm o canal de eventos da sesso.
   * 
   * @return O canal de eventos da sesso.
   */
  public SessionEventSink getSessionEventSink() {
    return this.sessionEventSink;
  }

  /**
   * Obtm o identificador do membro da sesso.
   * 
   * @return O identificador do membro da sesso.
   */
  public String getSessionMemberIdentifier() {
    return this.sessionMemberIdentifier;
  }

  /**
   * Obtm o servant do tratador de eventos de sesso.
   * 
   * @return O servant do tratador de eventos de sesso.
   */
  public SessionEventSinkPOA getSessionMemberEventSinkServant() {
    return this.sessionMemberEventSinkServant;
  }

  /**
   * Tratador de resultado de tramantos de eventos recebidos.
   * 
   * @param treated indicativo de tratamento
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param type tipo
   * @param value valor
   */
  final protected void signalEventReceived(boolean treated, String sourceId,
    String destId, String type, Object value) {
    final String msg =
      createEventDebugString("RECEIVED", treated, sourceId, destId, type, value);
    System.out.println(msg);
  }

  /**
   * Montagem de texto de depurao de eventos.
   * 
   * @param prefix prefixo a ser adicionado.
   * @param treated indicativo de tratamento
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param type tipo
   * @param value valor
   * @return texto.
   */
  private String createEventDebugString(String prefix, boolean treated,
    String sourceId, String destId, String type, Object value) {
    String className = "NULL";
    String valueText = "null";
    if (value != null) {
      className = value.getClass().getSimpleName();
      valueText = value.toString();
    }
    final Date now = new Date();
    final String fmt = "%1$tH:%1$tM:%1$tS:%1$tL - %1$td/%1$tm";
    final String timePart = String.format(fmt, now);

    final String headerPart = "[" + timePart + "] " + prefix + " EVENT ";
    final String pathPart = sourceId + " ---> " + destId + " / ";
    final String treatedPart = type + " (treated = " + treated + ") ";
    final String initPart = headerPart + pathPart + treatedPart;
    final String valuePart =
      "- [" + className + "]; value = \"" + valueText + "\"";
    final String msg = initPart + valuePart;
    return msg;
  }

  /**
   * Tratador de resultado de tramantos de eventos enviados.
   * 
   * @param treated indicativo de tratamento
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param type tipo
   * @param value valor
   */
  private void signalEventSent(boolean treated, String sourceId, String destId,
    String type, Object value) {
    final String msg =
      createEventDebugString("SENT", treated, sourceId, destId, type, value);
    System.out.println(msg);
  }

  /**
   * Retorna a lista de receptores de eventos.
   * 
   * @return a lista
   */
  final public List<OpenBusEventReceiverInterface> getEventReceivers() {
    synchronized (eventReceivers) {
      return eventReceivers;
    }
  }

  /**
   * Adiciona receptor de eventos.
   * 
   * @param receiver receptor
   */
  final public void addEventReceiver(OpenBusEventReceiverInterface receiver) {
    if (receiver == null) {
      final String err = getString("null.receiver.error");
      throw new IllegalArgumentException(err);
    }
    synchronized (eventReceivers) {
      eventReceivers.add(receiver);
    }
  }

  /**
   * Retira receptor de eventos.
   * 
   * @param receiver receptor
   */
  final public void delEventReceiver(OpenBusEventReceiverInterface receiver) {
    if (receiver == null) {
      final String err = getString("null.receiver.error");
      throw new IllegalArgumentException(err);
    }
    synchronized (eventReceivers) {
      eventReceivers.remove(receiver);
    }
  }

  /**
   * Cria um any CORBA
   * 
   * @return o any.
   */
  private Any createAny() {
    final ORB orb = OpenBusAccessPoint.getInstance().getORB();
    final Any any = orb.create_any();
    return any;
  }

  /**
   * @see OpenBusEventSenderInterface#sendEventWithNoValue(String, String,
   *      String)
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param type tipo
   * @return indicativo
   */
  boolean internalSendEventWithNoValue(String sourceId, String destId,
    String type) {
    final Any any = createAny();
    final String rawType = insertPathTypeData(sourceId, destId, type);
    sessionEventSink.push(sessionMemberIdentifier, new SessionEvent(rawType,
      any));

    System.out.println(rawType);

    signalEventSent(true, sourceId, destId, type, null);
    return true;
  }

  /**
   * @see OpenBusEventSenderInterface#sendEventWithStringValue(String, String,
   *      String, String)
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param type tipo
   * @param stringValue valor
   * @return indicativo
   */
  boolean internalSendEventWithStringValue(String sourceId, String destId,
    String type, String stringValue) {
    final Any any = createAny();
    any.insert_string(stringValue);
    final String rawType = insertPathTypeData(sourceId, destId, type);
    sessionEventSink.push(sessionMemberIdentifier, new SessionEvent(rawType,
      any));
    signalEventSent(true, sourceId, destId, type, stringValue);

    return true;
  }

  /**
   * Faz check de runtime de string para composio de fonte, destino e tipo.
   * 
   * @param string texto.
   */
  private void checkHasNoSeparatorString(final String string) {
    if (string.matches("")) {
      final String fmt = "Bad string for event mount: %s";
      final String err = String.format(fmt, string);
      throw new IllegalArgumentException(err);
    }
  }

  /**
   * Faz check de runtime de string no nula.
   * 
   * @param string string a ser testada.
   * @param text indicativo de localizao do erro.
   */
  private void checkNotNullString(final String string, final String text) {
    if (string == null) {
      final String fmt = "Null string event (part: %s) detected!";
      final String err = String.format(fmt, text);
      throw new IllegalArgumentException(err);
    }
  }

  /**
   * Monta tipo (raw) a ser enviado pelo OpenBis com base em fonte, destino e
   * tipo
   * 
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param type tipo
   * @return tipo (raw).
   */
  private String insertPathTypeData(String sourceId, String destId, String type) {
    checkNotNullString(type, "type");
    checkHasNoSeparatorString(sourceId);
    checkHasNoSeparatorString(destId);
    checkHasNoSeparatorString(type);

    final String sId = (sourceId == null ? "" : sourceId);
    final String dId = (destId == null ? "" : destId);

    final String typePart = type.trim();
    final String sourceIdPart = sId.trim();
    final String destIdPart = dId.trim();

    final String rawType = sourceIdPart + SEP + destIdPart + SEP + typePart;
    // System.out.println("$$$ INSERT: " + sourceId + ";" + destId + ";" + type
    // + " ---> " + rawType);
    return rawType;
  }

  /**
   * @param receivedRawType o tipo recebido (raw) do evento.
   * @return um array com as informaes
   */
  String[] extractPathTypeData(final String receivedRawType) {
    checkNotNullString(receivedRawType, "raw-type");
    final String rawType = receivedRawType.trim();
    final String[] array = rawType.split(SEP);
    final String sourceId;
    final String destId;
    final String type;
    if (array != null && array.length == 3) {
      final String arg0 = array[0];
      final String arg1 = array[1];
      final String arg2 = array[2];
      checkNotNullString(arg0, "source");
      checkNotNullString(arg1, "destination");
      checkNotNullString(arg2, "type");
      sourceId = arg0.trim();
      destId = arg1.trim();
      type = arg2.trim();
      if (type.isEmpty()) {
        final String err = "Empty converted type (\"\") not allowed!";
        throw new IllegalArgumentException(err);
      }
    }
    else {
      sourceId = null;
      destId = null;
      type = rawType;
    }
    final String[] extracted = new String[] { sourceId, destId, type };

    // System.out.println("$$$ EXTRACT: " + rawType + " ---> " + sourceId + ";"
    // + destId + ";" + type);
    return extracted;
  }

  /**
   * @see OpenBusEventSenderInterface#sendEventWithBooleanValue(String, String,
   *      String, boolean)
   * @param sourceId id da fonte.
   * @param destId id do destino.
   * @param type tipo
   * @param booleanValue valor
   * @return indicativo
   */
  boolean internalSendEventWithBooleanValue(String sourceId, String destId,
    String type, boolean booleanValue) {
    final Any any = createAny();
    any.insert_boolean(booleanValue);
    final String rawType = insertPathTypeData(sourceId, destId, type);
    sessionEventSink.push(sessionMemberIdentifier, new SessionEvent(rawType,
      any));
    signalEventSent(true, sourceId, destId, type, booleanValue);
    return true;
  }

  /**
   * Consulta texto internacionalizado.
   * 
   * @param key chave
   * @return texto.
   */
  private String getString(String key) {
    final String prefix = this.getClass().getSimpleName();
    return LNG.get(prefix + "." + key);
  }
}
