/*
 * $Id: FileTransferClient.java 116274 2011-03-17 14:15:13Z clinio $
 */

package csbase.client.applications.filetransferclient;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.MenuButton;
import tecgraf.javautils.gui.MenuButton.PopupPosition;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.filetransferclient.actions.ConnectionOffAction;
import csbase.client.applications.filetransferclient.actions.ConnectionOnAction;
import csbase.client.applications.filetransferclient.actions.DownloadAction;
import csbase.client.applications.filetransferclient.actions.FastConnectAction;
import csbase.client.applications.filetransferclient.actions.ToggleDotFilterLocalAction;
import csbase.client.applications.filetransferclient.actions.ToggleDotFilterRemoteAction;
import csbase.client.applications.filetransferclient.actions.UploadAction;
import csbase.client.applications.filetransferclient.dialogs.ConnectionFrame;
import csbase.client.applications.filetransferclient.panels.localpanel.FileTransferClientLocalPanel;
import csbase.client.applications.filetransferclient.panels.queuepanel.FileTransferClientQueuePanel;
import csbase.client.applications.filetransferclient.panels.remotepanel.FileTransferClientRemotePanel;
import csbase.client.desktop.DesktopFrame;
import csbase.logic.CommonClientProject;
import csbase.logic.filetransferservice.FileTransferConnection;
import csbase.logic.filetransferservice.FileTransferPredefinedConnection;
import csbase.remote.FileTransferServiceInterface;

/**
 * Classe que representa a aplio de transferidor de arquivos multi-protocolo.
 * 
 * @author Tecgraf/PUC-Rio
 */
final public class FileTransferClient extends ApplicationProject {

  /**
   * Thread de atualizaco da lista de requisicoes.
   */
  final private FileTransferQueueThread thread;

  /**
   * Conexo corrente
   */
  private FileTransferConnection connection;

  /**
   * Proxy para o servio.
   */
  final private FileTransferProxy serviceProxy;

  /**
   * Painel remoto
   */
  final private FileTransferClientRemotePanel remotePanel;

  /**
   * Painel local
   */
  final private FileTransferClientLocalPanel localPanel;

  /**
   * Painel da fila de requisies
   */
  final private FileTransferClientQueuePanel queuePanel;

  /**
   * Lista de conexes predefinidas
   */
  final private List<FileTransferPredefinedConnection> predefinedConnections;

  /**
   * Dilogo nico de conexo.
   */
  private ConnectionFrame connectionFrame;

  /**
   * Botao de download
   */
  final private JButton downloadButton = new JButton();

  /**
   * Boto de upload
   */
  final private JButton uploadButton = new JButton();

  /**
   * Boto de conexes rpidas
   */
  final private MenuButton fastConnectButton =
    new MenuButton(FileTransferClientUI.FAST_CONNECT_ICON, PopupPosition.BOTTOM);

  /**
   * Menu de conexes rpidas.
   */
  final private JMenu fastConnectItem = new JMenu();

  /**
   * Criao do painel de botes.
   * 
   * @return o painel.
   */
  private JPanel buildButtonsPanel() {
    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());

    downloadButton.setAction(new DownloadAction(this));
    uploadButton.setAction(new UploadAction(this));

    panel.add(new JPanel(), new GBC(0, 0).vertical().weighty(2.0));
    panel.add(downloadButton, new GBC(0, 1).none());
    panel.add(new JPanel(), new GBC(0, 2).vertical().weighty(1.0));
    panel.add(uploadButton, new GBC(0, 3).none());
    panel.add(new JPanel(), new GBC(0, 4).vertical().weighty(2.0));

    final JComponent[] buttons =
      new JComponent[] { uploadButton, downloadButton };
    GUIUtils.matchPreferredSizes(buttons);

    return panel;
  }

  /**
   * Constroi o menu "conexo"
   * 
   * @return o menu
   */
  private JMenu buildConectionMenu() {
    final JMenu menu = new JMenu(getString("menu.connection"));

    fastConnectItem.setText(getString("menu.connection.fast"));
    final JMenuItem manualConnectItem = menu.add(new ConnectionOnAction(this));
    manualConnectItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_M,
      InputEvent.CTRL_DOWN_MASK));

    menu.add(manualConnectItem);
    menu.add(fastConnectItem);
    menu.add(new ConnectionOffAction(this));
    menu.addSeparator();
    menu.add(new ApplicationExitAction(this));
    return menu;
  }

  /**
   * Constroi o menu de configurao
   * 
   * @return o menu
   */
  private JMenu buildConfigurationMenu() {
    final JMenu menu = new JMenu(getString("menu.configuration"));

    final JCheckBoxMenuItem localDotItem = new JCheckBoxMenuItem();
    localDotItem.setSelected(true);
    localDotItem.setAction(new ToggleDotFilterLocalAction(this));
    menu.add(localDotItem);

    final JCheckBoxMenuItem remoteDotItem = new JCheckBoxMenuItem();
    remoteDotItem.setSelected(true);
    remoteDotItem.setAction(new ToggleDotFilterRemoteAction(this));
    menu.add(remoteDotItem);

    return menu;
  }

  /**
   * Constroi o menu de help
   * 
   * @return o menu
   */
  private JMenu buildHelpMenu() {
    final JMenu helpMenu = new JMenu(getString("menu.help"));
    helpMenu.add(new ApplicationAboutAction(this));
    return helpMenu;
  }

  /**
   * Criao do painel principal da aplicao.
   * 
   * @return o painel principal
   */
  private JSplitPane buildMainPanel() {
    final JPanel buttonsPanel = buildButtonsPanel();
    final JToolBar toolBar = buildToolBar();

    final JPanel transfPanel = new JPanel(new GridBagLayout());
    transfPanel.add(localPanel, new GBC(0, 0).both().weightx(20.0));
    transfPanel.add(buttonsPanel, new GBC(1, 0).both().weightx(1.0));
    transfPanel.add(remotePanel, new GBC(2, 0).both().weightx(20.0));
    transfPanel.setPreferredSize(new Dimension(800, 350));

    final JPanel panel = new JPanel(new GridBagLayout());
    panel.add(toolBar, new GBC(0, 0).horizontal());
    panel.add(transfPanel, new GBC(0, 1).both());

    final JSplitPane split = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
    split.add(panel);
    split.add(queuePanel);
    split.setDividerLocation(0.9);
    split.setResizeWeight(0.9);
    split.setOneTouchExpandable(true);

    queuePanel.setPreferredSize(new Dimension(800, 100));

    return split;
  }

  /**
   * Criao da barra de menus da aplicao. A barra de menus contm apenas os
   * menus Arquivo e Ajuda
   * 
   * @return a barra de menus da aplicao
   */
  private JMenuBar buildMenuBar() {
    final JMenuBar menuBar = new JMenuBar();

    final JMenu connMenu = buildConectionMenu();
    final JMenu confMenu = buildConfigurationMenu();
    final JMenu helpMenu = buildHelpMenu();

    menuBar.add(connMenu);
    menuBar.add(confMenu);
    menuBar.add(helpMenu);

    return menuBar;
  }

  /**
   * Constroi a toolbar.
   * 
   * @return a barra
   */
  private JToolBar buildToolBar() {
    final JToolBar tb = new JToolBar(SwingConstants.HORIZONTAL);
    tb.setFloatable(false);
    tb.add(new ConnectionOnAction(this));
    tb.add(new ConnectionOffAction(this));
    tb.addSeparator();
    tb.add(fastConnectButton);
    tb.addSeparator();
    tb.add(new ApplicationAboutAction(this));
    return tb;
  }

  /**
   * @return a conexo
   */
  final public FileTransferConnection getConnection() {
    return connection;
  }

  /**
   * @return o painel local.
   */
  public FileTransferClientLocalPanel getLocalPanel() {
    return localPanel;
  }

  /**
   * @return o campo predefinedConnections
   */
  public final List<FileTransferPredefinedConnection> getPredefinedConnections() {
    return predefinedConnections;
  }

  /**
   * Consulta o identificador do projeto.
   * 
   * @return o identificador do projeto
   */
  final public Object getProjectId() {
    final DesktopFrame mainFrame = DesktopFrame.getInstance();
    final CommonClientProject project = mainFrame.getProject();
    return project.getId();
  }

  /**
   * @return o valor de queuePanel.
   */
  public final FileTransferClientQueuePanel getQueuePanel() {
    return queuePanel;
  }

  /**
   * @return o painel remoto
   */
  public FileTransferClientRemotePanel getRemotePanel() {
    return remotePanel;
  }

  /**
   * Consulta proxy do servio
   * 
   * @return o proxy
   */
  public final FileTransferProxy getServiceProxy() {
    return serviceProxy;
  }

  /**
   * Retorna a janela.
   * 
   * @return a janela
   */
  final public ConnectionFrame getSingleConnectionFrame() {
    if (connectionFrame == null) {
      connectionFrame = new ConnectionFrame(this);
    }
    return connectionFrame;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    if (thread != null) {
    thread.killThread();
    }
  }

  /**
   * Faz a carga de aes no menu button
   */
  private void loadFastButtonActions() {
    final int[] keys =
      new int[] { KeyEvent.VK_1, KeyEvent.VK_2, KeyEvent.VK_3, KeyEvent.VK_4,
          KeyEvent.VK_5, KeyEvent.VK_6, KeyEvent.VK_7, KeyEvent.VK_8,
          KeyEvent.VK_9 };

    int i = 0;
    for (final FileTransferPredefinedConnection predef : predefinedConnections) {
      final FastConnectAction action = new FastConnectAction(this, predef);
      final JMenuItem itMenu = new JMenuItem(action);
      final JMenuItem itButton = new JMenuItem(action);
      if (i < keys.length) {
        final KeyStroke keyStroke = KeyStroke.getKeyStroke(keys[i],
          InputEvent.CTRL_DOWN_MASK);
        itMenu.setAccelerator(keyStroke);
        itButton.setAccelerator(keyStroke);
      }
      fastConnectButton.add(itButton);
      fastConnectItem.add(itMenu);
      i++;
    }

    if (predefinedConnections.size() == 0) {
      fastConnectButton.setEnabled(false);
      fastConnectItem.setEnabled(false);
    }
  }

  /**
   * @param connection a conexo
   */
  final public void setConnection(final FileTransferConnection connection) {
    this.connection = connection;

    if (connection == null) {
      downloadButton.setEnabled(false);
      uploadButton.setEnabled(false);
    }
    else {
      downloadButton.setEnabled(true);
      uploadButton.setEnabled(true);
    }

    localPanel.setLocalDirectory(null);

    remotePanel.setConnection(connection);
    remotePanel.setRemoteDirectory(null);
  }

  /**
   * Exibe dilogo de exceo (amigvel ao usurio)
   * 
   * @param e exceo
   */
  public void showExceptionDialog(final Exception e) {
    final String yes = getString("error.dialog.yes.option");
    final String no = getString("error.dialog.no.option");
    final String title = getString("error.dialog.title");
    final String msg = getString("error.dialog.message");
    final String question = getString("error.dialog.question");
    final String exceptionMsg = e.getMessage();
    final String text = msg + "\n\n" + exceptionMsg + "\n\n+" + question;
    final Object[] options = { yes, no };
    final int op =
      JOptionPane.showOptionDialog(getApplicationFrame(), text, title,
        JOptionPane.DEFAULT_OPTION, JOptionPane.ERROR_MESSAGE, null, options,
        options[1]);
    if (op == 0) {
      showException(exceptionMsg, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userCanKillApplication() {
    return true;
  }

  /**
   * Construtor da aplicao de transferncia de arquivos via FTP.
   * 
   * @param id o identificador da aplicao
   */
  public FileTransferClient(final String id) {
    super(id);

    final FileTransferServiceInterface fileTransferService = FileTransferProxy.getFileTransferService();
    if (fileTransferService != null) {
      this.thread = new FileTransferQueueThread(this);
      thread.start();
    }
    else {
      this.thread = null;
    }

    serviceProxy = new FileTransferProxy(this);
    predefinedConnections = serviceProxy.getPedefinedConnections();
    loadFastButtonActions();
    
    final String serverName = FileTransferProxy.getServerName();
    remotePanel = new FileTransferClientRemotePanel(this);
    localPanel = new FileTransferClientLocalPanel(this, serverName);
    queuePanel = new FileTransferClientQueuePanel(this);

    final JFrame mainFrame = getApplicationFrame();
    mainFrame.setJMenuBar(buildMenuBar());
    final JSplitPane split = buildMainPanel();
    mainFrame.getContentPane().add(split);
    mainFrame.pack();

    setConnection(null);
    queuePanel.refresh();
  }
}
