/*
 * $Id$
 */

package csbase.client.applications.algorithmsmanager.versiontree.actions;

import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.rmi.RemoteException;
import java.security.AccessControlException;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTree;
import csbase.client.desktop.LocalTask;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.client.externalresources.LocalFile;
import csbase.client.externalresources.StandaloneLocalFile;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.SingletonFileChooser;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.FileInfo;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.Utilities;
import csbase.util.FileSystemUtils;
import csbase.util.Unzip;

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         Ao abstrata de exportao de arquivo.
 */
public abstract class AbstractExportFileAction extends
  AbstractVersionTreeNodeAction {

  /**
   * Arquivo a ser exportado.
   */
  private final FileInfo source;

  /**
   * Construtor.
   * 
   * @param tree rvore que detm o n fonte da ao.
   * @param name Nome da ao.
   * @param file Arquivo a ser exportado.
   */
  public AbstractExportFileAction(VersionTree tree, String name, FileInfo file) {
    super(tree, name);

    this.source = file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    if (FileSystemUtils.canRead()) {
      fileSystemExport();
    }
    else {
      /**
       * Exibe uma pgina para upload de um arquivo de documentao, a partir do
       * disco rgido do usurio para o servidor. Essa funcionalidade s est
       * disponvel via applet.
       */
      appletExport(source); // #TODO A remover quando todos sistemas CSBase
                            // forem assinados
    }
  }

  /**
   * Exportao via applet.
   * 
   * @param source Arquivo a ser exportado.
   */
  protected abstract void appletExport(FileInfo source);

  /**
   * Exportao via aplicao.
   * 
   * @param source Arquivo a ser exportado.
   * @return Informao para construo de um canal remoto.
   * @throws RemoteException EM caso de falha na operao.
   */
  protected abstract RemoteFileChannelInfo prepareDownload(FileInfo source)
    throws RemoteException;

  /**
   * Obtm o arquivo a ser exportado na rvore de algoritmos e o arquivo de
   * destino no sistema de arquivos do usurio. Cria ento uma
   * {@link csbase.client.desktop.LocalTask} para exportar o arquivo.
   */
  private void fileSystemExport() {
    final LocalFile targetFile = getTargetFile();
    if (targetFile == null) {
      return;
    }
    if (targetFile.exists()) {
      if (!confirmOverwrite(targetFile)) {
        return;
      }
    }
    final RemoteFileChannelInfo info = getRequestInfo();
    if (info == null) {
      return;
    }
    Task<Void> task = new LocalTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        SyncRemoteFileChannel channel = new SyncRemoteFileChannel(info);
        channel.open(true);
        long size = channel.getSize();
        OutputStream target = targetFile.getOutputStream(false);
        channel.syncTransferTo(0, size, target);
        target.flush();
        target.close();
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof IOException) {
          error.printStackTrace();
          StandardDialogs.showErrorDialog(getWindow(), getName(),
            LNG.get("algomanager.error.download_fatal"));
          return;
        }
        super.handleError(error);
      }
    };
    String waitMsg = LNG.get("algomanager.msg.download_wait");
    boolean exportResult =
      task.execute(getWindow(), getName(), waitMsg, false, true);

    // task para unzip o arquivo
    if (exportResult && source.isDirectory()) {
      Task<Void> unzipTask = new LocalTask<Void>() {

        @Override
        protected void performTask() throws Exception {
          extractZipFile(targetFile);
        }

        @Override
        protected void handleError(Exception error) {
          if (error instanceof IOException) {
            StandardDialogs.showErrorDialog(getWindow(), getName(), LNG
              .get("algomanager.error.download.extract_zip_file"));
            return;
          }
          super.handleError(error);
        }
      };
      String unzipWaitMsg = LNG.get("algomanager.msg.unzip_wait");
      unzipTask.execute(getWindow(), getName(), unzipWaitMsg, false, false);
    }
  }

  /**
   * Extrai o contedo de um arquivo zip.
   *
   * @param zipLocalFile nome do arquivo zip local no cliente que foi exportado.
   * @throws IOException Quando o arquivo zip for invlido ou no conseguir
   *         descompactar o arquivo.
   */
  private void extractZipFile(LocalFile zipLocalFile) throws IOException {
    File zipFile = new File(zipLocalFile.getAbsolutePath());
    try {
      Unzip unzip = new Unzip(zipFile);
      if (unzip.listZipEntries().isEmpty()) {
        String msg =
          String.format(LNG.get("algomanager.error.unzip_io"), zipFile
            .getName());
        throw new IOException(msg);
      }
      unzip.decompress(zipFile.getParentFile(), true);
    }
    finally {
      FileUtils.delete(zipFile);
    }
  }

  /**
   * Exibe um dilogo para confirmar com o usurio se este deseja sobrescrever o
   * arquivo existente.
   * 
   * @param targetFile arquivo de destino.
   * 
   * @return <code>true</code> se o usurio confirmar, <code>false</code> caso
   *         contrrio.
   */
  private boolean confirmOverwrite(LocalFile targetFile) {
    String msg;
    try {
      msg =
        String.format(LNG.get("algomanager.msg.confirm.file_exists"),
          targetFile.getName());
      int answer = StandardDialogs.showYesNoDialog(getWindow(), getName(), msg);
      return (answer == JOptionPane.YES_OPTION);
    }
    catch (IOException ex) {
      msg = LNG.get("algomanager.error.download_io");
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(), msg, ex);
      return false;
    }
  }

  /**
   * Retorna o arquivo de destino local para a importao (se existir, ser
   * sobrescrito).
   * 
   * @return arquivo de destino local.
   */
  private LocalFile getTargetFile() {
    /**
     * Acesso ao sistema de arquivos permitido: obtm arquivos com o
     * JFileChooser.
     */
    try {
      JFileChooser chooser = SingletonFileChooser.getInstance();
      chooser.setMultiSelectionEnabled(false);

      int result = JFileChooser.CANCEL_OPTION;
      if (!source.isDirectory()) {
        File suggestedFileName =
          new File(chooser.getCurrentDirectory() + File.separator
            + source.getName());
        chooser.setSelectedFile(suggestedFileName);
        chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
        result = chooser.showSaveDialog(getWindow());
      }
      else {
        chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        result = chooser.showOpenDialog(getWindow());
      }

      File target = null;
      if (result == JFileChooser.APPROVE_OPTION) {
        if (!source.isDirectory()) {
          target = new File(chooser.getSelectedFile().getAbsolutePath());
        }
        else {
          target =
            new File(chooser.getSelectedFile() + File.separator
              + source.getName() + ".zip");
        }
        return new StandaloneLocalFile(target);
      }
      return null;
    }
    catch (AccessControlException ex) {
      // #TODO JWS
      return null;
    }
  }

  /**
   * Retorna um {@link RemoteFileChannelInfo}, isto , um objeto contendo as
   * informaes necessrias para solicitar a transferncia.
   * 
   * @return objeto contendo as informaes necessrias para a transferncia.
   */
  private RemoteFileChannelInfo getRequestInfo() {
    Task<RemoteFileChannelInfo> task = new RemoteTask<RemoteFileChannelInfo>() {
      @Override
      protected void performTask() throws Exception {
        setResult(prepareDownload(source));
      }
    };

    String waitMsg = LNG.get("algomanager.msg.upload_wait");
    if (!task.execute(getWindow(), getName(), waitMsg, false, true)) {
      return null;
    }
    return task.getResult();
  }

  /**
   * Carrega o arquivo atualmente selecionado em um navegador, permitindo sua
   * visualizao e salvamento no disco rgido. Obs: Funcionalidade disponvel
   * somente via applet.
   * 
   * @param algoId .
   * @param filePath .
   * @param dialogTitle .
   */
  protected void export(Object algoId, String filePath, String dialogTitle) {
    if (ExternalResources.getInstance().isEnabled()) {
      String[] filePathArray = Utilities.splitProjectPath(filePath);
      String urlStr =
        AlgorithmManagementProxy.retrieveDownloadURL(algoId, filePathArray,
          getWindow());
      if (urlStr == null) {
        return;
      }
      try {
        URL url = new URL(urlStr);
        ExternalResources.getInstance().showDocument(url);
      }
      catch (Exception ex) {
        StandardErrorDialogs.showErrorDialog(getWindow(), dialogTitle, ex);
      }
    }
    else {
      StandardDialogs.showInfoDialog(getWindow(), dialogTitle,
        LNG.get("algomanager.error.download_not_available"));
    }
  }

  /**
   * Obtm o path de um arquivo de origem considerando o separador '/'
   *
   * @param source arquivo a ser exportado
   *
   * @return path de um arquivo de origem considerando o separador '/'
   */
  protected String getSourcePath(FileInfo source) {
    // FIXME:
    // Mtodo dever ser substitudo posteriormente para se obter de
    // alguma forma o separador do SERVIDOR (Win x Linux).
    // Uma outra forma  alterar no prprio servio de algoritmos passando um
    // array de Strings.
    //
    // @see AlgorithmService#prepareDownloadExecFile(Object algoId, Object
    // versionId, String platformName, String fileName)
    String[] path = source.getPathAsArray();
    StringBuilder sourcePath = new StringBuilder();
    for (int i = 0; i < path.length; i++) {
      if (i > 0) {
        sourcePath.append("/");
      }
      sourcePath.append(path[i]);
    }
    return sourcePath.toString();
  }
}
