/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.actions;

import java.awt.event.ActionEvent;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.commandsmonitor.CommandsMonitor;
import csbase.client.remote.srvproxies.SGAProxy;
import csbase.client.remote.srvproxies.SchedulerProxy;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;

/**
 * Interrompe a execuo dos comandos selecionados.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class KillCommandsAction extends AbstractCommandsAction {

  /**
   * Construtor.
   * 
   * @param application Aplicao que detm esta ao.
   */
  public KillCommandsAction(CommandsMonitor application) {
    super(application, true, ApplicationImages.ICON_STOP_16);
  }

  /**
   * Filtro indicando que esta ao aceita qualquer comando que esteja no
   * processo de execuo.<br>
   * Os estados so:
   * <ul>
   * <li>{@link CommandStatus#UPLOADING}</li>
   * <li>{@link CommandStatus#INIT}</li>
   * <li>{@link CommandStatus#EXECUTING}</li>
   * <li>{@link CommandStatus#DOWNLOADING}</li>
   * </ul>
   * {@inheritDoc}
   */
  @Override
  protected boolean accept(CommandInfo command) {
    return command.getStatus() == CommandStatus.SCHEDULED
      || command.getStatus() == CommandStatus.UPLOADING
      || command.getStatus() == CommandStatus.INIT
      || command.getStatus() == CommandStatus.EXECUTING
      || command.getStatus() == CommandStatus.DOWNLOADING;
  }

  /**
   * Interrompe a execuo dos comandos selecionados. {@inheritDoc}
   */
  @Override
  protected void handleActionPerformed(ActionEvent ae) throws Exception {
    killCommands(getSelectedCommands());
  }

  /**
   * Interrompe a execuo dos comandos selecionados.
   * 
   * @param commands Comandos a terem sua execuo interrompida.
   * 
   * @return <tt>false</tt> se o usurio cancelar a operao ou se houver um
   *         erro.
   */
  private boolean killCommands(final CommandInfo[] commands) {
    if (!warnKillCommand()) {
      return false;
    }

    boolean success = true;

    ApplicationFrame frame = getApplication().getApplicationFrame();
    for (int inx = 0; inx < commands.length; inx++) {
      CommandInfo cmd = commands[inx];
      String cmdId = cmd.getId();
      if (CommandStatus.SCHEDULED.equals(cmd.getStatus())) {
        success &= SchedulerProxy.removeCommand(frame, cmdId);
      }
      else {
        String sgaName = cmd.getSGAName();
        success &= SGAProxy.killCommand(sgaName, cmdId);
      }
    }

    /*
     * Em caso de erro, mostrar mensagem com comandos que no puderam ser
     * removidos.
     */
    if (!success) {
      String errorMessage =
        getString("KillCommandsAction.task.commands.kill.error",
          commands.length);
      getApplication().showError(errorMessage);
    }

    return success;
  }

  /**
   * Informa ao usurio que a execuo de um ou mais comandos ser interrompida
   * e pede confirmao da operao.
   * 
   * @return <tt>true</tt> se o usurio informou que deseja que a execuo do
   *         comando seja interrompida, ou <tt>false</tt> caso ele queira
   *         cancelar a operao.
   */
  private boolean warnKillCommand() {
    String message =
      getString("KillCommandsAction.task.commands.kill.confirmation.query");
    ApplicationFrame frame = getApplication().getApplicationFrame();
    String title = frame.getTitle();
    String[] options = new String[] { LNG.get("UTIL_YES"), LNG.get("UTIL_NO") };

    int option =
      StandardDialogs.showOptionDialog(frame, title, message, options);
    switch (option) {
      case 0:
        return true;
      case 1:
        return false;
      default:
        String errorMessage =
          getString("KillCommandsAction.task.commands.kill.confirmation.error",
            option);
        throw new IllegalStateException(errorMessage);
    }
  }
}
