package csbase.client.applications.serverdiagnostic.basic;

import java.awt.GridBagLayout;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.gui.GBC;
import csbase.client.applications.ApplicationComponentPanel;
import csbase.client.applications.serverdiagnostic.ServerDiagnostic;
import csbase.logic.diagnosticservice.ServerBasicInfo;

/**
 * Painel com as informaes bsicas do servidor.
 *
 * @author Tecgraf PUC-Rio
 */
public class IdentificationInfoPanel extends
  ApplicationComponentPanel<ServerDiagnostic> {

  /** Informaes bsicas da identificao do servidor */
  private ServerBasicInfo basicInfo;

  /**
   * Construtor
   *
   * @param basicInfo contm as informaes bsicas de identificao do
   *        servidor.
   *
   * @param application a aplicao que usa esse painel
   */
  public IdentificationInfoPanel(final ServerDiagnostic application,
    ServerBasicInfo basicInfo) {
    super(application);
    this.basicInfo = basicInfo;
    buildInterface();
  }

  /**
   * Constri os componentes do painel.
   */
  private void buildInterface() {
    setLayout(new GridBagLayout());
    add(buildSystemPanel(), new GBC(0, 0).horizontal().northwest().insets(0, 0,
      5, 0));
    add(buildHostPanel(), new GBC(0, 1).horizontal().northwest().insets(0, 0,
      5, 0));
    add(buildJVMPanel(), new GBC(0, 2).horizontal().northwest().insets(0, 0, 5,
      0));
    add(buildRepositoriesPanel(), new GBC(0, 3).both().northwest().insets(0, 0,
      0, 0));
    add(buildLibsPanel(), new GBC(1, 0).both().northwest().insets(0, 0, 0, 0)
      .height(4));

  }

  /**
   * Constri o painel com as informaes do sistema (nome, verso e data de
   * inicializao).
   *
   * @return o painel
   */
  private JPanel buildSystemPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.add((new JLabel(getString("label.system.name"))), new GBC(0, 0)
      .none().northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(basicInfo.systemName)), new GBC(1, 0).horizontal()
      .northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(getString("label.system.version"))), new GBC(0, 1)
      .none().northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(basicInfo.systemVersion)), new GBC(1, 1).horizontal()
      .northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(getString("label.system.date"))), new GBC(0, 2)
      .none().northwest().insets(0, 5, 5, 0));
    long startupTime = basicInfo.startUpTime;
    long delta = System.currentTimeMillis() - startupTime;
    long mins = delta / (1000 * 60);
    long hours = mins / 60;
    long days = hours / 24;
    panel.add((new JLabel(String.format("%s (%dd %dh %dmin)", FormatUtils
      .format(basicInfo.startUpTime), days, hours % 24, mins % 60))), new GBC(
        1, 2).horizontal().northwest().insets(0, 5, 5, 0));
    panel.setBorder(BorderFactory
      .createTitledBorder(getString("border.system")));
    return panel;
  }

  /**
   * Constri o painel com as informaes da mquina do servidor (sistema
   * operacional, nome, IP e charset).
   *
   * @return o painel
   */
  private JPanel buildHostPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.add((new JLabel(getString("label.host.so"))), new GBC(0, 0).none()
      .northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(basicInfo.operationalSystem)), new GBC(1, 0)
      .horizontal().northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(getString("label.host.id"))), new GBC(0, 1).none()
      .northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(String.format("%s (%s)", basicInfo.hostName,
      basicInfo.hostIP))), new GBC(1, 1).horizontal().northwest().insets(0, 5,
        5, 0));
    panel.add((new JLabel(getString("label.host.charset"))), new GBC(0, 2)
      .none().northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(basicInfo.charset)), new GBC(1, 2).horizontal()
      .northwest().insets(0, 5, 5, 0));
    panel.setBorder(BorderFactory.createTitledBorder(getString("border.host")));
    return panel;
  }

  /**
   * Constri o painel com as informaes da JVM do servidor.
   *
   * @return o painel
   */
  private JPanel buildJVMPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.add((new JLabel(getString("label.jvm.version"))), new GBC(0, 0)
      .none().northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(basicInfo.javaVersion)), new GBC(1, 0).horizontal()
      .northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(getString("label.jvm.args"))), new GBC(0, 1).none()
      .northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(basicInfo.javaArgs)), new GBC(1, 1).horizontal()
      .northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(getString("label.rmi.port"))), new GBC(0, 2).none()
      .northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(String.valueOf(basicInfo.rmiPort))), new GBC(1, 2)
      .horizontal().northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(getString("label.running.path"))), new GBC(0, 3)
      .none().northwest().insets(0, 5, 5, 0));
    panel.add((new JLabel(String.valueOf(basicInfo.serverRunningDir))),
      new GBC(1, 3).horizontal().northwest().insets(0, 5, 5, 0));
    panel.setBorder(BorderFactory.createTitledBorder(getString("border.jvm")));
    return panel;
  }

  /**
   * Constri o painel com as informaes das bibliotecacas carregadas na JVM do
   * servidor.
   *
   * @return o painel
   */
  private JPanel buildLibsPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    List<String> tmpList = Arrays.asList(basicInfo.serverLibs);
    Collections.sort(tmpList, new Comparator<String>() {
      @Override
      public int compare(String o1, String o2) {
        String s1 = o1;
        String s2 = o2;
        return s1.toLowerCase().compareTo(s2.toLowerCase());
      }
    });
    JList list = new JList(tmpList.toArray());
    panel.add(new JScrollPane(list), new GBC(0, 0).both().northwest().insets(0,
      5, 0, 0));
    panel.setBorder(BorderFactory.createTitledBorder(getString("border.libs")));
    return panel;
  }

  /**
   * Constri o painel com as informaes dos paths de repositrios de dados.
   *
   * @return o painel
   */
  private JPanel buildRepositoriesPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.add((new JLabel(getString("label.repository.projects"))), new GBC(0,
      0).none().northwest().insets(0, 5, 5, 0));
    JLabel projectRepository = new JLabel(basicInfo.projectDir);
    panel.add(projectRepository, new GBC(1, 0).horizontal().northwest().insets(
      0, 5, 5, 0));
    panel.add((new JLabel(getString("label.repository.algorithms"))), new GBC(
      0, 1).none().northwest().insets(0, 5, 5, 0));
    JLabel algorithmRepository = new JLabel(basicInfo.algorithmDir);
    panel.add(algorithmRepository, new GBC(1, 1).horizontal().northwest()
      .insets(0, 5, 5, 0));
    panel.add((new JLabel(getString("label.repository.persistency"))), new GBC(
      0, 2).none().northwest().insets(0, 5, 5, 0).pushy());
    JLabel persistencyRepository = new JLabel(basicInfo.persistencyDir);
    panel.add(persistencyRepository, new GBC(1, 2).horizontal().northwest()
      .insets(0, 5, 5, 0).pushy());
    panel.setBorder(BorderFactory
      .createTitledBorder(getString("border.repositories")));
    return panel;
  }
}
