package csbase.client.applications.sgamonitor;

import java.awt.Component;
import java.awt.Container;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationComponentDialog;
import csbase.client.applications.sgamonitor.filters.NameFilter;
import csbase.client.facilities.configurabletable.UIFactory;
import csbase.client.facilities.configurabletable.UI.UI4Tables;
import csbase.client.facilities.configurabletable.stringprovider.ApplicationStringProvider;
import csbase.client.facilities.configurabletable.stringprovider.IStringProvider;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;
import csbase.client.facilities.configurabletable.table.RowToKey;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.types.PVTables;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;

/**
 * Dilogo que exibe as informaes dos ns de um cluster.
 * 
 * @author Tecgraf
 */
public class ClusterMonitor extends ApplicationComponentDialog<SGAMonitor>
  implements Observer {

  /**
   * Identificador do painel definido no XML de configurao das tabelas.
   */
  private static final String CLUSTER_PANEL = "cluster-panel";

  /**
   * Tabela que exibe as informaes de um cluster.
   */
  private UI4Tables ui4Tables;

  /**
   * Campo em que o usurio filtra os ns do cluster pelo nome.
   */
  private JTextField searchField;

  /**
   * Nome do cluster monitorado.
   */
  private String clusterName;

  /**
   * Construtor padro.
   * 
   * @param app - aplicao pai.
   */
  public ClusterMonitor(SGAMonitor app) {
    super(app);

    IStringProvider stringProvider =
      new ApplicationStringProvider(getApplication());
    UIFactory uiFactory = new UIFactory(app.getConfig(), stringProvider);
    ui4Tables = uiFactory.getUI4Tables(CLUSTER_PANEL);

    buildInterface();

    pack();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(boolean visible) {

    PreferenceCategory appPrefs = getApplication().getPreferences();
    PVTables pv =
      (PVTables) appPrefs.getPreference(SGAMonitorPref.CLUSTER_TABLES);

    if (visible) {
      pv.loadTables(ui4Tables.getAllTables());
    }
    else {
      pv.storeTables(ui4Tables.getAllTables());
      getApplication().savePreferences();
    }

    super.setVisible(visible);
  }

  /**
   * Atualiza os dados da tabela de ns de um cluster
   * 
   * @param cluster - atualiza os dados da tabela de ns de um cluster.
   */
  public void updateClusterInfo(SGASet cluster) {
    final List<SGAInfo> sgasInfo = new ArrayList<SGAInfo>();
    for (SGAInfo info : cluster.getAllInfo()) {
      sgasInfo.add(info);
    }

    clusterName = cluster.getName();

    final Application app = getApplication();
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        setTitle(String.format(app.getString("cluster.monitor"), clusterName));
        for (ConfigurableTable<SGAInfo> table : ui4Tables
          .getAllTables(SGAInfo.class)) {
          table.updateRows(sgasInfo);
        }
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  @SuppressWarnings("unchecked")
  public void update(Observable o, Object arg) {
    List<SGASet> sgaSetList = (List<SGASet>) arg;

    for (SGASet sgaSet : sgaSetList) {
      if (sgaSet.getName().equals(this.clusterName)) {

        final List<SGAInfo> sgaInfoList = new ArrayList<SGAInfo>();
        for (SGAInfo info : sgaSet.getAllInfo()) {
          sgaInfoList.add(info);
        }
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            for (ConfigurableTable<SGAInfo> table : ui4Tables
              .getAllTables(SGAInfo.class)) {

              table.updateRows(sgaInfoList);
            }
          }
        });
      }
    }
  }

  // -------------------- mtodos privados ----------------------------------

  /**
   * Constri a interface que exibe a tabela de monitorao dos ns de um
   * Cluster.
   */
  private void buildInterface() {
    setJMenuBar(createMenuBar());

    Container container = getContentPane();
    container.setLayout(new GridBagLayout());

    JLabel searchLabel = new JLabel(getApplication().getString("search.label"));
    this.searchField = new JTextField();
    this.searchField.getDocument().addDocumentListener(
      new FilterNameListener(this));

    JPanel searchPanel = new JPanel(new GridBagLayout());
    searchPanel.add(searchLabel, new GBC(0, 0).west().top(15).left(10));
    searchPanel.add(searchField, new GBC(1, 0).top(15).left(10).right(50)
      .horizontal());

    for (ConfigurableTable<SGAInfo> table : ui4Tables
      .getAllTables(SGAInfo.class)) {
      table.setFilter(new NameFilter());
      table.setRowToKey(new RowToKey<SGAInfo>() {
        @Override
        public String getKey(SGAInfo row) {
          return row.getHostName();
        }
      });
    }

    container.add(searchPanel, new GBC(0, 0).west().left(10).bottom(0)
      .horizontal());
    container.add((Component) ui4Tables, new GBC(0, 1).both().insets(10, 10,
      10, 10));

    // Adicionando boto que fecha o dilogo.
    JButton close = new JButton(getApplication().getString("button.close"));
    close.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ClusterMonitor.this.setVisible(false);
      }
    });

    container.add(close, new GBC(0, 2).none().east().insets(0, 0, 10, 20));
  }

  /**
   * Cria a barra de menu do dilogo.
   * 
   * @return barra de menu.
   */
  private JMenuBar createMenuBar() {

    JMenu menu = new JMenu(getApplication().getString("menu.tables"));

    for (ConfigurableTable<?> table : ui4Tables.getAllTables()) {
      JMenu subMenu = new JMenu(ui4Tables.getTableLabel(table.getId()));

      for (JCheckBoxMenuItem item : table.createColumnsCheckBoxes()) {
        subMenu.add(item);
      }

      menu.add(subMenu);
    }

    JMenuBar menuBar = new JMenuBar();
    menuBar.add(menu);

    return menuBar;
  }

  /**
   * Mtodo que atualiza o filtro da tabela.
   */
  private void updateNameFilter() {
    try {
      String regex = searchField.getText();
      Pattern newPattern = Pattern.compile(regex);

      for (ConfigurableTable<SGAInfo> table : ui4Tables
        .getAllTables(SGAInfo.class)) {

        NameFilter filter = (NameFilter) table.getFilter();
        filter.setPattern(newPattern);

        table.updateRows();
      }
    }
    catch (PatternSyntaxException e) {
    }
  }

  // ----------------------- classe privada -------------------------------

  /**
   * Ouvinte que filtra as linhas da tabela de ns de um cluster a partir da
   * String digitada no JTextField.
   * 
   * @author Tecgraf
   */
  private class FilterNameListener implements DocumentListener {

    /**
     * Referncia para a aplicao.
     */
    private ClusterMonitor clusterMonitor;

    /**
     * Construtor padro.
     * 
     * @param clusterMonitor - referncia para o dialogo.
     */
    public FilterNameListener(ClusterMonitor clusterMonitor) {
      this.clusterMonitor = clusterMonitor;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void changedUpdate(DocumentEvent e) {
      clusterMonitor.updateNameFilter();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void insertUpdate(DocumentEvent e) {
      changedUpdate(e);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeUpdate(DocumentEvent e) {
      changedUpdate(e);
    }

  }

}
