/*
 * PlatformManagerPanel.java
 * 
 * $Author: cviana $ $Revision: 179359 $ - $Date: 2006-08-16 15:17:41 -0300
 * (Wed, 16 Aug 2006) $
 */
package csbase.client.ias;

import java.text.MessageFormat;
import java.util.Comparator;
import java.util.Vector;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.table.TableModel;

import csbase.client.remote.srvproxies.PlatformProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.SwingObserverWrapper;
import csbase.logic.Platform;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;

/**
 * A classe {@code PlatformManagerPanel} representa o painel de
 * administrao das plataformas de execuo. Mostra uma tabela com o nome e a
 * descrio de cada plataforma. O administrador pode incluir, remover e alterar
 * as informaes de uma plataforma.
 * 
 * @author Marcos Machado, Anselmo e Ana
 */
public class PlatformManagerPanel extends ManagerPanel {
  /** A janela onde esse painel est sendo usado. */
  private JFrame parent;

  // Constante utilizada para testes de condies de erro.
  // 0: funcionamento normal
  // 1: exibe a mensagem de comunicao com o servidor e dispara o tratamento 
  //    para plataforma no encontrada
  private static final int debug = 0;

  /**
   * Remove o observador associado  classe Platform.
   */
  @Override
  public void beforeClose() {
    Platform.deleteObserver(new SwingObserverWrapper(this));
  }

  /**
   * Exibe a janela de incluso de Plataformas
   */
  @Override
  public void add() {
    new PlatformInfoDialog(parent);
  }

  /**
   * Exibe a janela de modificao de Plataformas
   * 
   * @param id identificador da plataforma a ser modificada.
   */
  @Override
  public void modify(Object id) {
    Platform platform = getPlatform(id);
    if (platform == null) {
      return;
    }
    new PlatformInfoDialog(parent, platform);
  }

  /**
   * Obtm a plataforma identificada pelo parmetro especificado. Caso a
   * plataforma no seja encontrada, exibe mensagem de erro adequada.
   * 
   * @param id identificador da plataforma.
   * 
   * @return plataforma ou <code>null</code> caso esta no seja encontrada.
   */
  private Platform getPlatform(Object id) {
    Platform platform =
      PlatformProxy.getPlatform(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_PLATFORM"), id);
    if (debug == 1) { //@debug
      displayMessage(parent.getTitle(), LNG.get("IAS_WAITING_PLATFORM"));
      platform = null; // @debug
    } //@debug
    if (platform == null) {
      displayError(LNG.get("IAS_PLATMGR_PLATFORM_NOT_FOUND"));
    }
    return platform;
  }

  /**
   * Exibe a janela de remoo de Plataforma.
   * 
   * @param id identificador da plataforma a ser removida.
   */
  @Override
  public void delete(Object id) {
    Platform platform = getPlatform(id);
    if (platform == null) {
      return;
    }
    String platformName = platform.getName();
    int res =
      StandardDialogs.showYesNoDialog(parent, parent.getTitle(), MessageFormat
        .format(LNG.get("IAS_PLATMGR_PLATFORM_REMOVAL_CONFIRMATION"),
          new Object[] { platformName }));
    if (res == JOptionPane.YES_OPTION) {
      PlatformProxy.deletePlatform(parent, parent.getTitle(), MessageFormat
        .format(LNG.get("IAS_PLATMGR_WAITING_REMOVE_PLATFORM"),
          new Object[] { platformName }), platform.getId());
    }
  }

  /**
   * Mostra uma janela com uma mensagem.
   * 
   * @param title ttulo da janela de mensagem
   * @param message mensagem a ser exibida
   */
  protected void displayMessage(String title, String message) {
    StandardDialogs.showInfoDialog(parent, title, message);
  }

  /**
   * Mostra uma janela com uma mensagem de erro.
   * 
   * @param message mensagem de erro a ser exibida
   */
  protected void displayError(String message) {
    StandardErrorDialogs.showErrorDialog(parent, parent.getTitle(), message);
  }

  /**
   * Cria um painel de gerncia de Plataformas de Execuo
   * 
   * @param parent A janela onde esse painel est sendo usado.
   */
  public PlatformManagerPanel(JFrame parent) {
    this.parent = parent;
    Vector<Platform> platforms =
      PlatformProxy.getAllPlatforms(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_ADMIN_DATA"));
    if (platforms == null) {
      platforms = new Vector<Platform>();
    }
    ObjectTableProvider provider = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        Platform platform = (Platform) item;
        if (platform == null) {
          return null;
        }
        return new Object[] { platform.getId(), platform.getName(),
            platform.getDescription() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { null, LNG.get("IAS_PLATFORM"),
            LNG.get("IAS_PLATFORM_DESCRIPTION") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { Object.class, String.class, String.class };
      }
    };
    TableModel model = new ObjectTableModel<Platform>(platforms, provider);
    Comparator<?>[] comparators =
      { null, ClientUtilities.getStringComparatorIgnoreCase(),
          ClientUtilities.getStringComparatorIgnoreCase() };
    make(model, comparators, false);
    Platform.addObserver(new SwingObserverWrapper(this));
  }
}
