package csbase.client.util.charset;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.ButtonGroup;
import javax.swing.JMenu;
import javax.swing.JRadioButtonMenuItem;

import csbase.client.Client;

/**
 * Combo com a lista de charsets padronizados conforme a lista definida em:
 * {@code StandardCharsets}.
 * 
 * @author Tecgraf/PUC-Rio
 */
final public class CharsetRadioMenu extends JMenu {
  /**
   * Lista de listeners.
   */
  List<CharsetRadioMenuChangeListener> listeners =
    new ArrayList<CharsetRadioMenuChangeListener>();

  /**
   * Lista de itens.
   */
  final private Map<Charset, JRadioButtonMenuItem> menuItens =
    new HashMap<Charset, JRadioButtonMenuItem>();

  /**
   * Criao de um item de menu (radio) para um charset.
   * 
   * @param charset charset.
   * @return item
   */
  final private JRadioButtonMenuItem createItem(final Charset charset) {
    final JRadioButtonMenuItem item = new JRadioButtonMenuItem();
    final StandardCharsets stdCharsets = StandardCharsets.getInstance();
    final String chName = stdCharsets.getLegibleName(charset);
    item.setText(chName);
    item.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        final Charset selectedCharset = getSelectedCharset();
        fireListeneners(selectedCharset);
      }
    });
    return item;
  }

  /**
   * Dispara listeners de troca.
   * 
   * @param selectedCharset charset selecionado
   */
  private void fireListeneners(Charset selectedCharset) {
    for (CharsetRadioMenuChangeListener lst : listeners) {
      lst.charsetChanged(selectedCharset);
    }
  }

  /**
   * Construtor
   */
  public CharsetRadioMenu() {
    this(StandardCharsets.getInstance().getAllStandardCharsets());
  }

  /**
   * Construtor Lista de charsets a ser exibida.
   * 
   * @param charsets lista dos charsets a serem exibidos.
   */
  public CharsetRadioMenu(final List<Charset> charsets) {
    final Client client = Client.getInstance();
    final Charset sysCharset = client.getSystemDefaultCharset();
    final ButtonGroup grp = new ButtonGroup();
    for (Charset ch : charsets) {
      final JRadioButtonMenuItem item = createItem(ch);
      menuItens.put(ch, item);
      if (ch.equals(sysCharset)) {
        item.setSelected(true);
      }
      add(item);
      grp.add(item);
    }
  }

  /**
   * Ajuste o charset selecionado.
   * 
   * @param charset o objeto.
   */
  final public void setSelectedCharset(final Charset charset) {
    final Set<Charset> charsets = menuItens.keySet();
    for (Charset ch : charsets) {
      if (ch.equals(charset)) {
        final JRadioButtonMenuItem item = menuItens.get(ch);
        if (item != null) {
          item.setSelected(true);
        }
      }
    }
  }

  /**
   * Consulta o charset selecionado.
   * 
   * @return o charset selecionado.
   */
  final public Charset getSelectedCharset() {
    final Set<Charset> charsets = menuItens.keySet();
    for (Charset ch : charsets) {
      final JRadioButtonMenuItem item = menuItens.get(ch);
      if (item.isSelected()) {
        return ch;
      }
    }
    return null;
  }

  /**
   * Adiciona listener a lista do widget.
   * 
   * @param listener listener
   */
  final public void addCharsetChangeListener(
    CharsetRadioMenuChangeListener listener) {
    listeners.add(listener);
  }

  /**
   * Adiciona listener a lista do widget.
   * 
   * @param listener listener
   */
  final public void delCharsetChangeListener(
    CharsetRadioMenuChangeListener listener) {
    listeners.remove(listener);
  }

}
