/*
 * $Id$
 */

package csbase.client.algorithms.commands.cache.events;

import csbase.client.util.event.IEvent;
import csbase.logic.CommandStatus;

/**
 * Como podem existir muitos comandos nos estados {@link CommandStatus#FINISHED}
 * e {@link CommandStatus#SYSTEM_FAILURE} persistidos no sistema, a cache os
 * carrega aos poucos, para evitar que o cliente espere muito tempo por uma
 * resposta.<br>
 * Esse evento  lanado sempre que h esse carregamento parcial dos comandos,
 * indicando quantos comandos faltam para serem carregados.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class FinishedOrSystemFailureCommandsLoadingEvent implements IEvent {
  /**
   * O nmero de comandos que faltam serem carregados na cache.
   */
  private long missing;
  /**
   * O total de comandos sendo carregados na cache.
   */
  private long total;

  /**
   * Construtor usado para criar evento indicando que nenhum comando ainda foi
   * carregado.
   * 
   * @param total O total de comandos faltam serem carregados na cache.
   */
  public FinishedOrSystemFailureCommandsLoadingEvent(long total) {
    this.missing = total;
    this.total = total;
  }
  
  /**
   * Construtor.
   * 
   * @param missing O nmero de comandos que faltam serem carregados na cache.
   * @param total O total de comandos que esto sendo carregados na cache.
   */
  public FinishedOrSystemFailureCommandsLoadingEvent(long missing, long total) {
    this.missing = missing;
    this.total = total;
  }

  /**
   * Obtm o nmero de comandos que j foram carregados na cache.
   * 
   * @return O nmero de comandos que j foram carregados na cache.
   */
  public long getLoaded() {
    return total - missing;
  }

  /**
   * Obtm o nmero de comandos que faltam serem carregados na cache.
   * 
   * @return O nmero de comandos que faltam serem carregados na cache.
   */
  public long getMissing() {
    return missing;
  }

  /**
   * Obtm o total de comandos que esto sendo carregados na cache.
   * 
   * @return O total de comandos que esto sendo carregados na cache.
   */
  public long getTotal() {
    return total;
  }

  /**
   * Indica se todos os comandos j foram carregados.
   * 
   * @return <tt>true</tt> se todos os comandos j foram carregados.
   */
  public boolean finished() {
    return missing == 0;
  }
}
