package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.MultipleProjectSpaceTask;
import csbase.client.applications.projectsmanager.proxy.ReloadSelectedProjectsTask;

/**
 * Ao responsvel por recarregar todas as informaes dos projetos
 * selecionados.
 * 
 * @author jnlopes
 * 
 */
public class RefreshProjectsAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public RefreshProjectsAction(ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    List<String> projectNames = new ArrayList<String>();
    List<Object> ownerId = new ArrayList<Object>();
    for (ProjectsManagerData pmd : projectList) {
      projectNames.add(pmd.getProjectName());
      ownerId.add(pmd.getOwnerId());
    }

    List<Double> occupiedSpaces =
      runMultipleProjectSpaceTask(projectNames, ownerId);
    List<ProjectsManagerData> updatedProjectList =
      runUpdateProjectDataTask(projectList);

    int count = 0;
    for (ProjectsManagerData pmd : updatedProjectList) {
      Double occupiedSpace = occupiedSpaces.get(count);
      if (occupiedSpace != pmd.getOccupiedSpace()) {
        pmd.setOccupiedSpace(occupiedSpace);
        projectsManager.updateProjectOccupiedSpace(pmd);
      }
      count++;
    }

    projectsManager.refreshProjectsTable();
  }

  /**
   * Executa task que obtm dados atualizados dos projetos.
   * 
   * @param projectList Lista de projetos.
   * @return Lista de projetos atualizada.
   */
  private List<ProjectsManagerData> runUpdateProjectDataTask(
    List<ProjectsManagerData> projectList) {
    ProjectsManager projectsManager = getProjectsManager();
    ReloadSelectedProjectsTask rspt =
      new ReloadSelectedProjectsTask(projectsManager, projectList);

    rspt.execute(projectsManager.getApplicationFrame(),
      getString("RefreshProjectsAction.title"),
      getString("RefreshProjectsAction.message"));
    if (rspt.wasCancelled()) {
      String err = getString("RefreshProjectsAction.cancelled.message");
      rspt.showError(err);
      return null;
    }

    if (rspt.getStatus() != true) {
      Exception exception = rspt.getError();
      ProjectsManagerUI.handleException(this.getProjectsManager(), exception);
      return null;
    }

    List<ProjectsManagerData> updatedProjectList = rspt.getResult();
    if (updatedProjectList == null) {
      String errTitle =
        getString("RefreshProjectsAction.modification.date.title");
      String errMsg =
        getString("RefreshProjectsAction.modification.date.message");
      StandardDialogs.showErrorDialog(getProjectsManager()
        .getApplicationFrame(), errTitle, errMsg);
      return null;
    }

    return updatedProjectList;
  }

  /**
   * Executa task que obtm espao ocupado por projetos.
   * 
   * @param projectNames Nomes dos projetos.
   * @param ownerId Ids dos donos dos projetos
   * @return Espaos ocupados pelos projetos
   */
  private List<Double> runMultipleProjectSpaceTask(List<String> projectNames,
    List<Object> ownerId) {

    ProjectsManager projectsManager = getProjectsManager();
    MultipleProjectSpaceTask spst =
      new MultipleProjectSpaceTask(projectsManager, projectNames, ownerId);

    spst.execute(projectsManager.getApplicationFrame(),
      getString("RefreshProjectsAction.title"),
      getString("RefreshProjectsAction.message"));
    if (spst.wasCancelled()) {
      String err = getString("RefreshProjectsAction.cancelled.message");
      spst.showError(err);
      return null;
    }

    if (spst.getStatus() != true) {
      Exception exception = spst.getError();
      ProjectsManagerUI.handleException(this.getProjectsManager(), exception);
      return null;
    }

    List<Double> occupiedSpace = spst.getResult();
    if (occupiedSpace == null) {
      String errTitle =
        getString("RefreshProjectsAction.occupied.space.error.title");
      String errMsg =
        getString("RefreshProjectsAction.occupied.space.error.message");
      StandardDialogs.showErrorDialog(getProjectsManager()
        .getApplicationFrame(), errTitle, errMsg);
      return null;
    }

    return occupiedSpace;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ApplicationImages.ICON_REFRESH_ONE_16;
  }

}
