package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.util.gui.ComponentUtilities;
import csbase.logic.User;
import csbase.logic.UserInfo;

/**
 * Classe abstrata que representa o que h de comum aos dilogos que exibem
 * dados de um usurio.
 * 
 * @author Marcus Leal
 */
public abstract class UserInfoDialog {
  /** Instncia do dilogo */
  protected DesktopComponentDialog userInfoDialog;
  /** Janela que criou o dilogo */
  protected Window ownerWindow;
  /** Campo nome do usurio */
  protected JTextField name;
  /** Campo login do usurio */
  protected JTextField login;
  /** Modelo da lista que contm os e-mails do usurio. */
  protected DefaultListModel emailsModel;
  /** Campo utilizado para conter e-mails que sero armazenados na lista. */
  protected JTextField addEmailTf;
  /** Ttulo para o dilogo e seus subdilogos */
  protected String dialogTitle;
  /**
   * Objeto que centraliza as informaes do usurio (padro <i>Data Transfer
   * Object</i> - DTO), compartilhadas entre esse dilogo e seus dilogos de
   * informaes complementares (dilogos de perfis, permisses e senha)
   */
  protected UserInfo userInfo;

  /**
   * Construtor que inicializa atributos comuns (referncia para a janela-pai,
   * ttulo do dilogo e DTO de informaos do usurio).
   * 
   * @param ownerWindow janela que criou este dilogo.
   * @param dialogTitle ttulo deste dilogo, usado tambm em alguns dos seus
   *        subdilogos.
   */
  protected UserInfoDialog(Window ownerWindow, String dialogTitle) {
    this.ownerWindow = ownerWindow;
    this.dialogTitle = dialogTitle;
    // UserInfo tem uma inconsistncia potencial: embora o construtor abaixo
    // substitua nulo por arrays vazios de Object nos casos dos parmetros de
    // perfis e permisses, o mtodo setAttribute permite atribuir diretamente
    // nulo a estes atributos. Para evitar dvidas entre teste de nulo ou vazio,
    // esta classe estar sempre atribuindo um array vazio para designar 
    // ausncia de parmetros.
    userInfo =
      new UserInfo(null, null, new String[0], new Object[0], new Object[0]);
  }

  /**
   * Exibe/esconde o dilogo. Se o dilogo ainda no tiver sido construdo,
   * realiza essa tarefa antes de invocar as operaes de visibilidade.
   * 
   * @param b indica se o dilogo deve estar visvel (<code>true</code>) ou
   *        invisvel (<code>false</code>).
   */
  public void setVisible(boolean b) {
    if (userInfoDialog == null) {
      makeDialog();
      addDataComponentsListeners();
    }
    userInfoDialog.setVisible(b);
  }

  /**
   * <p>
   * Adiciona observadores aos componentes que mantm os dados do usurio.
   * </p>
   * <p>
   * Se ao extender essa classe voc sobrescrever o mtodo
   * {@link #createDataComponents()} para adicionar novos componentes,
   * sobrescreva tambm este mtodo adicionando aqui os observadores dos dados
   * destes componentes para que quando estes forem alterados, eles chamem o
   * mtodo {@link #fireOnDataChanged()}.
   * </p>
   * <p>
   * Este mtodo  chamado logo aps o mtodo {@link #makeDialog()}, garantindo
   * assim que todos os componentes grficos j existam.
   * </p>
   */
  protected void addDataComponentsListeners() {
    DocumentListener docListener = new DocumentListener() {
      @Override
      public void changedUpdate(DocumentEvent e) {
        fireOnDataChanged();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        fireOnDataChanged();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        fireOnDataChanged();
      }
    };
    this.login.getDocument().addDocumentListener(docListener);
    this.name.getDocument().addDocumentListener(docListener);
    this.emailsModel.addListDataListener(new ListDataListener() {
      @Override
      public void intervalRemoved(ListDataEvent e) {
        fireOnDataChanged();
      }

      @Override
      public void intervalAdded(ListDataEvent e) {
        fireOnDataChanged();
      }

      @Override
      public void contentsChanged(ListDataEvent e) {
        fireOnDataChanged();
      }
    });
  }

  /**
   * Cria um dilogo bsico de dados de um usurio, dividido em 3 reas:
   * <ol>
   * <li>um painel superior com os campos de texto login, nome e e-mail;</li>
   * <li>um painel central vazio, preparado para botes que acionam dilogos de
   * informaes complementares, como perfis, permisses e senha;</li>
   * <li>e finalmente um painel inferior para botes de ao, contendo um boto
   * "Fechar".</li>
   * </ol>
   * Tanto o boto "X" no canto superior esquerdo da janela quanto o boto
   * "Fechar" executam o mesmo procedimento {@link UserInfoDialog#close()} ao
   * fechar.
   * 
   * @see UserInfoDialog#close()
   */
  protected void makeDialog() {
    userInfoDialog = new DesktopComponentDialog(ownerWindow, dialogTitle);
    JPanel userInfoPanel = new JPanel(new BorderLayout());
    userInfoPanel.setBorder(BorderFactory.createTitledBorder(LNG
      .get("IAS_USER_REGISTRATION")));
    userInfoPanel.add(makeUpperPanel(), BorderLayout.NORTH);
    userInfoPanel.add(makeMiddlePanel(), BorderLayout.SOUTH);
    Container contentPane = userInfoDialog.getContentPane();
    contentPane.setLayout(new BorderLayout());
    contentPane.add(userInfoPanel, BorderLayout.CENTER);
    contentPane.add(makeLowerPanel(), BorderLayout.SOUTH);
    userInfoDialog.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        close();
      }
    });
    userInfoDialog.pack();
    userInfoDialog.center(ownerWindow);
  }

  public DesktopComponentDialog getUserInfoDialog() {
    return userInfoDialog;
  }

  /**
   * <p>
   * Cria uma lista com os componentes que iro conter os dados do usurios.<br>
   * Cada elemento dessa lista dever ser um array de dois componentes. O 1o
   * dever ser o <i>label</i> do 2o componente.
   * </p>
   * <p>
   * Se esse mtodo for sobrescrito para adicionar novos componentes, voc
   * tambm dever sobrescrever os mtodos {@link #addDataComponentsListeners()}
   * e {@link #updateUserInfo()}.
   * </p>
   * 
   * @return uma lista de tuplas label-componente que iro conter os dados do
   *         usurio.
   * @see #addDataComponentsListeners()
   */
  protected List<JComponent[]> createDataComponents() {
    this.login = getLogin();
    this.name = new JTextField(20);

    // Cria os componentes internos do componente de e-mail.
    this.emailsModel = new DefaultListModel();
    this.addEmailTf = new JTextField(20);
    final JButton addEmailBtn = new JButton();
    final JList emails = new JList(emailsModel);
    final JButton removeEmailBtn = new JButton();

    this.addEmailTf.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        addEmail();
      }
    });

    // Cria a ao para o boto de adicionar um e-mail a lista.
    final Action addEmailAction =
      new AbstractAction(LNG.get("IAS_USER_EMAIL_ADD_BUTTON")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          addEmail();
        }
      };
    /*
     * Habilita e desabilita o boto de adicionar e-mail de acordo com a
     * existncia ou ausncia de texto na caixa de texto addEmailTf.
     */
    addEmailAction.setEnabled(!addEmailTf.getText().trim().isEmpty());
    this.addEmailTf.getDocument().addDocumentListener(new DocumentListener() {
      @Override
      public void changedUpdate(DocumentEvent e) {
        addEmailAction.setEnabled(!addEmailTf.getText().trim().isEmpty());
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        addEmailAction.setEnabled(!addEmailTf.getText().trim().isEmpty());
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        addEmailAction.setEnabled(true);
      }
    });
    addEmailBtn.setAction(addEmailAction);

    // Cria a ao para o boto de remover um e-mail da lista.
    final Action removeEmailAction =
      new AbstractAction(LNG.get("IAS_USER_EMAIL_REMOVE_BUTTON")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          int index = emails.getSelectedIndex();
          if (index < 0) {
            return;
          }

          emailsModel.remove(index);
          if (emailsModel.getSize() > 0) {
            if (index == emailsModel.getSize()) {
              // removed item in last position
              index--;
            }
            emails.setSelectedIndex(index);
            emails.ensureIndexIsVisible(index);
          }
        }
      };
    /*
     * Habilita o boto de remover e-mail caso haja algum e-mail selecionado na
     * lista de e-mails.
     */
    removeEmailAction.setEnabled(emails.getSelectedIndex() >= 0);
    emails.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        removeEmailAction.setEnabled(emails.getSelectedIndex() >= 0);
      }
    });
    removeEmailBtn.setAction(removeEmailAction);

    // Configura a lista de e-mails.
    emails.setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
    emails.setVisibleRowCount(4);
    emails.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        removeEmailBtn.setEnabled(emails.getSelectedIndex() >= 0);
      }
    });
    JScrollPane scrollableEmails = new JScrollPane(emails);

    // Arruma o tamanho dos componentes, 
    ComponentUtilities.setMaxPreferredWidth(addEmailTf, scrollableEmails);
    ComponentUtilities.setMaxPreferredWidth(addEmailBtn, removeEmailBtn);

    // Cria o painel de e-mails. 
    JPanel emailPanel = new JPanel(new GridBagLayout());
    GBC gbc = new GBC(0, 0).northwest().horizontal().insets(0, 0, 5, 5);
    emailPanel.add(addEmailTf, gbc);
    gbc.gridx(1).northeast().none().insets(0, 0, 5, 0);
    emailPanel.add(addEmailBtn, gbc);
    gbc.gridx(0).gridy(1).northwest().horizontal().insets(0, 0, 0, 5);
    emailPanel.add(scrollableEmails, gbc);
    gbc.gridx(1).northeast().none().insets(0, 0, 0, 0);
    emailPanel.add(removeEmailBtn, gbc);

    // Retorna os componentes numa lista ordenada. 
    ArrayList<JComponent[]> components = new ArrayList<JComponent[]>();
    components.add(new JComponent[] { new JLabel(LNG.get("IAS_USER")), login });
    components.add(new JComponent[] { new JLabel(LNG.get("IAS_USER_NAME")),
        name });
    components.add(new JComponent[] { new JLabel(LNG.get("IAS_USER_EMAIL")),
        emailPanel });
    return components;
  }

  /**
   * Pega o e-mail digitado no campo {@code #addEmailTf} e o adiciona na lista
   * de e-mails.
   */
  public void addEmail() {
    if (!addEmailTf.getText().isEmpty()) {
      emailsModel.addElement(addEmailTf.getText());
      addEmailTf.setText(null);
    }
  }

  /**
   * Constri o painel superior contendo as principais informaes do usurio
   * (login, nome e e-mail).
   * 
   * @return o painel superior do dilogo.
   */
  protected JPanel makeUpperPanel() {
    // Cria os componentes que representam as propriedades do usurio.
    List<JComponent[]> components = createDataComponents();

    final int T = 12;
    final int TI = 6;
    final int L = 11;
    final int B = 12;
    final int R = 11;

    JPanel upperPanel = new JPanel(new GridBagLayout());
    for (int inx = 0; inx < components.size(); inx++) {
      JComponent label = components.get(inx)[0];
      JComponent component = components.get(inx)[1];

      GBC gbc = new GBC(0, inx);
      gbc = gbc.gridheight(1).gridwidth(1);
      gbc = gbc.west().weights(1.0, 0.0);

      if (components.size() == 1) {
        gbc = gbc.insets(T, L, B, R);
      }
      else {
        if (inx == 0) {//Se for a 1a propriedade. 
          gbc = gbc.insets(T, L, 0, R);
        }
        else if (inx == components.size() - 1) {//Se for a ltima propriedade.
          gbc = gbc.insets(TI, L, B, R);
        }
        else {
          gbc = gbc.insets(TI, L, 0, R);
        }
      }

      gbc = gbc.none().northwest();
      upperPanel.add(label, gbc);
      gbc = gbc.gridx(1).horizontal().weightx(100.0);
      upperPanel.add(component, gbc);
    }

    return upperPanel;
  }

  /**
   * Constri o painel central, reservado para a entrada de informaes
   * complementares do usurio. Essa implementao retorna um painel vazio.
   * 
   * @return painel central do dilogo.
   */
  protected JPanel makeMiddlePanel() {
    return new JPanel();
  }

  /**
   * Cria o boto de fechamento da janela, atribuindo-lhe como ao o mtodo
   * {@link UserInfoDialog#close()}.
   * 
   * @return o boto de fechamento.
   * 
   * @see UserInfoDialog#close()
   */
  protected JButton makeCloseButton() {
    JButton button = new JButton(LNG.get("IAS_CLOSE"));
    button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        close();
      }
    });
    return button;
  }

  /**
   * Mtodo para fechamento do dilogo. Verifica se houve entrada de dados no-
   * salvos, e em caso positivo, confirma com o usurio se este deseja sair
   * ignorando os dados entrados, cancelar ou gravar suas alteraes antes de
   * fechar. Termina por fechar o dilogo, a no ser que o usurio tenha
   * cancelado a operao.
   * 
   * @see UserInfoDialog#unsavedDataInput()
   * @see UserInfoDialog#confirmClose()
   */
  private void close() {
    if (unsavedDataInput()) {
      if (!confirmClose()) {
        return;
      }
    }

    userInfoDialog.close();
  }

  /**
   * Indica se os campos requeridos esto preenchidos.
   * 
   * @return <code>true</code> se os campos requeridos estiverem preenchidos ou
   *         <code>false</code> se estiverem vazios.
   */
  protected boolean requiredFieldsAreFilled() {
    return (!login.getText().trim().isEmpty())
      && (!name.getText().trim().isEmpty());
  }

  /**
   * <p>
   * Atualiza o contedo do DTO de usurio com as informaes contidas nos
   * campos do dilogo (login, nome e e-mail).
   * </p>
   * <p>
   * Se o mtodo {@link #createDataComponents()} for sobrescrito para adicionar
   * novos componentes, este mtodo tambm dever ser sobrescrito para que o
   * valor desses novos componentes sejam salvos no {@link UserInfo userInfo}.
   * </p>
   * 
   * @see #createDataComponents()
   */
  protected void updateUserInfo() {
    String nameText = name.getText().trim();
    String loginText = login.getText().trim();
    String[] emails = new String[emailsModel.getSize()];
    for (int inx = 0; inx < emails.length; inx++) {
      emails[inx] = emailsModel.getElementAt(inx).toString().trim();
    }
    userInfo.setAttribute(User.NAME, nameText);
    userInfo.setAttribute(User.LOGIN, loginText);
    userInfo.setAttribute(User.EMAILS, emails);
  }

  /**
   * Constri o painel inferior, contendo os botes de ao (o painel bsico s
   * contm o boto fechar).
   * 
   * @return o painel inferior do dilogo.
   */
  protected JPanel makeLowerPanel() {
    JPanel buttonPanel = new JPanel();
    buttonPanel.add(makeCloseButton());
    return buttonPanel;
  }

  protected JTextField getLogin() {
    if (login == null) {
      login = new JTextField(20);
    }
    return login;
  }

  /**
   * Mtodo chamado sempre que algum dado for alterado.
   */
  protected void fireOnDataChanged() {
  }

  /**
   * Verifica se houve entrada de dados no-salvos na interface.
   * 
   * @return <code>true</code> se algum dado foi editado ou <code>false</code>
   *         caso contrrio
   */
  protected abstract boolean unsavedDataInput();

  /**
   * Avisa o usurio de que h dados entrados no-salvos, e confirma se este
   * deseja realmente sair ignorando os dados entrados, cancelar ou gravar suas
   * alteraes antes de fechar.
   * 
   * @return <code>true</code> caso o dilogo deva ser realmente fechado, ou
   *         <code>false</code> caso o usurio queira cancelar o fechamento para
   *         revisar suas alteraes.
   */
  protected abstract boolean confirmClose();

}
