/*
 * $Id:$
 */

package csbase.client.applications.desktoplauncher;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.lang.reflect.Constructor;

import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationHideAction;
import csbase.client.applications.desktoplauncher.actions.ClearLogAction;
import csbase.client.applications.desktoplauncher.actions.DesktopHideAction;
import csbase.client.applications.desktoplauncher.actions.DesktopShowAction;
import csbase.client.applications.desktoplauncher.actions.DesktopShutdownAction;
import csbase.client.applications.desktoplauncher.panels.eventpanel.EventPanel;
import csbase.client.applications.desktoplauncher.panels.logpanel.LogPanel;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.logic.User;

/**
 * Aplicao de disparo de cliente de sistemas CSBASE.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DesktopLauncher extends Application {

  /**
   * Tratador de eventos da aplicao.
   */
  private DesktopLauncherEventHandler eventHandler;

  /**
   * Id do desktop filho a ser lanado.
   */
  final private String sonId;

  /**
   * Painel de eventos
   */
  final private EventPanel eventPanel;

  /**
   * Painel de eventos
   */
  final private LogPanel logPanel;

  /**
   * Montagem do menu.
   * 
   * @return o menu do programa.
   */
  private JMenuBar buildMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(buildOptionsMenu());
    menuBar.add(buildDesktopMenu());
    return menuBar;
  }

  /**
   * Montagem de menu de opes.
   * 
   * @return menu
   */
  private JMenu buildOptionsMenu() {
    final JMenu menu = new JMenu(getString("options.menu"));
    menu.add(new ClearLogAction(this));
    menu.add(new ApplicationHideAction(this));
    menu.addSeparator();
    menu.add(new ApplicationExitAction(this));
    return menu;
  }

  /**
   * Montagem de menu de opes.
   * 
   * @return menu
   */
  private JMenu buildDesktopMenu() {
    final JMenu menu = new JMenu(getString("desktop.menu"));
    menu.add(new DesktopHideAction(this));
    menu.add(new DesktopShowAction(this));
    menu.addSeparator();
    menu.add(new DesktopShutdownAction(this));
    return menu;
  }

  /**
   * Montagem da toolbar.
   * 
   * @return o painel.
   */
  private JToolBar buildToolBar() {
    final JToolBar toolBar = new JToolBar(JToolBar.HORIZONTAL);
    toolBar.setFloatable(false);
    toolBar.add(new ClearLogAction(this));
    toolBar.add(new ApplicationHideAction(this));
    toolBar.addSeparator();
    toolBar.add(new DesktopHideAction(this));
    toolBar.add(new DesktopShowAction(this));
    toolBar.add(new DesktopShutdownAction(this));
    toolBar.addSeparator();
    return toolBar;
  }

  /**
   * Adiciona evento  lista interna.
   * 
   * @param ev evento
   */
  protected void addLauncherEvent(final DesktopLauncherEvent ev) {
    final Client client = Client.getInstance();
    final String myId = client.getClientInstanceId();
    if (ev.getOtherId().equals(myId)) {
      return;
    }
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        eventPanel.addLauncherEvent(ev);
      }
    });
  }

  /**
   * Limpa a lista interna de eventos.
   */
  protected void clearLauncherEvents() {
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        eventPanel.clearLauncherEvents();
      }
    });
  }

  /**
   * Informa o id da aplicao que deve ser inciada quando o desktop e o projeto
   * incial iniciar
   * 
   * @return o id da aplicao
   */
  final String getInitialApplicationId() {
    final String propName = "initial.application.id";
    final String appId = getStringSpecificProperty(propName);
    if (appId.equals("null")) {
      return null;
    }
    return appId;
  }

  /**
   * Informa o id do project que deve ser aberto quando o desktop inciar.
   * 
   * @return o id da aplicao
   */
  final String getInitialProjectId() {
    final String propName = "initial.project.id";
    String projId = getStringSpecificProperty(propName);
    if (projId.equals("null")) {
      return null;
    }
    final User loggedUser = User.getLoggedUser();
    projId = projId.replaceAll("\\$LOGIN", loggedUser.getLogin());
    return projId;
  }

  /**
   * Indica se aplicativo incial depende da abertura prvia de um projeto.
   * 
   * @return indicativo
   */
  final public boolean isInitalApplicationProjectDependent() {
    final String propName = "initial.application.requires.project";
    boolean appDep = getBooleanSpecificProperty(propName, false);
    return appDep;
  }

  /**
   * Consulta das proprieades se o desktop deve aparecer inicialmente visvel.
   * 
   * @return indicativo
   */
  final public boolean isChildDesktopInitiallyVisible() {
    final String propName = "child.desktop.visible";
    boolean appDep = getBooleanSpecificProperty(propName, false);
    return appDep;
  }

  /**
   * Cria o tratador de eventos da aplicao.
   * 
   * @return o tratador.
   */
  private DesktopLauncherEventHandler createEventHandler() {
    final String className = getStringSpecificProperty("event.handler.class");
    if (className.equals("null")) {
      final String err = getString("event.handler.explicit.null.message");
      log(err);
      return null;
    }

    try {
      // A classe tem que ser do subtipo!
      final String loadingTag = getString("class.loading.message");
      log(String.format(loadingTag, className));

      final Class<?> forNameClass = Class.forName(className);
      final String castingTag = getString("class.casting.message");
      final String forNameClassName = forNameClass.getName();
      final Class<DesktopLauncherEventHandler> superClass =
        DesktopLauncherEventHandler.class;
      final String superClassName = superClass.getName();
      log(String.format(castingTag, forNameClassName, superClassName));
      final Class<? extends DesktopLauncherEventHandler> handlerClass =
        forNameClass.asSubclass(superClass);

      final String findingTag = getString("constructor.finding.message");
      final String handlerClassName = handlerClass.getName();
      log(String.format(findingTag, handlerClassName));
      final Constructor<? extends DesktopLauncherEventHandler> constructor =
        handlerClass.getDeclaredConstructor(DesktopLauncher.class);
      if (constructor == null) {
        final String err =
          getString("event.handler.constructor.not.found.message");
        log(err);
        return null;
      }
      log("Constructor: " + constructor.toGenericString());

      final DesktopLauncherEventHandler handler = constructor.newInstance(this);
      return handler;
    }
    catch (Exception e) {
      final String msg = getString("event.handler.constructor.failed.message");
      final String excpClassName = e.getClass().getSimpleName();
      final String excpMessage = e.getMessage();
      final String err = msg + " - " + excpClassName + " - " + excpMessage;
      log(err);
      log(null);

      final StackTraceElement[] stackTrace = e.getStackTrace();
      for (StackTraceElement stackTraceElement : stackTrace) {
        log("    - " + stackTraceElement.toString());
      }
      log(null);

      return null;
    }
  }

  /**
   * Consulta o atributo (ver documentao de {@link #eventHandler}).
   * 
   * @return o atributo.
   */
  public final DesktopLauncherEventHandler getEventHandler() {
    return eventHandler;
  }

  /**
   * Consulta o atributo (ver documentao de {@link #sonId}).
   * 
   * @return o atributo.
   */
  public final String getSonId() {
    return sonId;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() throws ApplicationException {
    try {
      if (eventHandler != null) {
        eventHandler.shutdown();
      }
    }
    catch (final Exception e) {
      throw new ApplicationException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void postInitialization() throws ApplicationException {
    final String systemLabel = getStringSpecificProperty("system.label");
    if (systemLabel == null) {
      final String msg = getString("no.system.label.error");
      throw new ApplicationException(msg);
    }

    final ExternalResources externalResources = ExternalResources.getInstance();
    if (!externalResources.isEnabled()) {
      final String msg = getString("no.external.resources.error");
      throw new ApplicationException(msg);
    }

    final DesktopFrame desktop = DesktopFrame.getInstance();
    final DesktopComponentFrame desktopFrame = desktop.getDesktopFrame();
    final Task<Void> connTask = new DesktopLauncherTask(this, systemLabel);
    final String title = getString("connection.title");
    final String message = getString("connection.message");
    final boolean executed = connTask.execute(desktopFrame, title, message);
    if (!executed) {
      final Exception exception = connTask.getError();
      throw new ApplicationException(exception);
    }

    eventHandler = createEventHandler();
    if (eventHandler == null) {
      final String noHandlerMsg = getString("no.event.handler.warning");
      log(noHandlerMsg);
      return;
    }

    final String handlerMsg = getString("event.handler.class.message");
    final Class<? extends DesktopLauncherEventHandler> eventHandlerClass =
      eventHandler.getClass();
    final String eventHandlerClassName = eventHandlerClass.getName();
    log(handlerMsg + ": " + eventHandlerClassName);

    eventHandler.init();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * Construtor
   * 
   * @param id identificador da aplicao.
   */
  public DesktopLauncher(final String id) {
    super(id);

    eventPanel = new EventPanel(this);
    logPanel = new LogPanel(this);

    final Client client = Client.getInstance();
    sonId = client.generateNewSonId();

    final JFrame mainFrame = getApplicationFrame();
    mountFrame(mainFrame);
    mainFrame.setJMenuBar(buildMenuBar());
    mainFrame.setSize(new Dimension(700, 400));
  }

  /**
   * Montagem do frame da aplicao.
   * 
   * @param mainFrame frame
   */
  private void mountFrame(final JFrame mainFrame) {
    final Client client = Client.getInstance();
    final Container contentPane = mainFrame.getContentPane();
    contentPane.setLayout(new GridBagLayout());

    final String myId = client.getClientInstanceId();
    final JTextField idText = new JTextField();
    idText.setEditable(false);
    idText.setText(myId);
    idText.setHorizontalAlignment(SwingConstants.LEFT);

    final JToolBar toolBar = buildToolBar();
    contentPane.add(toolBar, new GBC(0, 0).horizontal());
    final JSplitPane split =
      new JSplitPane(JSplitPane.VERTICAL_SPLIT, eventPanel, logPanel);
    split.setOneTouchExpandable(true);
    split.setResizeWeight(0.75);
    contentPane.add(split, new GBC(0, 1).both().weights(2.0, 1.0));
    contentPane.add(idText, new GBC(0, 2).horizontal());
  }

  /**
   * Insere uma linha de texto no log
   * 
   * @param text texto
   */
  final public void log(final String text) {
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        logPanel.appendText(text);
      }
    });
  }

  /**
   * Consulta o painel de log.
   * 
   * @return o painel.
   */
  public LogPanel getLogPanel() {
    return logPanel;
  }
}
