package csbase.client.applications.projectsmanager.actions;

import java.awt.Window;

import javax.swing.ImageIcon;

import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.GetProjectHistoryTask;
import csbase.client.facilities.textviewer.TextViewer;
import csbase.logic.User;

/**
 * Ao que exibe o histrico de um projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ShowProjectHistoryAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public ShowProjectHistoryAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ApplicationImages.ICON_LIST_16;
  }

  /**
   * <p>
   * Apresenta o histrico do projeto selecionado ao usurio.
   * </p>
   * <p>
   * Se mais de um projeto estiver selecionado, apenas o histrico do primeiro
   * ser selecionado.
   * </p>
   * 
   * @see ProjectsManager#getSelectedProjects()
   */
  @Override
  public void actionDone() throws Exception {
    final ProjectsManagerData pmd =
      getProjectsManager().getSelectedProjects().get(0);

    if (!validateProject(pmd)) {
      final String deniedMessage =
        getString("ShowProjectHistoryAction.project.selection.denied.message")
          + getString("ShowProjectHistoryAction.project.requirements.message");
      StatusDialog
        .showNoneOkDialog(
          getProjectsManager(),
          deniedMessage,
          pmd,
          getString("ShowProjectHistoryAction.project.selection.failure.message"));
      return;
    }

    final String projectName = pmd.getProjectName();

    final GetProjectHistoryTask task =
      new GetProjectHistoryTask(getProjectsManager(), pmd);

    String taskMsgTemplate =
      getString("ShowProjectHistoryAction.retrieving.history.message");
    final String taskMsg = String.format(taskMsgTemplate, projectName);
    final String taskTitle =
      getString("ShowProjectHistoryAction.retrieving.history.title");
    final Window owner = getProjectsManager().getApplicationFrame();
    task.execute(owner, taskTitle, taskMsg);

    if (task.wasCancelled()) {
      final String taskCancelled =
        getString("ShowProjectHistoryAction.retrieving.history.cancelled.message");
      task.showError(taskCancelled);
      return;
    }

    if (task.getStatus() != true) {
      final Exception exception = task.getError();
      throw exception;
    }

    final String history = task.getResult();
    String viewerTitleTemplate =
      getString("ShowProjectHistoryAction.showing.history.title");
    String viewerTitle = String.format(viewerTitleTemplate, projectName);
    TextViewer textViewer = new TextViewer(history, viewerTitle);
    textViewer.showDialog();
  }

  /**
   * Valida se  possvel obter o histrico do projeto.
   * 
   * @param pmd Projeto a ser validado.
   * @return <tt>true</tt> se for possvel obter o histrico do projeto.
   */
  protected boolean validateProject(ProjectsManagerData pmd) {
    final User loggedUser = User.getLoggedUser();
    final Object loggedUserId = loggedUser.getId();

    final ProjectSpaceAllocation spaceAllocation =
      pmd.getProjectSpaceAllocation();

    if (!spaceAllocation.isOpenable()) {
      // O projeto no poder ser aberto pois est aguardando (des)alocao.
      return false;
    }

    if (loggedUser.isAdmin()) {
      // O administrador tem acesso a todos os projetos.
      return true;
    }

    if (pmd.getOwnerId().equals(loggedUserId)) {
      // O usurio  o dono do projeto.
      return true;
    }

    /*
     * A princpio no precisaramos checar a permisso de leitura pois, a
     * gerncia de projetos apresenta ao usurio apenas projetos que ele tenha
     * acesso de leitura. A checagem foi includa para dar maior segurana caso
     * o cdigo seja modificado, reaproveitado ou um novo tipo de
     * compartilhamento seja criado.
     */
    switch (pmd.getSharingType()) {
      case PRIVATE:
        /*
         * O usurio no tem acesso pois, o acesso ao projeto  privado e ele
         * no  o administrador e nem o dono.
         */
        return false;
      case PARTIAL:
        /*
         * Se o acesso  parcial, deve-se verificar se o usurio est em alguma
         * das listas de usurios com acesso, RW ou RO.
         */
        return pmd.getUsersRO().contains(loggedUserId)
          || pmd.getUsersRW().contains(loggedUserId);
      case ALL_RO:
      case ALL_RW:
        /*
         * O compartilhamento do projeto permite que todos os usurios tenham
         * acesso a leitura dele.
         */
        return true;
      default:
        /*
         * No deve cair aqui a menos que um novo tipo de compartilhamento tenha
         * sido criado e no tenha sido includo neste switch.
         */
        String[] queue = new String[] { getClass().getSimpleName() };
        String[] info =
          new String[] { "[ERRO] No existe a validao para projetos com compartilhamento "
            + pmd.getSharingType().name() + " no foi encontrada." };
        getProjectsManager().logDetailedApplicationEvent(queue, info);
        return false;
    }
  }
}
